/************************************************************************/
/* 
 * Module:	cdctrl.c
 * Purpose:     to control Linux CD-ROMs
 * Language/Compiler/O/S:   GCC 2.7.2, Linux 2.0.X
 * Author:      Wade Hampton
 * Date:        12/22/97
 * Revision History:
 * [from RCS]
 *
 * $Log: cdctrl.c,v $
 * Revision 1.5  1998/01/06 21:00:33  wadeh
 * Added DoCr, changes to fix cdeject crash.
 *
 * Revision 1.4  1997/12/30 22:59:31  wadeh
 * Added fflush to properly work with pipes...
 *
 * Revision 1.3  1997/12/24 15:39:10  wadeh
 * Minor bug fixes:
 * 1) Fixed cdctrl eject command causing a signal 11.
 * 2) Fixed Makefile debug option to not strip symbols, can now debug.
 * 3) Added command.h and more prototypes, more documentation.
 * 4) Fixed read_hw to ralways return buffer, even on error.
 * ,
 *
 * Revision 1.2  1997/12/23 22:52:51  wadeh
 * 2.1 beta 1 release
 *
 * Revision 1.1  1997/12/23 21:38:52  wadeh
 * Initial revision
 *
 *
 * Notes:
 * 1) 
 *----------------------------------------------------------------------*/
/* TYPES- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
/* INCLUDE FILES- - - - - - - - - - - - - - - - - - - - - - - - - - - - */
#include <stdio.h> 
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <linux/cdrom.h>

#include "config.h"
#include "cdctrl.h"
#include "commands.h"
#include "hardware.h"
#include "info.h"

/* CONSTANTS- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
/*   RCS ID string: */
static char rcsid[] = "$Id: cdctrl.c,v 1.5 1998/01/06 21:00:33 wadeh Exp wadeh $";   


/* LOCAL DATA - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
char caLineIn[256];

/* EXTERNAL PROCEDURES- - - - - - - - - - - - - - - - - - - - - - - - - */
/* EXTERNAL DATA- - - - - - - - - - - - - - - - - - - - - - - - - - - - */
/* PROTOTYPES - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
void print_userhelp(int iVersion);
void print_help (int docr);
void print_info(char *progname, int cdfile, char *pszCmd, int docr);
int  cmd_parse (char *progname, int cdfile, char *pszBuf, int iRead, int docr);
int  main (int argc, char *argv[]);
/*
 * Copyright (C) 1997, 1998, Wade Hampton
 */
/************************************************************************/

/************************************************************************/
/* Procedure:  print_userhelp
 * Purpose:    to print command and user help on commands
 * 
 * Inputs:     none
 * Outputs:    to stdout, to stderr
 * Returns:    none
 * Notes:  
 *   1.
 */
/************************************************************************/
void print_userhelp(int iVersion)
{
    printf ("cdctrl:\nCommand line CDROM control\nOptions:\n");
    printf ("  cdctrl -c device  -- output using CRLF\n");
    printf ("  cdctrl device     -- normal output\n");
    printf ("  cdctrl [-h|-?|-V] -- help/version\n");

    if (iVersion == TRUE) {
	printf ("  Version=%s\n", rcsid);
	printf ("  Built on %s %s\n", __DATE__, __TIME__);
    } else {
	printf ("Command-line control options are:\n");
	print_help (DOLF);
    }
}

/************************************************************************/
/* Procedure:  print_help
 * Purpose:    to print help on commands
 * 
 * Inputs:     none
 * Outputs:    to stdout, to stderr
 * Returns:    none
 * Notes:  
 *   1.
 */
/************************************************************************/
void print_help (int docr)
{
    printf ("1    play");
    DoCr(docr);
    printf ("s    stop");
    DoCr(docr);
    printf ("p    pause");
    DoCr(docr);
    printf ("r    resume");
    DoCr(docr);
    printf ("e    eject");
    DoCr(docr);
    printf ("i    info");
    DoCr(docr);
    printf ("d    dir");
    DoCr(docr);
    printf ("-    previous track");
    DoCr(docr);
    printf ("+    next track");
    DoCr(docr);
    printf ("N    play track N");
    DoCr(docr);
    printf ("q    quit");
    DoCr(docr);
    printf ("?    help");
    DoCr(docr);
    printf ("info format:  \"CMD cmd-name cd-status track abs-time rel-time\"");
    DoCr(docr);
    printf ("  cmd-name  := {play, stop, ... quit} from above list");
    DoCr(docr);
    printf ("  cd-status := {invalid, play, paused, completed, error, no status}");
    DoCr(docr);
    printf ("  track     := {1..99} CD track");
    DoCr(docr);
    printf ("  abs-time  := HH:MM:SS elapsed since CD start");
    DoCr(docr);
    printf ("  rel-time  := HH:MM:SS elapsed since track start");
    DoCr(docr);
    printf ("Program info:  %s", rcsid);
    DoCr(docr);
    printf ("Built on: %s %s", __DATE__, __TIME__);
    DoCr(docr);
}

/************************************************************************/
/* Procedure:  print_info
 * Purpose:    to print info on status
 * 
 * Inputs:     command
 * Outputs:    to stdout, to stderr
 * Returns:    none
 * Notes:  
 *   1.
 */
/************************************************************************/
void print_info(char *progname, int cdfile, char *pszCmd, int docr)
{
    printf ("CMD %s ", pszCmd);
    do_info (progname, cdfile, I_DISP_ALL, DONADA); 
    DoCr (docr);
}

/************************************************************************/
/* Procedure:  cmd_parse
 * Purpose:    to parse user's commands
 * 
 * Inputs:     program name, file description for CDROM, buffer, count
 * Outputs:    to CDROM, to stdout, to stderr
 * Returns:    T/F, T when "done"
 * Notes:  
 *   1.
 */
/************************************************************************/
int cmd_parse (char *progname, int cdfile, char *pszBuf, int iRead, int docr)
{
int iDone = FALSE;
int iTrk;
char *pszCmd;

/*     printf ("Processing command:  %s\n", pszBuf); */

    pszCmd = "NULL";

    if ((pszBuf[0] >= '0') && (pszBuf[0] <= '9')) {
	iTrk = atoi (pszBuf);
	do_play(progname, cdfile, iTrk, 0);
	pszCmd = "play";
    } else {

	/* handle commands.... */
	switch (pszBuf[0]) {
	    
	case 's':
	    do_stop(progname, cdfile);
	    pszCmd = "stop";
	    break;
	    
	case 'p':
	    do_pause(progname, cdfile);
	    pszCmd = "pause";
	    break;
	  
	case 'r':
	    do_play(progname, cdfile, 0, 0);
	    pszCmd = "resume";
	    break;
	  
	case 'e':
	    do_eject(progname, cdfile);
	    sleep (3);
	    pszCmd = "eject";
	    break;
	  
	case 'i':
	    pszCmd = "info";
	    break;
	    
	case 'd':
	    do_dir (progname, cdfile, P_RAW, 0, docr); 
	    pszCmd = "dir";
	    break;

	case '-':
	    do_play(progname, cdfile, SKIPBACK, 0);
	    pszCmd = "prevous";
	    break;

	case '+':
	    do_play(progname, cdfile, SKIPFORWARD, 0);
	    pszCmd = "next";
	    break;

	case 'q':
	    iDone = TRUE;
	    pszCmd = "QUIT";
	    break;
	
	case '?':
	    print_help(docr);
	    pszCmd = "help";
	    break;

	default:
	    printf ("ERROR invalid command %s", pszBuf);
	    DoCr(docr);
	    pszCmd = "default";
	    break;
	}
    }

    print_info(progname, cdfile, pszCmd, docr);
    printf ("END");
    DoCr(docr);
    return (iDone);

}

/************************************************************************/
/* Procedure:  main
 * Purpose:    cdctrl's main 
 * 
 * Inputs:     command line args
 * Outputs:    to CDROM, to stdout, to stderr
 * Returns:    T/F, T when "done"
 * Notes:  
 *   1.
 */
/************************************************************************/
int main (int argc, char *argv[])
{
    char *progname = "CDCTRL";
    int cdfile;

    int iRead;
    int iDone = FALSE;
    int docr  = DOLF;

    /* parse command line */
    char *cd_device = CD_DEVICE;
    if (argc > 1) {
	if (argv[1][0] == '-') {
	    if ((argv[1][1] == '?') || (argv[1][1] == 'h')) {
		print_userhelp(FALSE);
		exit(0);
	    } 
	    else if (argv[1][1] == 'V') {
		print_userhelp(TRUE);
		exit(0);
	    } 
	    else if (argv[1][1] == 'c') {
		docr = DOCRLF;
	    }
	    else {
		print_userhelp(FALSE);
		exit(0);
	    }
            if (argc > 2) {
		cd_device=argv[2];
	    }
	} 
	else {
	    cd_device=argv[1];
	}
    }
    
    /* close stderr and redirect to stdout */
    close (2);
    dup2 (1, 2);
    
    /* try to open the CD device(s) */
    if ( (cdfile=open(cd_device, O_RDONLY)) == -1 ) {
	fprintf(stderr, "%s: can't open cdrom (%s)", progname, cd_device);
	DoCr(docr);
	exit(1);
    }
    
    printf ("Opened %s as device %d", cd_device, cdfile);
    DoCr(docr);

    /* process commands */
    while (iDone == FALSE) {
#ifdef DEBUG1
printf ("Reading input\n");
fflush (stdout);
#endif
	if (fgets (caLineIn, sizeof(caLineIn)-1, stdin) != NULL) {

#ifdef DEBUG1
printf ("Read data, computing length\n");
fflush (stdout);
#endif
	    iRead = strlen (caLineIn);
	    if (iRead > 0) {
#ifdef DEBUG1
printf ("Read %d chars=%s\n", iRead, caLineIn);
fflush (stdout);
#endif
                iDone = cmd_parse (progname, cdfile, caLineIn, iRead, docr);
	    }
	    fflush (stdout);
	}
    }
    
    /* cleanup */
    close (cdfile);
    
    printf ("EXITING");
    DoCr(docr);
    fflush (stdout);

}
