/*
 * tnmWinIcmp.c --
 *
 *	Make an ICMP request to a list of IP addresses. The Windows
 *	implementation uses the unofficial Microsoft ICMP DLL since
 *	there are currently no RAW sockets shipped with Windows NT
 *	3.51 and Windows '95.
 *
 * Copyright (c) 1996      University of Twente.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */

#include "tnmInt.h"
#include "tnmPort.h"

/*
 * The following structure is used by Microsoft's ICMP.DLL.
 */

typedef struct {
   unsigned char Ttl;			/* Time To Live. */
   unsigned char Tos;			/* Type Of Service. */
   unsigned char Flags;			/* IP header flags. */
   unsigned char OptionsSize;		/* Size in bytes of options data. */
   unsigned char *OptionsData;		/* Pointer to options data. */
} IP_OPTION_INFORMATION, * PIP_OPTION_INFORMATION;

typedef struct {
   DWORD Address;			/* Replying address. */
   unsigned long  Status;		/* Reply status. */
   unsigned long  RoundTripTime;	/* RTT in milliseconds/ */
   unsigned short DataSize;		/* Echo data size. */
   unsigned short Reserved;		/* Reserved for system use. */
   void *Data;				/* Pointer to the echo data. */
   IP_OPTION_INFORMATION Options;	/* Reply options. */
} IP_ECHO_REPLY, * PIP_ECHO_REPLY;

/*
 * Functions exported by Microsoft's ICMP.DLL.
 */

HANDLE ( WINAPI *pIcmpCreateFile )( VOID );
BOOL ( WINAPI *pIcmpCloseHandle )( HANDLE );
DWORD ( WINAPI *pIcmpSendEcho )( HANDLE, DWORD, LPVOID, WORD,
                                    PIP_OPTION_INFORMATION, LPVOID, 
                                    DWORD, DWORD );
/*
 * Forward declarations for procedures defined later in this file:
 */

static void
IcmpExit	_ANSI_ARGS_((ClientData clientData));


/*
 *----------------------------------------------------------------------
 *
 * IcmpExit --
 *
 *	Cleanup upon exit of the Tnm process.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

void
IcmpExit(clientData)
    ClientData clientData;
{
    HANDLE hIcmp = (HANDLE) clientData;
    FreeLibrary(hIcmp);
}

/*
 *----------------------------------------------------------------------
 *
 * TnmIcmp --
 *
 *	XXX This one will change slightly. XXX
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
TnmIcmp(interp, icmpPtr)
    Tcl_Interp *interp;
    TnmIcmpRequest *icmpPtr;
{
    static HANDLE hIcmp = 0;	/* The handle for ICMP.DLL. */
    HANDLE hIP;
    PIP_ECHO_REPLY pIpe;
    DWORD dwStatus;
    int i;

    if (! hIcmp) {
	hIcmp = LoadLibrary("ICMP.DLL");
	if (hIcmp) {
	    Tcl_CreateExitHandler(IcmpExit, hIcmp);
	    (FARPROC) pIcmpCreateFile 
		= (FARPROC) GetProcAddress(hIcmp, "IcmpCreateFile");
	    (FARPROC) pIcmpCloseHandle
		= (FARPROC) GetProcAddress(hIcmp, "IcmpCloseHandle");
	    (FARPROC) pIcmpSendEcho 
		= (FARPROC) GetProcAddress(hIcmp, "IcmpSendEcho");
	    if (! pIcmpCreateFile || ! pIcmpCloseHandle || ! pIcmpSendEcho) {
		FreeLibrary(hIcmp);
		hIcmp = 0;
	    }
	}

    }
    if (! hIcmp) {
	Tcl_SetResult(interp, "unable to load ICMP.DLL", TCL_STATIC);
	return TCL_ERROR;
    }

    hIP = pIcmpCreateFile();
    if (hIP == INVALID_HANDLE_VALUE) {
	Tcl_SetResult(interp, "unable to open ICMP handle", TCL_STATIC);
	return TCL_ERROR;
    }

    pIpe = (PIP_ECHO_REPLY) ckalloc(sizeof(IP_ECHO_REPLY) + icmpPtr->size);
    memset((char *) pIpe, 0, sizeof(IP_ECHO_REPLY) + icmpPtr->size);
    pIpe->Data = pIpe + sizeof(IP_ECHO_REPLY);
    pIpe->DataSize = icmpPtr->size;

    for (i = 0; i < icmpPtr->argc; i++) {
	char buf[80];
	TnmIcmpTarget target = icmpPtr->targets[i];

	dwStatus = pIcmpSendEcho(hIP, ntohl(target.dst.sin_addr.s_addr),
				 pIpe->Data, pIpe->DataSize, NULL,
				 pIpe, sizeof(IP_ECHO_REPLY) + icmpPtr->size,
				 icmpPtr->timeout);
	if (! dwStatus) {
	    Tcl_SetResult(interp, "unable to send ICMP packet", TCL_STATIC);
	    ckfree((char *) pIpe);
	    pIcmpCloseHandle(hIP);
	    return TCL_ERROR;
	}

	sprintf(buf, "Addr:%d.%d.%d.%d,  RTT: %dms,  TTL: %d", 
		LOBYTE(LOWORD(pIpe->Address)),
		HIBYTE(LOWORD(pIpe->Address)),
		LOBYTE(HIWORD(pIpe->Address)),
		HIBYTE(HIWORD(pIpe->Address)),
		pIpe->RoundTripTime,
		pIpe->Options.Ttl);
	Tcl_AppendElement(interp, buf);
    }

    ckfree((char *) pIpe);

    pIcmpCloseHandle(hIP);
    return TCL_OK;
}
