/*
 * tnmMib.h --
 *
 *	Definitions for the SNMP MIB parser and query interface.
 *
 * Copyright (c) 1994-1996 Technical University of Braunschweig.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */

#ifndef _TNMMIB
#define _TNMMIB

/*
 *----------------------------------------------------------------
 * A structure to store an integer value and a name. This is 
 * used to convert enumerated integers to/from readable names.
 *----------------------------------------------------------------
 */

typedef struct Tnm_MibEnum { 
    int value;			 /* The value of the enumerated integer. */
    char *label;		 /* The name of this integer value. */
    struct Tnm_MibEnum *nextPtr; /* Next value/name pair in list. */
} Tnm_MibEnum;

/*
 *----------------------------------------------------------------
 * A structure to store the Textual-Convention macros. The 
 * Textual-Convention stores a format string if displayHint 
 * is NULL. If displayHint is not NULL, the Textual-Convention 
 * stores an INTEGER enumeration.
 *----------------------------------------------------------------
 */

typedef struct Tnm_MibTC {
    char *name;			/* The name of the Textual Convention. */
    char *fileName;		/* The file with the textual description. */
    char *moduleName;		/* The name of the MIB module. */
    short syntax;		/* The ASN.1 syntax used e.g. INTEGER. */
    char *displayHint;		/* The display hint, eg. 2d. */
    Tnm_MibEnum	*enumList;	/* The list of enumarations. */
    struct Tnm_MibTC *nextPtr;	/* Next Tnm_MibTC in the list. */
} Tnm_MibTC;

/*
 *----------------------------------------------------------------
 * The following structure is used to hold the MIB tree in 
 * memory. Every node is linked with its parent, a list of child 
 * nodes and the next node on the current MIB level.
 *----------------------------------------------------------------
 */

typedef struct Tnm_MibNode {
    char 	 *label;	/* Node's textual name (not always unique). */
    char	 *parentName;	/* Name of parent node during parse.        */
    char         *fileName;	/* The file with the textual description.   */
    char	 *moduleName;	/* The name of the MIB module.		    */
    int		 fileOffset;	/* Offset for node's textual description.   */
    short	 syntax;	/* This node's object type syntax.          */
    u_char	 access;	/* The access mode of the object.	    */
    u_char	 macro;		/* The ASN.1 macro used for the definition. */
    char	 *index;	/* The list of index nodes in a table entry.*/
    Tnm_MibTC    *tc;		/* Optional Textual Convention.		    */
    u_int	 subid;		/* This node's integer subidentifier.       */
    struct Tnm_MibNode *parentPtr; /* The parent of this node.	            */
    struct Tnm_MibNode *childPtr;  /* List of child nodes.	            */
    struct Tnm_MibNode *nextPtr;   /* List of peer nodes.		    */
} Tnm_MibNode;

/*
 *----------------------------------------------------------------
 * The SMI MIB tree node access modes:
 *----------------------------------------------------------------
 */

#define TNM_MIB_NOACCESS	0
#define TNM_MIB_READONLY	1
#define TNM_MIB_READCREATE	2
#define TNM_MIB_READWRITE 	3
#define TNM_MIB_WRITEONLY	4
#define TNM_MIB_FORNOTIFY	5

EXTERN TnmTable tnmSnmpMibAccessTable[];

/*
 *----------------------------------------------------------------
 * The SMI macros used to define MIB tree nodes:
 *----------------------------------------------------------------
 */

#define TNM_MIB_OBJECTTYPE		1
#define TNM_MIB_OBJECTIDENTITY		2
#define TNM_MIB_MODULEIDENTITY		3
#define TNM_MIB_NOTIFICATIONTYPE	4
#define TNM_MIB_TRAPTYPE		5
#define TNM_MIB_OBJECTGROUP		6
#define TNM_MIB_NOTIFICATIONGROUP	7
#define TNM_MIB_COMPLIANCE		8
#define TNM_MIB_CAPABILITIES		9

EXTERN TnmTable tnmSnmpMibMacroTable[];

/*
 *----------------------------------------------------------------
 * Exported variables:
 *----------------------------------------------------------------
 */

EXTERN char *tnm_MibFileName;		/* Current MIB file name loaded. */
EXTERN int tnm_MibLineNumber;		/* Current MIB file line number. */
EXTERN char *tnm_MibModuleName;		/* Current MIB module name loaded. */
EXTERN Tnm_MibNode *tnm_MibTree;	/* The root of the MIB tree. */
EXTERN Tnm_MibTC *tnm_MibTCList;	/* List of textual conventions. */
EXTERN Tnm_MibTC *tnm_MibTCSaveMark;	/* The first already saved */
					/* element of mib_TCList. */

/*
 *----------------------------------------------------------------
 * Exported functions to access the information stored 
 * in the internal MIB tree.
 *----------------------------------------------------------------
 */

EXTERN char*
Tnm_MibGetOid		_ANSI_ARGS_((char *name, int exact));

EXTERN char*
Tnm_MibGetName		_ANSI_ARGS_((char *oid,  int exact));

EXTERN char*
Tnm_MibGetSyntax	_ANSI_ARGS_((char *name, int exact));

EXTERN int
Tnm_MibGetBaseSyntax	_ANSI_ARGS_((char *name, int exact));

EXTERN char*
Tnm_MibGetDescription	_ANSI_ARGS_((char *name, int exact));

EXTERN char*
Tnm_MibGetAccess	_ANSI_ARGS_((char *name, int exact));

EXTERN char*
Tnm_MibGetMacro		_ANSI_ARGS_((char *name, int exact));

EXTERN char*
Tnm_MibGetModule	_ANSI_ARGS_((char *name, int exact));

EXTERN char*
Tnm_MibGetSucc		_ANSI_ARGS_((char *name));

EXTERN char*
Tnm_MibGetParent	_ANSI_ARGS_((char *name, int exact));

EXTERN char*
Tnm_MibGetTC		_ANSI_ARGS_((char *name, int exact));

EXTERN char*
Tnm_MibGetFile		_ANSI_ARGS_((char *name, int exact));

EXTERN char*
Tnm_MibGetIndex		_ANSI_ARGS_((char *name, int exact));

EXTERN char*
Tnm_MibGetDefault	_ANSI_ARGS_((char *name, int exact));

EXTERN int
Tnm_MibNodeGetOid	_ANSI_ARGS_((Tnm_MibNode *nodePtr, Tnm_Oid *oid));

EXTERN Tnm_MibNode*
Tnm_MibFindNode		_ANSI_ARGS_((char *name, int *offset, int exact));

EXTERN char*
Tnm_MibFormat		_ANSI_ARGS_((char *name, int exact, char *arg));

EXTERN char*
Tnm_MibScan		_ANSI_ARGS_((char *name, int exact, char *arg));

/*
 *----------------------------------------------------------------
 * Functions to read a file containing MIB definitions.
 *----------------------------------------------------------------
 */

EXTERN Tnm_MibNode*
Tnm_MibParse		_ANSI_ARGS_((char *file, char *frozen,
				     Tnm_MibNode *root));
EXTERN Tnm_MibNode*
Tnm_MibReadFrozen	_ANSI_ARGS_((FILE *fp));

EXTERN void
Tnm_MibWriteFrozen	_ANSI_ARGS_((FILE *fp, Tnm_MibNode *nodePtr));

/*
 *----------------------------------------------------------------
 * Functions used by the parser or the frozen file reader to
 * build the MIB tree structure(s).
 *----------------------------------------------------------------
 */

EXTERN Tnm_MibNode*
Tnm_MibNewNode		_ANSI_ARGS_((char *label));

EXTERN void
Tnm_MibAddNode		_ANSI_ARGS_((Tnm_MibNode **rootPtr, 
				     Tnm_MibNode *nodePtr));
EXTERN Tnm_MibTC*
Tnm_MibAddTC		_ANSI_ARGS_((Tnm_MibTC *tcPtr));

EXTERN Tnm_MibTC*
Tnm_MibFindTC		_ANSI_ARGS_((char *name));

/*
 *----------------------------------------------------------------
 * The Tcl initialization function:
 *----------------------------------------------------------------
 */

EXTERN int
Tnm_MibInit		_ANSI_ARGS_((Tcl_Interp *interp));

#endif /* _TNMMIB */
