/*
 * tnmNetdb.c --
 *
 *	This file contains the source of the netdb command that
 *	provides access to local network configuration
 *	information. Its mostly just a wrapper around the C
 *	interface defined in netdb.h.
 *
 * Copyright (c) 1995-1996 Technical University of Braunschweig.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */

#include "tnmInt.h"
#include "tnmPort.h"

#include <rpc/rpc.h>

/*
 * Some machines have no rpcent structure. Here is definition that
 * seems to work in this cases.
 */

#ifndef HAVE_RPCENT
struct rpcent {
    char *r_name;	/* name of server for this rpc program */
    char **r_aliases;	/* alias list */
    int r_number;	/* rpc program number */
};
#endif

/*
 * Forward declarations for procedures defined later in this file:
 */

static int
NetdbHosts		_ANSI_ARGS_((Tcl_Interp *interp, 
				     int argc, char **argv));
static int
NetdbIp			_ANSI_ARGS_((Tcl_Interp *interp, 
				     int argc, char **argv));
static int
NetdbNetworks		_ANSI_ARGS_((Tcl_Interp *interp, 
				     int argc, char **argv));
static int
NetdbProtocols		_ANSI_ARGS_((Tcl_Interp *interp, 
				     int argc, char **argv));
static int
NetdbServices		_ANSI_ARGS_((Tcl_Interp *interp, 
				     int argc, char **argv));
static int
NetdbSunrpcs		_ANSI_ARGS_((Tcl_Interp *interp, 
				     int argc, char **argv));


/*
 *----------------------------------------------------------------------
 *
 * NetdbHosts --
 *
 *	This procedure is invoked to process the "netdb hosts" command.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *----------------------------------------------------------------------
 */

static int
NetdbHosts(interp, argc, argv)
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    /*
     * Process the "netdb hosts" command option:
     */

    if (argc == 2) {
#ifdef HAVE_GETHOSTENT
	struct hostent *host;
	unsigned long addr;
	struct in_addr *paddr;
	char buffer[20];
	int i = 0;
	
	sethostent(0);
	while ((host = gethostent())) {	
	    Tcl_AppendResult(interp, i++ ? " {" : "{", host->h_name, 
			     (char *) NULL);
	    if (*host->h_addr_list) {
		paddr = (struct in_addr *) *host->h_addr_list++;
		addr = ntohl(paddr->s_addr);
		sprintf(buffer, "%lu.%lu.%lu.%lu",
			(addr >> 24) & 0xff, (addr >> 16) & 0xff,
			(addr >> 8) & 0xff, addr & 0xff);
		Tcl_AppendResult(interp, " ", buffer, "}", (char *) NULL);
	    }
	}
	endhostent();
#endif
	return TCL_OK;
    }

    /*
     * Process the "netdb hosts name" command option:
     */

    if (strcmp(argv[2], "name") == 0) {
	struct sockaddr_in addr;
	char *name;
	if (argc != 4) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			     " hosts name address\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (TnmValidateIpAddress(interp, argv[3]) != TCL_OK) {
	    return TCL_ERROR;
	}
	if (TnmSetIPAddress(interp, argv[3], &addr) != TCL_OK) {
	    return TCL_ERROR;
	}
	name = TnmGetIPName(interp, &addr);
	if (! name) {
	    return TCL_ERROR;
	}
	Tcl_SetResult(interp, name, TCL_VOLATILE);
	return TCL_OK;
    }

    /*
     * Process the "netdb hosts address" command option:
     */

    if (strcmp(argv[2], "address") == 0) {
	struct sockaddr_in addr;
	if (argc != 4) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			     " hosts address name\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (TnmValidateIpHostName(interp, argv[3]) != TCL_OK) {
	    return TCL_ERROR;
	}
	if (TnmSetIPAddress(interp, argv[3], &addr) != TCL_OK) {
	    return TCL_ERROR;
	}
	Tcl_SetResult(interp, inet_ntoa(addr.sin_addr), TCL_VOLATILE);
	return TCL_OK;
    }

    Tcl_AppendResult(interp, "bad query \"", argv[2], 
		     "\": should be address, or name", (char *) NULL);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * NetdbIp --
 *
 *	This procedure is invoked to process the "netdb ip" command.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *----------------------------------------------------------------------
 */

static int
NetdbIp(interp, argc, argv)
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    if (argc == 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			 " ip option arg\"", (char *) NULL);
	return TCL_ERROR;
    }

    /*
     * Process the "netdb ip class" command option:
     */

    if (strcmp(argv[2], "class") == 0) {
	unsigned long addr;
	if (argc != 4) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			     " ip class address\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (TnmValidateIpAddress(interp, argv[3]) != TCL_OK) {
	    return TCL_ERROR;
	}
	if ((addr = inet_addr(argv[3])) == -1) {
	    Tcl_AppendResult(interp, "invalid IP address \"",
			     argv[3], "\"", (char *) NULL);
	    return TCL_ERROR;
	}
	addr = ntohl(addr);
	if ((addr >> 24) == 127) {
	    Tcl_SetResult(interp, "loopback", TCL_STATIC);
	    return TCL_OK;
	} else if (IN_CLASSA(addr)) {
	    Tcl_SetResult(interp, "A", TCL_STATIC);
	} else if (IN_CLASSB(addr)) {
	    Tcl_SetResult(interp, "B", TCL_STATIC);
	} else if (IN_CLASSC(addr)) {
	    Tcl_SetResult(interp, "C", TCL_STATIC);
	} else if (IN_CLASSD(addr)) {
	    Tcl_SetResult(interp, "D", TCL_STATIC);
	} else {
	    Tcl_SetResult(interp, "unknown IP class", TCL_STATIC);
	    return TCL_ERROR;
	}
    
	return TCL_OK;
    }

    /*
     * Process the "netdb ip range" command option:
     */

    if (strcmp(argv[2], "range") == 0) {
	unsigned long net, mask, addr;
	char buf[20];
	if (argc != 5) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			     " ip range address mask\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (TnmValidateIpAddress(interp, argv[3]) != TCL_OK) {
	    return TCL_ERROR;
	}
	if ((net = inet_addr(argv[3])) == -1) {
	    Tcl_AppendResult(interp, "invalid IP address \"", 
			     argv[3], "\"", (char *) NULL);
	    return TCL_ERROR;
	}
        net = ntohl(net);
	if (TnmValidateIpAddress(interp, argv[4]) != TCL_OK) {
	    return TCL_ERROR;
	}
	if ((mask = inet_addr(argv[4])) == -1) {
	    Tcl_AppendResult(interp, "invalid IP address mask \"",
			     argv[4], "\"", (char *) NULL);
	    return TCL_ERROR;
	}
	mask = ntohl(mask);
    
	for (addr = net + 1; addr < net + ~mask; addr++) {
	    sprintf(buf, "%lu.%lu.%lu.%lu",
		    (addr >> 24) & 0xff, (addr >> 16) & 0xff,
		    (addr >> 8) & 0xff, addr & 0xff);
	    Tcl_AppendElement(interp, buf);
	}
	return TCL_OK;
    }

    Tcl_AppendResult(interp, "bad query \"", argv[2], 
		     "\": should be class, or range", (char *) NULL);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * NetdbNetworks --
 *
 *	This procedure is invoked to process the "netdb networks" command.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *----------------------------------------------------------------------
 */

static int
NetdbNetworks(interp, argc, argv)
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    /*
     * Process the "netdb networks" command option:
     */

    if (argc == 2) {
#ifdef HAVE_GETNETENT
	struct netent *net;
	char buf[20];
	int i = 0;
	
	setnetent(0);
	while ((net = getnetent())) {
	    while (net->n_net && ! ((net->n_net >> 24) & 0xff)) {
		net->n_net <<= 8;
	    }
	    sprintf(buf, "%lu.%lu.%lu.%lu",
		    (net->n_net >> 24) & 0xff, (net->n_net >> 16) & 0xff,
		    (net->n_net >> 8) & 0xff, net->n_net & 0xff);
	    Tcl_AppendResult(interp, i++ ? " {" : "{", net->n_name, 
			     " ", buf, "}", (char *) NULL);
	}
	endnetent();
#endif
	return TCL_OK;
    }

    /*
     * Process the "netdb networks name" command option:
     */

    if (strcmp(argv[2], "name") == 0) {
	struct netent *net;
	unsigned long addr;
	if (argc != 4) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			     " networks name address\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (TnmValidateIpAddress(interp, argv[3]) != TCL_OK) {
	    return TCL_ERROR;
	}
	if ((addr = inet_addr(argv[3])) == -1) {
	    Tcl_AppendResult(interp, "invalid IP address \"", 
			     argv[3], "\"", (char *) NULL);
	    return TCL_ERROR;
	}
	while (addr && ! (addr & 0xff)) {
	    addr >>= 8;
	}
	if (! (net = getnetbyaddr(addr, AF_INET))) {
	    Tcl_AppendResult(interp, "can not lookup \"", 
			     argv[3], "\"", (char *) NULL);
	    return TCL_ERROR;
	}
	Tcl_SetResult(interp, (char *) net->n_name, TCL_VOLATILE);
	return TCL_OK;
    }

    /*
     * Process the "netdb networks address" command option:
     */

    if (strcmp(argv[2], "address") == 0) {
	struct netent *net;
	if (argc != 4) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			     " networks address name\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if ((net = getnetbyname(argv[3])) == NULL) {
	    Tcl_AppendResult(interp, "can not lookup \"", argv[3], "\"",
			     (char *) NULL);
	    return TCL_ERROR;
	}
	while (net->n_net && ! ((net->n_net >> 24) & 0xff)) {
	    net->n_net <<= 8;
	}
	sprintf(interp->result, "%lu.%lu.%lu.%lu",
		(net->n_net >> 24) & 0xff, (net->n_net >> 16) & 0xff,
		(net->n_net >> 8) & 0xff, net->n_net & 0xff);
	return TCL_OK;
    }

    Tcl_AppendResult(interp, "bad query \"", argv[2], 
		     "\": should be address, or name", (char *) NULL);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * NetdbProtocols --
 *
 *	This procedure is invoked to process the "netdb protocols" command.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *----------------------------------------------------------------------
 */

static int
NetdbProtocols(interp, argc, argv)
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    /*
     * Process the "netdb protocols" command option:
     */

    if (argc == 2) {
#ifdef HAVE_GETPROTOENT
	struct protoent *proto;
	char buf[20];
	int i = 0;
	
	setprotoent(0);
	while ((proto = getprotoent())) {	
	    sprintf(buf, "%d", proto->p_proto);
	    Tcl_AppendResult(interp, i++ ? " {" : "{", proto->p_name, 
			     " ", buf, "}", (char *) NULL);
	}
	endprotoent();
#endif
	return TCL_OK;
    }

    /*
     * Process the "netdb protocols name" command option:
     */

    if (strcmp(argv[2], "name") == 0) {
	struct protoent *proto;
	int num;
	if (argc != 4) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			     " protocols name number\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (Tcl_GetInt(interp, argv[3], &num) != TCL_OK) {
	    return TCL_ERROR;
	}
	if ((proto = getprotobynumber(num)) == NULL) {
	    Tcl_AppendResult(interp, "can not lookup \"", argv[3], "\"",
			     (char *) NULL);
	    return TCL_ERROR;
	}
	Tcl_SetResult(interp, proto->p_name, TCL_VOLATILE);
	return TCL_OK;
    }

    /*
     * Process the "netdb protocols number" command option:
     */

    if (strcmp(argv[2], "number") == 0) {
	struct protoent *proto;
	if (argc != 4) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0], 
			     " protocols number name\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if ((proto = getprotobyname(argv[3])) == NULL) {
	    Tcl_AppendResult(interp, "can not lookup \"", argv[3], "\"",
			     (char *) NULL);
	    return TCL_ERROR;
	}
	sprintf(interp->result, "%d", proto->p_proto);
	return TCL_OK;
    }

    Tcl_AppendResult(interp, "bad query \"", argv[2], 
		     "\": should be name, or number", (char *) NULL);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * NetdbServices --
 *
 *	This procedure is invoked to process the "netdb services" command.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *----------------------------------------------------------------------
 */

static int
NetdbServices(interp, argc, argv)
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    /*
     * Process the "netdb services" command option:
     */

    if (argc == 2) {
#ifdef HAVE_GETSERVENT
	struct servent *serv;
	char buf[20];
	int i = 0;
	
	setservent(0);
	while ((serv = getservent())) {	
	    sprintf(buf, "%d", ntohs(serv->s_port));
	    Tcl_AppendResult(interp, i++ ? " {" : "{", serv->s_name, 
			     " ", buf, " ", serv->s_proto, "}", (char *) NULL);
	}
	endservent();
#endif
	return TCL_OK;
    }

    /*
     * Process the "netdb services name" command option:
     */

    if (strcmp(argv[2], "name") == 0) {
	struct sockaddr_in addr;
	int port;
	char *name;
	if (argc != 5) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			     " services name number protocol\"", (char *)NULL);
	    return TCL_ERROR;
	}
	if (TnmGetUnsigned(interp, argv[3], &port) != TCL_OK) {
	    return TCL_ERROR;
	}
	addr.sin_port = htons(port);
	name = TnmGetIPPort(interp, argv[4], &addr);
	if (! name) {
	    return TCL_ERROR;
	}
	Tcl_SetResult(interp, name, TCL_STATIC);
	return TCL_OK;
    }

    /*
     * Process the "netdb services number" command option:
     */

    if (strcmp(argv[2], "number") == 0) {
	struct sockaddr_in addr;
	if (argc != 5) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			     " services number name protocol\"", (char *)NULL);
	    return TCL_ERROR;
	}
	if (TnmSetIPPort(interp, argv[4], argv[3], &addr) != TCL_OK) {
	    return TCL_ERROR;
	}
	sprintf(interp->result, "%d", ntohs(addr.sin_port));
	return TCL_OK;
    }

    Tcl_AppendResult(interp, "bad query \"", argv[2], 
		     "\": should be name, or number", (char *) NULL);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * NetdbSunrpcs --
 *
 *	This procedure is invoked to process the "netdb sunrpcs" command.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *----------------------------------------------------------------------
 */

static int
NetdbSunrpcs(interp, argc, argv)
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    /*
     * Process the "netdb sunrpcs" command option:
     */

    if (argc == 2) {
#ifdef HAVE_GETRPCENT
	struct rpcent *rpc;
	char buf[20];
	int i = 0;
	
	setrpcent(0);
	while ((rpc = (struct rpcent *) getrpcent())) {	
	    sprintf(buf, "%d", rpc->r_number);
	    Tcl_AppendResult(interp, i++ ? " {" : "{", rpc->r_name, 
			     " ", buf, "}", (char *) NULL);
	}
	endrpcent();
#endif
	return TCL_OK;
    }

    /*
     * Process the "netdb sunrpcs name" command option:
     */

    if (strcmp(argv[1], "name") == 0) {
	struct rpcent *rpc;
	int num;
	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			     " sunrpcs name number\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (Tcl_GetInt(interp, argv[2], &num) != TCL_OK) {
	    return TCL_ERROR;
	}
	rpc = (struct rpcent *) getrpcbynumber(num);
	if (rpc == NULL) {
	    Tcl_AppendResult(interp, "can not lookup \"", argv[2], "\"",
			     (char *) NULL);
	    return TCL_ERROR;
	}
	Tcl_SetResult(interp, rpc->r_name, TCL_STATIC);
	return TCL_OK;
    }

    /*
     * Process the "netdb sunrpcs number" command option:
     */

    if (strcmp(argv[1], "number") == 0) {
	struct rpcent *rpc;
	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			     " sunrpcs number name\"", (char *) NULL);
	    return TCL_ERROR;
	}
	rpc = (struct rpcent *) getrpcbyname(argv[2]);
	if (rpc == NULL) {
	    Tcl_AppendResult(interp, "can not lookup \"", argv[2], "\"",
			     (char *) NULL);
	    return TCL_ERROR;
	}
	sprintf(interp->result, "%d", rpc->r_number);
	return TCL_OK;
    }

    Tcl_AppendResult(interp, "bad query \"", argv[1], 
		     "\": should be name, or number", (char *) NULL);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * Tnm_NetdbCmd --
 *
 *	This procedure is invoked to process the "netdb" command.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *----------------------------------------------------------------------
 */

int
Tnm_NetdbCmd(clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			 " db ?arg arg ...?\"", (char *) NULL);
	return TCL_ERROR;
    }

    if (strcmp(argv[1], "hosts") == 0) {
	return NetdbHosts(interp, argc, argv);
    } else if (strcmp(argv[1], "ip") == 0) {
	return NetdbIp(interp, argc, argv);
    } else if (strcmp(argv[1], "networks") == 0) {
	return NetdbNetworks(interp, argc, argv);
    } else if (strcmp(argv[1], "protocols") == 0) {
	return NetdbProtocols(interp, argc, argv);
    } else if (strcmp(argv[1], "services") == 0) {
	return NetdbServices(interp, argc, argv);
    } else if (strcmp(argv[1], "sunrpcs") == 0) {
	return NetdbSunrpcs(interp, argc, argv);
    }

    Tcl_AppendResult(interp, "bad database \"", argv[1], "\": should be ",
		     "hosts, ip, networks, protocols, services, or sunrpcs",
		     (char *) NULL);
    return TCL_ERROR;
}
