/*
 * tnmIcmp.c --
 *
 *	Extend a Tcl interpreter with an icmp command. This
 *	module depends only the platform independent part.
 *
 * Copyright (c) 1993-1996 Technical University of Braunschweig.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */

#include "tnmInt.h"
#include "tnmPort.h"

/*
 * Every Tcl interpreter has an associated IcmpControl record. It
 * keeps track of the default settings for this interpreter.
 */

static char tnmIcmpControl[] = "tnmIcmpControl";

typedef struct IcmpControl {
    int retries;		/* Default number of retries. */
    int timeout;		/* Default timeout in seconds. */
    int size;			/* Default size of the ICMP packet. */
    int delay;			/* Default delay between ICMP packets. */
} IcmpControl;

/*
 * Forward declarations for procedures defined later in this file:
 */

static void
AssocDeleteProc	_ANSI_ARGS_((ClientData clientData, Tcl_Interp *interp));

static int
IcmpRequest	_ANSI_ARGS_((Tcl_Interp *interp, char *hosts, int type, 
			     int ttl, int timeout, int retries, int delay,
			     int size));


/*
 *----------------------------------------------------------------------
 *
 * AssocDeleteProc --
 *
 *	This procedure is called when a Tcl interpreter gets destroyed
 *	so that we can clean up the data associated with this interpreter.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static void
AssocDeleteProc(clientData, interp)
    ClientData clientData;
    Tcl_Interp *interp;
{
    IcmpControl *control = (IcmpControl *) clientData;

    if (control) {
	ckfree((char *) control);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * IcmpRequest --
 *
 *	This procedure is called to process a single ICMP request.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static int
IcmpRequest(interp, hosts, type, ttl, timeout, retries, delay, size)
    Tcl_Interp *interp;
    char *hosts;
    int type, ttl, timeout, retries, delay, size;
{
    int i, code, largc, len;
    char **largv;
    TnmIcmpRequest *icmpPtr;
    
    code = Tcl_SplitList(interp, hosts, &largc, &largv);
    if (code != TCL_OK) {
	return TCL_ERROR;
    }
    
    len = sizeof(TnmIcmpRequest) + largc * sizeof(TnmIcmpTarget);
    icmpPtr = (TnmIcmpRequest *) ckalloc(len);
    memset((char *) icmpPtr, 0, len);
    
    icmpPtr->type = type;
    icmpPtr->ttl = ttl;
    icmpPtr->timeout = timeout;
    icmpPtr->retries = retries;
    icmpPtr->delay = delay;
    icmpPtr->size = size;
    icmpPtr->argc = largc;
    icmpPtr->argv = largv;
    icmpPtr->targets = (TnmIcmpTarget *) icmpPtr + sizeof(TnmIcmpRequest);

    for (i = 0; i < largc; i++) {
	TnmIcmpTarget target;
	target = icmpPtr->targets[i];
	code = TnmSetIPAddress(interp, largv[i], &target.dst);
	if (code != TCL_OK) {
	    ckfree((char *) icmpPtr->argv);
	    ckfree((char *) icmpPtr);
		return TCL_ERROR;
	}
    }

    code = TnmIcmp(interp, icmpPtr);
    
    ckfree((char *) icmpPtr->argv);
    ckfree((char *) icmpPtr);

    return code;
}

/*
 *----------------------------------------------------------------------
 *
 * Tnm_IcmpCmd --
 *
 *	This procedure is invoked to process the "icmp" command.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *----------------------------------------------------------------------
 */

int
Tnm_IcmpCmd(clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    int actTimeout = -1;	/* actually used timeout */
    int actRetries = -1;	/* actually used retries */
    int actSize = -1;		/* actually used size */
    int actDelay = -1;		/* actually used delay */

    int type = 0;		/* the request type */
    int ttl = -1;		/* the time to live field */

    char *cmdName = argv[0];

    IcmpControl *control = (IcmpControl *) 
	Tcl_GetAssocData(interp, tnmIcmpControl, NULL);

    if (! control) {
	control = (IcmpControl *) ckalloc(sizeof(IcmpControl));
	control->retries = 2;
	control->timeout = 5;
	control->size = 64;
	control->delay = 0;
	Tcl_SetAssocData(interp, tnmIcmpControl, AssocDeleteProc, 
			 (ClientData) control);
    }

    if (argc == 1) {
      icmpWrongArgs:
	Tcl_AppendResult(interp, "wrong # args: should be \"", cmdName,
			 " ?-retries n? ?-timeout n? ?-size n? ?-delay n?",
			 " option ?arg? hosts\"", (char *) NULL);
	return TCL_ERROR;
    }

    /*
     * Parse the options.
     */

    argc--; argv++;
    while (argc > 0 && (*argv[0] == '-')) {
	if (strcmp(argv[0], "-retries") == 0) {
	    argc--, argv++;
	    if (argc < 1) {
		sprintf(interp->result, "%d", control->retries);
                return TCL_OK;
	    }
	    if (TnmGetUnsigned(interp, argv[0], &actRetries) != TCL_OK)
	        return TCL_ERROR;
	    argc--, argv++;
	} else 	if (strcmp(argv[0], "-timeout") == 0) {
	    argc--, argv++;
	    if (argc < 1) {
		sprintf(interp->result, "%d", control->timeout);
                return TCL_OK;
	    }
	    if (TnmGetPositive(interp, argv[0], &actTimeout) != TCL_OK) {
                return TCL_ERROR;
	    }
	    argc--, argv++;
	} else 	if (strcmp(argv[0], "-size") == 0) {
	    argc--, argv++;
	    if (argc < 1) {
		sprintf(interp->result, "%d", control->size);
		return TCL_OK;
	    }
	    if (TnmGetUnsigned(interp, argv[0], &actSize) != TCL_OK) {
                return TCL_ERROR;
	    }
	    if (actSize < 44) actSize = 44;
	    argc--, argv++;
	} else  if (strcmp(argv[0], "-delay") == 0) {
	    argc--, argv++;
	    if (argc < 1) {
		sprintf(interp->result, "%d", control->delay);
		return TCL_OK;
	    }
	    if (TnmGetUnsigned(interp, argv[0], &actDelay) != TCL_OK)
                return TCL_ERROR;
	    argc--, argv++;
	} else {
	    Tcl_AppendResult(interp, "unknown option \"", argv [0], "\"",
			     (char *) NULL);
            return TCL_ERROR;
	}
    }

    /*
     * No arguments left? Set the default values and return.
     */

    if (argc == 0) {
        if (actRetries >= 0) {
            control->retries = actRetries;
        }
        if (actTimeout > 0) {
            control->timeout = actTimeout;
        }
	if (actSize > 0) {
	    control->size = actSize;
	}
	if (actDelay >= 0) {
	    control->delay = actDelay;
	}
        return TCL_OK;
    }

    /*
     * Now we should have at least two arguments left!
     */

    if (argc < 2) {
	goto icmpWrongArgs;
    }

    actRetries = actRetries < 0 ? control->retries : actRetries;
    actTimeout = actTimeout < 0 ? control->timeout : actTimeout;
    actSize  = actSize  < 0 ? control->size  : actSize;
    actDelay = actDelay < 0 ? control->delay : actDelay;

    /*
     * Get the query type.
     */

    if (strcmp(argv [0], "echo") == 0) {
        type = TNM_ICMP_ECHO;
    } else if (strcmp(argv [0], "mask") == 0) {
        type = TNM_ICMP_MASK;
    } else if (strcmp(argv [0], "timestamp") == 0) {
        type = TNM_ICMP_TIMESTAMP;
    } else if (strcmp(argv [0], "ttl") == 0) {
        type = TNM_ICMP_TTL;
	argc--, argv++;
	if (argc < 2) {
	    goto icmpWrongArgs;
	}
	if (TnmGetUnsigned(interp, argv[0], &ttl) != TCL_OK) {
	    return TCL_ERROR;
	}
    } else if (strcmp(argv [0], "trace") == 0) {
        type = TNM_ICMP_TRACE;
	argc--, argv++;
	if (argc < 2) {
	    goto icmpWrongArgs;
	}
	if (TnmGetUnsigned(interp, argv[0], &ttl) != TCL_OK) {
            return TCL_ERROR;
        }
    } else {
	Tcl_AppendResult(interp, "bad option \"", argv[0], "\": should be ",
			 "echo, mask, timestamp, ttl, or trace",
			 (char *) NULL);
	return TCL_ERROR;
    }
    argc--, argv++;

    /*
     * There should be one argument left which contains the list
     * of target IP asdresses. 
     */

    if (argc != 1) {
	goto icmpWrongArgs;
    }

    /*
     * Create an ICMP request structure.
     */

    return IcmpRequest(interp, argv[0], type, ttl,
		       actTimeout, actRetries, actDelay, actSize);
}
