/*
 * mknbi.c  -  MaKe NetBoot Image for DOS
 *
 * Copyright (C) 1996-1998 Gero Kuhlmann   <gero@gkminix.han.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "common.h"
#include "mknbi.h"
#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif

#ifndef _MKNBI_H_DOS_
#error Included wrong header file
#endif



/*
 * Global variables
 */
char *progname;
int verbose = 0;		/* Verbosity level; currently just 0, 1 or 2 */
int usehd = 0;			/* Non-zero if ramdisk is hard disk */
int nohd = 0;			/* Non-zero if hard disk accesses not allowed */
int skipchk = 0;		/* Skip OEM name check */



/*
 * Variables private to this module
 */
static char outname[MAXPATHLEN];	/* Name of output file */
static char rdname[MAXPATHLEN];		/* Name of ramdisk image file */
static int outfile;			/* File handle for output file */
static int rdimage;			/* File handle for ramdisk image */
static int rdsize;			/* Size of ramdisk in kB */

static int cur_rec_num = -1;		/* Number of current load record */
static struct load_header header;	/* Load header */
static struct load_record *cur_rec;	/* Pointer to current load record */



/*
 * List of recognized OEM names
 */
static char *oem_list[] = {
	"IBM  5.0",		/* IBM PC-DOS version 5.0 and higher */
	"MSDOS5.0",		/* MS-DOS version 5.0 and higher */
	"MSWIN4.0",		/* Windows-95 */
	"NWDOS7.0",		/* Caldera Open-DOS */
	NULL
};



/*
 * Read one sector from ram disk image file
 */
static void readsect(buf)
unsigned char *buf;
{
  int i;

  if ((i = read(rdimage, buf, SECTSIZE)) < 0) {
	perror(rdname);
	exit(EXIT_DOS_RDREAD);
  }
  if (i != SECTSIZE) {
	fprintf(stderr, "%s: Unexpected end of ram disk image file\n",
								progname);
	exit(EXIT_DOS_RDEOF);
  }
}



/*
 * Write a buffer into the output file and update the load record
 */
static void putrec(recnum, src, size)
int recnum;
char *src;
int size;
{
  unsigned long l;
  size_t isize;
  char *buf;

  if (cur_rec_num != recnum) {
    fprintf(stderr, "%s: Internal error; image chunks mis-ordered!\n",
	    progname);
    exit(EXIT_DOS_MISORDER);
  }
  isize = ((size / (SECTSIZE + 1)) + 1) * SECTSIZE;
  if ((buf = malloc(isize)) == NULL) {
	perror(progname);
	exit(EXIT_MEMORY);
  }
  memset(buf, 0, isize);
  memcpy(buf, src, size);
  if (write(outfile, buf, isize) != isize) {
	perror(outname);
	exit(EXIT_WRITE);
  }
  free(buf);
  l = get_long(cur_rec->ilength) + isize;
  assign(cur_rec->ilength.low, htot(low_word(l)));
  assign(cur_rec->ilength.high, htot(high_word(l)));
  l = get_long(cur_rec->mlength) + isize;
  assign(cur_rec->mlength.low, htot(low_word(l)));
  assign(cur_rec->mlength.high, htot(high_word(l)));
}



/*
 * Initialize a load record
 */
static void initrec(recnum, segment, flags, vendor_size)
int recnum;
int segment;
int flags;
int vendor_size;
{
  if (++cur_rec_num != recnum) {
	fprintf(stderr, "%s: Internal error; image chunks mis-ordered!\n",
		progname);
	exit(EXIT_DOS_MISORDER);
  }

  if (cur_rec_num > 0)
	cur_rec = (struct load_record *)((unsigned char *)cur_rec +
					((cur_rec->rlength << 2) & 0x3c) +
					((cur_rec->rlength >> 2) & 0x3c));
  cur_rec->rlength      = (sizeof(struct load_record) -
                           sizeof(union vendor_data)) >> 2;
  cur_rec->rlength     |= ((vendor_size + 3) & 0x3c) << 2;
  cur_rec->rtag1        = recnum + VENDOR_OFF;
  cur_rec->rflags       = flags;
  assign(cur_rec->address.low, htot(low_word((unsigned long) segment << 4)));
  assign(cur_rec->address.high, htot(high_word((unsigned long) segment << 4)));
}



/*
 * Read the FAT from the ram disk image and convert it into 16 bit format.
 */
static unsigned char *readfat(oldnum, newnum, copynum, use16)
unsigned int oldnum;
unsigned int newnum;
unsigned int copynum;
int use16;
{
  unsigned char *oldbuf, *newbuf, *cp;
  unsigned long entry;
  unsigned int entries, i;
  __u16 *u16p;

  /*
   * Read the old FATs, and check that the first two FATs are identical. Also
   * ensure that all copies in the buffer are cleared. If we don't have to
   * create a 16 bit FAT table, we can then just return.
   */
  if ((i = (oldnum + 1) * copynum) < newnum)
	i = newnum;
  if ((oldbuf = malloc((size_t)(i * SECTSIZE))) == NULL) {
	fprintf(stderr, "%s: Not enough memory for old FAT\n", progname);
	exit(EXIT_MEMORY);
  }
  memset(oldbuf, 0, i * SECTSIZE);

  cp = oldbuf;
  for (i = 0; i < (oldnum * copynum); i++) {
	readsect(cp);
	cp += SECTSIZE;
  }
  cp = oldbuf + oldnum * SECTSIZE;
  if (copynum > 1) {
	if (memcmp(oldbuf, cp, oldnum * SECTSIZE)) {
		fprintf(stderr, "%s: First two FATs in ram disk image differ\n",
								progname);
		exit(EXIT_DOS_FATCMP);
	}
	memset(cp, 0, oldnum * (copynum - 1) * SECTSIZE);
  }
  if (!use16)
	return(oldbuf);

  /*
   * Copy the old FAT into the new buffer while adjusting each entry
   * to 16 bit.
   */
  if ((newbuf = malloc((size_t)(newnum * SECTSIZE))) == NULL) {
	fprintf(stderr, "%s: Not enough memory for new FAT\n", progname);
	exit(EXIT_MEMORY);
  }
  memset(newbuf, 0, newnum * SECTSIZE);
  entries = (oldnum * SECTSIZE * 8) / 12;
  if ((((entries * 2) / SECTSIZE) + 1) > newnum) {
	fprintf(stderr, "%s: Internal error; wrong new FAT size\n", progname);
	exit(EXIT_INTERNAL);
  }
  u16p = (__u16 *)newbuf;
  for (i = 0; i < entries; i++) {
	entry  = ttoh(getval(*((__u16 *)(oldbuf + (i * 12) / 8))));
	entry |= ttoh(getval(*((__u16 *)(oldbuf + (i * 12) / 8 + 2)))) << 16;
	entry >>= (i * 12) % 8;
	entry &= 0x0fff;
	if (entry >= 0x0ff0)
		entry |= 0xf000;
	assign(*u16p, htot((__u16)entry));
	u16p++;
  }

  free(oldbuf);
  return(newbuf);
}



/*
 * Determine number of last used cluster
 */
static unsigned int getlast(fatbuf, sectnum, use16)
unsigned char *fatbuf;
int sectnum;
int use16;
{
  __u8 *cp;
  int last;

  cp = (__u8 *)fatbuf + (sectnum * SECTSIZE) - 1;
  while (cp > (__u8 *)fatbuf && !*cp)
	cp--;
  if ((last = cp - (__u8 *)fatbuf) < 2)
	return(0);
  if (use16)
	last = ((last * 8) + 15) / 16;
  else
	last = ((last * 8) + 11) / 12;
  if (last > 2)
	last -= 2;
  else
	last = 0;
  return(last);
}



/*
 * Write a simple partition table into the boot image file
 */
static void putpart(boot, use16)
struct boot_record *boot;
int use16;
{
  struct partition *part;
  unsigned char part_buf[SECTSIZE];
  unsigned int sector, cylinder;
  int i;

  /* Setup the partition table */
  memset(part_buf, 0, SECTSIZE);
  assign(*((__u16 *)&(part_buf[BOOT_SIG_OFF])), BOOT_SIGNATURE);
  part = (struct partition *)&(part_buf[BOOT_PART_OFF]);
  part->status = BOOT_ACTIVE;
  part->type = use16 ? BOOT_TYPE_16 : BOOT_TYPE_12;
  assign(part->number_sectors.low, getval(boot->sect_num));
  assign(part->number_sectors.high, htot(0));
  memcpy(&(part->first_abs_sector), &(boot->hidden_num),
					sizeof(part->first_abs_sector));

  /* Determine starting sector of partition */
  sector = get_long(boot->hidden_num) % ttoh(getval(boot->sect_per_track));
  cylinder = get_long(boot->hidden_num) / ttoh(getval(boot->sect_per_track));
  part->first_sector = (sector + 1) & 0x00ff;
  part->first_track = (cylinder / ttoh(getval(boot->head_num))) & 0x00ff;
  part->first_head = (cylinder % ttoh(getval(boot->head_num))) & 0x00ff;

  /* Determine last sector of partition */
  sector = (get_long(boot->hidden_num) + ttoh(getval(boot->sect_num)) - 1) %
					ttoh(getval(boot->sect_per_track));
  cylinder = (get_long(boot->hidden_num) + ttoh(getval(boot->sect_num)) - 1) /
					ttoh(getval(boot->sect_per_track));
  part->last_sector = (sector + 1) & 0x00ff;
  part->last_track = (cylinder / ttoh(getval(boot->head_num))) & 0x00ff;
  part->last_head = (cylinder % ttoh(getval(boot->head_num))) & 0x00ff;

  /* Write partition table */
  putrec(RAMDISKNUM, (char *)part_buf, SECTSIZE);

  /* Insert all hidden sectors */
  memset(part_buf, 0, SECTSIZE);
  for (i = 0; i < (get_long(boot->hidden_num) - 1); i++)
	putrec(RAMDISKNUM, (char *)part_buf, SECTSIZE);
}



/*
 * Process ramdisk image file
 */
static void do_ramdisk()
{
  unsigned long sectnum;		/* number of sectors on ram disk */
  unsigned long cylnum;			/* number of cylinders on ram disk */
  unsigned long clusternum;		/* number of cluster on ram disk */
  unsigned char *fatbuf;		/* buffer for FAT */
  struct disk_geometry geometry;	/* ramdisk geometry */
  struct boot_record old_boot;		/* old boot record */
  struct boot_record *new_boot;		/* new boot record */
  static unsigned char boot_buf[SECTSIZE];
  static unsigned char copyrec_buf[SECTSIZE];
  char **ccp;
  unsigned long l;
  int use_fat16 = FALSE;		/* use 16 bit fat */
  int i;

  /*
   * Read the boot record of the ram disk image
   */
  readsect(boot_buf);
  if (getval(*((__u16 *)&(((__u8 *)boot_buf)[BOOT_SIG_OFF]))) !=
						htot(BOOT_SIGNATURE)) {
	fprintf(stderr, "%s: Ram disk image has no boot signature\n", progname);
	exit(EXIT_DOS_NOBOOT);
  }
  old_boot = *((struct boot_record *)boot_buf);

  /*
   * Check that the boot record of the ram disk image is correct. Most of
   * these checks test for requirements imposed by the ramdisk driver.
   */
  if (!skipchk) {
	for (ccp = oem_list; *ccp != NULL; ccp++)
		if (!memcmp((char *)old_boot.oem_name, *ccp,
						sizeof(old_boot.oem_name)))
			break;
	if (*ccp == NULL) {
		fprintf(stderr, "%s: Ram disk image is not a DOS disk\n",
								progname);
		exit(EXIT_DOS_NODOS);
	}
  }
  if (strncmp((char *)old_boot.fat_name, BOOT_FAT12_NAME, sizeof(old_boot.fat_name))) {
	fprintf(stderr, "%s: Ram disk image has invalid filesystem\n", progname);
	exit(EXIT_DOS_INVFS);
  }
  if (getval(old_boot.bytes_per_sect) != htot(SECTSIZE)) {
	fprintf(stderr, "%s: Ram disk image has wrong sector size\n", progname);
	exit(EXIT_DOS_SECTSIZE);
  }
  if (getval(old_boot.hidden_num.low) != htot(0) ||
				getval(old_boot.hidden_num.high) != htot(0)) {
	fprintf(stderr, "%s: Ram disk image has hidden sectors\n", progname);
	exit(EXIT_DOS_HIDDEN);
  }
  if (old_boot.fat_num != 2) {
	fprintf(stderr, "%s: Ram disk image has invalid number of FATs\n",
								progname);
	exit(EXIT_DOS_FATNUM);
  }
  if (getval(old_boot.reserved_sect) != htot(1)) {
	fprintf(stderr, "%s: Ram disk image has invalid number of "
					"reserved sectors\n", progname);
	exit(EXIT_DOS_RESVDSECT);
  }
  if (getval(old_boot.head_num) != htot(2)) {
	fprintf(stderr, "%s: Ram disk image has to be double sided\n",
								progname);
	exit(EXIT_DOS_DOUBLESIDE);
  }
  sectnum = ttoh(getval(old_boot.sect_num));
  if (!usehd) {
	/* Simulating a floppy requires certain additional checks */
	if (rdsize != 0 && sectnum != (rdsize * 2)) {
		fprintf(stderr, "%s: Actual ram disk image size doesn't "
					"match required size\n", progname);
		exit(EXIT_DOS_RDSIZE);
	}
	if (sectnum != 640 && sectnum != 720  && sectnum != 1440 &&
	    sectnum != 2400 && sectnum != 2880 && sectnum != 5760) {
		fprintf(stderr, "%s: Invalid ram disk image size "
					"(%ld sectors)\n", progname, sectnum);
		exit(EXIT_DOS_RDSIZE);
	}
  }

  /*
   * Determine number of cylinders and clusters. The total number of sectors
   * has already been determined above.
   */
  l = ttoh(getval(old_boot.sect_per_track)) * 2;
  cylnum = (sectnum + l - 1) / l;
  l = sectnum - (ttoh(getval(old_boot.sect_per_fat)) * old_boot.fat_num);
  l -= (ttoh(getval(old_boot.dir_num)) * 32) / SECTSIZE;
  l -= ttoh(getval(old_boot.reserved_sect));
  clusternum = (l + old_boot.sect_per_cluster - 1) / old_boot.sect_per_cluster;

  /*
   * Now create the new boot record. We have to change a couple of values
   * to optionally support a new 16-bit FAT if we are going to simulate
   * a hard disk drive with the ram disk.
   */
  new_boot = (struct boot_record *)boot_buf;
  new_boot->boot_id = BOOT_DEV_FD;

  /*
   * If we have to simulate a hard disk drive there are some other things
   * to care for. We have to adjust the number of tracks and sectors to
   * include the partition table, and also optionally determine the size
   * of the new FAT.
   */
  if (usehd) {
	new_boot->boot_id = BOOT_DEV_HD;
	new_boot->media_id = BOOT_MEDIA_ID;

	/* Determine new ram disk size */
	sectnum += ttoh(getval(old_boot.sect_per_track));
	if (rdsize > 0) {
		if ((rdsize * 2) < sectnum) {
			fprintf(stderr, "%s: Specified ram disk size is smaller "
			                "than image\n", progname);
			exit(EXIT_DOS_RDSIZE);
		}
		sectnum = rdsize * 2;
	}

	/* Determine new number of cylinders and sectors per track */
	l = ttoh(getval(old_boot.sect_per_track)) * 2;
	cylnum = (sectnum + l - 1) / l;
	if (cylnum > MAX_CYLS) {
		cylnum = MAX_CYLS;
		l = (sectnum + (MAX_CYLS * 2) - 1) / (MAX_CYLS * 2);
		if (l > MAX_SECTS) {
			fprintf(stderr, "%s: Ram disk size too large\n",
								progname);
			exit(EXIT_DOS_RDSIZE);
		}
		assign(new_boot->sect_per_track, htot((unsigned short)l));
	}

	/* Determine number of clusters in data area and new FAT size */
	sectnum = cylnum * ttoh(getval(new_boot->sect_per_track)) * 2;
	sectnum -= ttoh(getval(new_boot->sect_per_track));
	sectnum -= ttoh(getval(new_boot->reserved_sect));
	sectnum -= (ttoh(getval(new_boot->dir_num)) * 32) / SECTSIZE;
	l = (((BOOT_FAT12_MAX + 2) * 12) + (8 * SECTSIZE) - 1) / (8 * SECTSIZE);
	l = new_boot->fat_num * l + BOOT_FAT12_MAX * new_boot->sect_per_cluster;
	if (sectnum > l) {
		use_fat16 = TRUE;
		strncpy((char *)(new_boot->fat_name), BOOT_FAT16_NAME, sizeof(new_boot->fat_name));
		l = (new_boot->sect_per_cluster * SECTSIZE) + (2 * new_boot->fat_num);
		l = ((4 * new_boot->sect_per_cluster) + (2 * sectnum) + l - 1) / l;
	} else {
		use_fat16 = FALSE;
		strncpy((char *)(new_boot->fat_name), BOOT_FAT12_NAME, sizeof(new_boot->fat_name));
		l = (2 * new_boot->sect_per_cluster * SECTSIZE) + (3 * new_boot->fat_num);
		l = ((6 * new_boot->sect_per_cluster) + (3 * sectnum) + l - 1) / l;
	}
	assign(new_boot->sect_per_fat, htot((unsigned short)l));
	sectnum -= l * new_boot->fat_num;
	/* don't round up the following, must be a true integer division! */
	clusternum = sectnum / new_boot->sect_per_cluster;

	/* Determine new total number of sectors */
	sectnum = clusternum * new_boot->sect_per_cluster;
	sectnum += (ttoh(getval(new_boot->dir_num)) * 32) / SECTSIZE;
	sectnum += ttoh(getval(new_boot->sect_per_fat)) * new_boot->fat_num;
	sectnum += ttoh(getval(new_boot->reserved_sect));
	assign(new_boot->sect_num, htot(sectnum));

	/* Finally write the number of hidden sectors for partition table */
	assign(new_boot->hidden_num.low, getval(new_boot->sect_per_track));
	assign(new_boot->hidden_num.high, htot(0));
	sectnum += ttoh(getval(new_boot->sect_per_track));
  }

  /*
   * Generate the load record which contains the disk geometry, and
   * then write the partition table, all hidden sectors and the new
   * boot record.
   */
  initrec(RAMDISKNUM, 0, FLAG_EOF, sizeof(struct disk_geometry));
  assign(geometry.cylinders, htot(cylnum));
  assign(geometry.sect_per_track, getval(new_boot->sect_per_track));
  assign(geometry.num_sectors, htot(sectnum));
  geometry.boot_drive = new_boot->boot_id;
  geometry.no_hd_flag = nohd & 0xff;
  memcpy(&(cur_rec->vendor_data.geometry), &geometry, sizeof(geometry));
  if (usehd)
	putpart(new_boot, use_fat16);
  putrec(RAMDISKNUM, (char *)boot_buf, SECTSIZE);

  /*
   * Read the FAT from the ram disk image, convert it into 16 bit format,
   * and write it back into the output image. Also find the last used
   * sector in the ram disk image.
   */
  fatbuf = readfat(ttoh(getval(old_boot.sect_per_fat)),
					ttoh(getval(new_boot->sect_per_fat)),
					old_boot.fat_num, use_fat16);
  *fatbuf = (unsigned char)(new_boot->media_id);
  for (i = 0; i < new_boot->fat_num; i++)
	putrec(RAMDISKNUM, (char *)fatbuf,
			ttoh(getval(new_boot->sect_per_fat)) * SECTSIZE);

  sectnum = getlast(fatbuf, ttoh(getval(new_boot->sect_per_fat)), use_fat16);
  if (sectnum > clusternum)
	sectnum = clusternum;
  sectnum *= new_boot->sect_per_cluster;
  sectnum += (ttoh(getval(new_boot->dir_num)) * 32) / SECTSIZE;

  /*
   * Write the remainder of the ramdisk image.
   */
  while (sectnum > 0) {
	readsect(copyrec_buf);
	putrec(RAMDISKNUM, (char *)copyrec_buf, SECTSIZE);
	sectnum--;
  }
  if (get_long(cur_rec->ilength) > (MAX_RDSIZE * 1024L)) {
	fprintf(stderr, "%s: Ram disk image too large\n", progname);
	exit(EXIT_DOS_RDSIZE);
  }
  l = ttoh(getval(geometry.num_sectors)) * SECTSIZE;
  assign(cur_rec->mlength.low,  htot(low_word(l)));
  assign(cur_rec->mlength.high, htot(high_word(l)));
  assign(cur_rec->address.low,  htot(low_word(RDADDR)));
  assign(cur_rec->address.high, htot(high_word(RDADDR)));
}


/*
 * Dump the load record information to stderr
 */
static void dump_header(lh)
struct load_header *lh;
{
  static char *s_tags[] = { /* LOADERNUM */  "primary boot loader",
			    /* RAMDISKNUM */ "ramdisk image"};
  static char *s_flags[]= { "absolute address", "after previous segment",
			    "at end of memory", "before previos segment"};
  struct load_record *lr;
  int    i, num = 0;

  printf("\nLoad record information:\n"
	  "  Magic number:     0x%08lX\n"
	  "  Length of header: %d bytes (standard) + %d bytes (vendor)\n"
	  "  Flags:            0x%08lX\n"
	  "  Location address: %04X:%04X\n"
	  "  Execute address:  %04X:%04X\n"
	  "  Vendor data:      %s\n"
	  "\n",
	  get_long(lh->magic),
	  (lh->hlength << 2) & 0x3c,
	  (lh->hlength >> 2) & 0x3c,
	  (unsigned long)lh->hflags1 +
		((unsigned long)lh->hflags2 << 8) +
		((unsigned long)lh->hflags3 << 16),
	  ttoh(getval(lh->locn.segment)), ttoh(getval(lh->locn.offset)),
	  ttoh(getval(lh->execute.segment)), ttoh(getval(lh->execute.offset)),
	  lh->dummy);

  i  = ((lh->hlength >> 2) & 0x3c) + ((lh->hlength << 2) & 0x3c);
  lr = (struct load_record *)&(((__u8 *)lh)[i]);

  for (;;) {
	printf("Record #%d:\n"
	    "  Length of header: %d bytes (standard) + %d bytes (vendor)\n"
	    "  Vendor tag:       0x%02X (%s)\n"
	    "  Reserved flags:   0x%02X\n"
	    "  Flags:            0x%02X (%s%s)\n"
	    "  Load address:     0x%08lX%s\n"
	    "  Image length:     0x%08lX bytes\n"
	    "  Memory length:    0x%08lX bytes\n",
	    ++num,
	    (lr->rlength << 2) & 0x3c,
	    (lr->rlength >> 2) & 0x3c,
	    (int)lr->rtag1,
	    lr->rtag1 < 16 || lr->rtag1-16 >= NUM_RECORDS ? "unknown" : s_tags[lr->rtag1-16],
	    (int)lr->rtag2,
	    (int)lr->rflags, s_flags[lr->rflags & 0x03],
	    lr->rflags & FLAG_EOF ? ", last record" : "",
	    get_long(lr->address),
	    get_long(lr->address) >= 0x100000L &&
	    (lr->rflags & 0x03) == 0? " (high memory)" : "",
	    get_long(lr->ilength),
	    get_long(lr->mlength));

	if (lr->rtag1-16 == RAMDISKNUM) {
		int cyls = ttoh(getval(lr->vendor_data.geometry.cylinders));
		int sects = ttoh(getval(lr->vendor_data.geometry.num_sectors));
		int spt = ttoh(getval(lr->vendor_data.geometry.sect_per_track));
		int diskid = ttoh(getval(lr->vendor_data.geometry.boot_drive));

		printf("  Vendor data:      "
		      "%d cylinders; %d heads; %d sectors; disk id: 0x%02X\n"
	              "\n",
	              cyls,
	              (sects + (cyls * spt) - 1) / (cyls * spt),
	              spt,
		      diskid);
	} else {
		printf("  Vendor data:      %s\n"
	              "\n",
	              lr->rlength & 0xf0 ? "unknown" : "none");
	}

	if (lr->rflags & FLAG_EOF)
		break;

	i  = ((lr->rlength >> 2) & 0x3c) + ((lr->rlength << 2) & 0x3c);
	lr = (struct load_record *)&(((__u8 *)lr)[i]);
  }
}



/*
 * Print the usage
 */
static void usage()
{
  fprintf(stderr,
	"Usage: %s [-f] [-x] [-s <ramdisk size>] [-c | -n]\n"
	"\t\t\t\t[-r] <ramdisk image> [-o] <outfile>\n"
	"       %s [-h]\n"
	"       %s [-v]\n",
	  progname, progname, progname);
  exit(EXIT_USAGE);
}



/*
 * Main program
 */
void main(argc, argv)
int argc;
char **argv;
{
  char *cp;
  int vendor_size, i;

  /* Determine my own name for error output */
  if ((cp = strrchr(argv[0], '/')) == NULL)
	progname = argv[0];
  else
	progname = ++cp;

  /* Initialize option arguments */
  strcpy(outname, "");
  strcpy(rdname, "");
  rdsize = 0;

  /* Parse options */
  opterr = 0;
  while ((i = getopt(argc, argv, "cfhno:r:s:vx")) != EOF)
	switch (i) {
		case 'o': strncpy(outname, optarg, MAXPATHLEN-1);
		          outname[MAXPATHLEN-1] = '\0';
		          break;
		case 'r': strncpy(rdname, optarg, MAXPATHLEN-1);
		          rdname[MAXPATHLEN-1] = '\0';
		          break;
		case 's': if ((i = atoi(optarg)) > MAX_RDSIZE) {
				fprintf(stderr, "%s: Ramdisk size must "
				                "be < %d kB\n",
						progname, MAX_RDSIZE);
				exit(EXIT_DOS_RDSIZE);
		          }
		          rdsize = i;
		          break;
		case 'v': fprintf(stderr, VERSION"\n");
		          exit(EXIT_SUCCESS);
		case 'f': skipchk++;
		          break;
		case 'n': nohd++;
		          break;
		case 'c': usehd++;
		          break;
		case 'x': verbose++;
		          break;
		case 'h': /* fall thru */
		default:  usage();
	}

  /* Parse additional arguments */
  if (optind < argc) {
	strncpy(rdname, argv[optind++], MAXPATHLEN-1);
	rdname[MAXPATHLEN-1] = '\0';
  }
  if (optind < argc) {
	strncpy(outname, argv[optind++], MAXPATHLEN-1);
	outname[MAXPATHLEN-1] = '\0';
  }
  if (optind != argc || !*rdname || !*outname || (usehd & nohd))
	usage();

  /*
   * If we are going to simulate a floppy disk only certain ramdisk sizes
   * are allowed.
   */
  if (!usehd) {
	if (rdsize > 0 &&
	    rdsize != 320  && rdsize != 360  && rdsize != 720 &&
	    rdsize != 1200 && rdsize != 1440 && rdsize != 5760) {
		fprintf(stderr, "%s: Invalid size requested for floppy "
						"ramdisk image\n", progname);
		exit(EXIT_DOS_RDSIZE);
	}
  } else {
	if (rdsize > 0 && rdsize < MIN_RDSIZE) {
		fprintf(stderr, "%s: Requested ramdisk size too small\n",
								progname);
		exit(EXIT_DOS_RDSIZE);
	}
  }

  /* Open the input and output files */
  if ((outfile = creat(outname, 0644)) < 0) {
	perror(outname);
	exit(EXIT_DOS_IMGCREATE);
  }
  rdimage = openrd(rdname, rdsize);
  if (verbose > 0) {
	printf("Ramdisk filename = %s\n", rdname);
	printf("Output file name = %s\n", outname);
  }

  /* Initialize the boot header */
  vendor_size = (sizeof(VENDOR_ID) / sizeof(__u32) + 1) * sizeof(__u32);
  memset(&header, 0, sizeof(header));
  assign(header.magic.low,       htot(low_word(HEADER_MAGIC)));
  assign(header.magic.high,      htot(high_word(HEADER_MAGIC)));
  assign(header.locn.segment,    htot(HEADERSEG));
  assign(header.locn.offset,     htot(0));
  assign(header.execute.segment, htot(LOADERSEG));
  assign(header.execute.offset,  htot(0));
  assign(header.bootsig,         htot(BOOT_SIGNATURE));
  header.hlength         = (__u8)(((int)header.dummy - (int)&header)
                           / sizeof(__u32)) & 0x0f;
  header.hlength        |= (__u8)((vendor_size/sizeof(__u32)) << 4) & 0xf0;
  memcpy(header.dummy, VENDOR_ID, sizeof(VENDOR_ID));
  if (write(outfile, &header, sizeof(header)) < 0) {
	perror(outname);
	exit(EXIT_WRITE);
  }

  /* Initialize pointer to first load record */
  cur_rec = (struct load_record *)&(header.dummy[vendor_size]);

  /* Process the boot loader record */
  if (first_data_size > LOADERLSIZE) {
	fprintf(stderr, "%s: Boot loader too large\n", progname);
	exit(EXIT_DOS_BOOTLSIZE);
  }
  initrec(LOADERNUM, LOADERSEG, 0, 0);
  putrec(LOADERNUM, (char *) first_data, first_data_size);
  assign(cur_rec->mlength.low,  htot(low_word(LOADERMSIZE)));
  assign(cur_rec->mlength.high, htot(high_word(LOADERMSIZE)));

  /* Process the ramdisk image */
  do_ramdisk();

  /* After writing out all this stuff, finally update the boot header */
  if (lseek(outfile, 0, 0) != 0) {
	perror(outname);
	exit(EXIT_SEEK);
  }
  if (write(outfile, &header, sizeof(header)) < 0) {
	perror(outname);
	exit(EXIT_WRITE);
  }

  /* If user asked for detailed output, parse the header and output all of */
  /* the load record information */
  if (verbose > 1)
	dump_header(&header);

  exit(EXIT_SUCCESS);
}
