/***

itrafmon.c - the IP traffic monitor module
Written by Gerard Paul Java
Copyright (c) Gerard Paul Java 1997, 1998

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

***/

#include <time.h>
#include <sys/wait.h>
#include <errno.h>
#include "tcptable.h"
#include "othptab.h"
#include "packet.h"
#include "ipcsum.h"
#include "utfdefs.h"
#include "utfilter.h"
#include "othfilter.h"
#include "deskman.h"
#include "error.h"
#include "attrs.h"
#include "log.h"
#include "rvnamed.h"
#include "dirs.h"
#include "timer.h"

#define SCROLLUP 0
#define SCROLLDOWN 1

int rvnamedactive();     /* external; found in revname.c */
int killrvnamed();	 /* external; found in revname.c */

/* Hot key indicators for the bottom line */

void ipmonhelp()
{
    move(24, 1);
    printkeyhelp("Up/Dn/PgUp/PgDn", "-scrl actv win  ", stdscr);
    printkeyhelp("W", "-chg actv win  ", stdscr);
    printkeyhelp("M", "-more TCP info  ", stdscr);
    stdexitkeyhelp();
};

/* Mark general packet count indicators */

void prepare_ip_statwin(WINDOW * win)
{
    wmove(win, 0, 1);
    wprintw(win, "IP:");
    wmove(win, 0, 16);
    wprintw(win, "TCP:");
    wmove(win, 0, 31);
    wprintw(win, "UDP:");
    wmove(win, 0, 46);
    wprintw(win, "ICMP:");
    wmove(win, 0, 62);
    wprintw(win, "Non-IP:");
}

void markactive(int curwin, WINDOW * tw, WINDOW * ow)
{
    WINDOW *win1;
    WINDOW *win2;
    int x1, y1, x2, y2;

    if (!curwin) {
	win1 = tw;
	win2 = ow;
    } else {
	win1 = ow;
	win2 = tw;
    }

    getmaxyx(win1, y1, x1);
    getmaxyx(win2, y2, x2);

    wmove(win1, --y1, 68);
    wprintw(win1, " Active ");
    wmove(win2, --y2, 68);
    whline(win2, ACS_HLINE, 8);
}

void show_ip_stats(WINDOW * win, unsigned int iptotal, unsigned int tcptotal,
		   unsigned int udptotal, unsigned int icmptotal,
		   unsigned int noniptotal)
{
    wmove(win, 0, 7);
    wprintw(win, "%8d", iptotal);
    wmove(win, 0, 22);
    wprintw(win, "%8d", tcptotal);
    wmove(win, 0, 37);
    wprintw(win, "%8d", udptotal);
    wmove(win, 0, 53);
    wprintw(win, "%8d", icmptotal);
    wmove(win, 0, 70);
    wprintw(win, "%8d", noniptotal);
}


/* 
 * Scrolling and paging routines for the upper (TCP) window
 */

void scrollupperwin(struct tcptable *table, int direction, int *idx, int mode)
{
    wattrset(table->tcpscreen, STDATTR);
    if (direction == SCROLLUP) {
	if (table->lastvisible != table->tail) {
	    wscrl(table->tcpscreen, 1);
	    table->lastvisible = table->lastvisible->next_entry;
	    table->firstvisible = table->firstvisible->next_entry;
	    (*idx)++;
	    wmove(table->tcpscreen, 12, 0);
	    scrollok(table->tcpscreen, 0);
	    wprintw(table->tcpscreen, "%78c", 32);
	    scrollok(table->tcpscreen, 1);
	    printentry(table, table->lastvisible, *idx, mode);
	}
    } else {
	if (table->firstvisible != table->head) {
	    wscrl(table->tcpscreen, -1);
	    table->firstvisible = table->firstvisible->prev_entry;
	    table->lastvisible = table->lastvisible->prev_entry;
	    (*idx)--;
	    wmove(table->tcpscreen, 0, 0);
	    scrollok(table->tcpscreen, 0);
	    wprintw(table->tcpscreen, "%78c", 32);
	    scrollok(table->tcpscreen, 1);
	    printentry(table, table->firstvisible, *idx, mode);
	}
    }
}

void pageupperwin(struct tcptable *table, int direction, int *idx, int mode)
{
    int i = 1;

    wattrset(table->tcpscreen, STDATTR);
    if (direction == SCROLLUP) {
	while ((i <= 10) && (table->lastvisible != table->tail)) {
	    i++;
	    scrollupperwin(table, direction, idx, mode);
	}
    } else {
	while ((i <= 10) && (table->firstvisible != table->head)) {
	    i++;
	    scrollupperwin(table, direction, idx, mode);
	}
    }
}

/*
 * Scrolling and paging routines for the lower (non-TCP) window.
 */

void scrolllowerwin(struct othptable *table, int direction)
{
    if (direction == SCROLLUP) {
	if (table->lastvisible != table->tail) {
	    wscrl(table->othpwin, 1);
	    table->lastvisible = table->lastvisible->next_entry;
	    table->firstvisible = table->firstvisible->next_entry;
	    
	    if (table->htstat == HIND) {       /* Head indicator on? */
	        wmove(table->borderwin, 6,1);
	        whline(table->borderwin, ACS_HLINE, 8);
	        table->htstat = NOHTIND;
	    }
	    
	    printothpentry(table, table->lastvisible, 4, 0, (FILE *) NULL);
	}
    } else {
	if (table->firstvisible != table->head) {
	    wscrl(table->othpwin, -1);
	    table->firstvisible = table->firstvisible->prev_entry;
	    table->lastvisible = table->lastvisible->prev_entry;

	    if (table->htstat == TIND) {       /* Tail indicator on? */
	        wmove(table->borderwin, 6,1);
	        whline(table->borderwin, ACS_HLINE, 8);
	        table->htstat = NOHTIND;
	    }
	    
	    printothpentry(table, table->firstvisible, 0, 0, (FILE *) NULL);
	}
    }
}

void pagelowerwin(struct othptable *table, int direction)
{
    int i = 1;

    if (direction == SCROLLUP) {
	while ((i <= 3) && (table->lastvisible != table->tail)) {
	    i++;
	    scrolllowerwin(table, direction);
	}
    } else {
	while ((i <= 3) && (table->firstvisible != table->head)) {
	    i++;
	    scrolllowerwin(table, direction);
	}
    }
}

/*
 * Attempt to communicate with rvnamed, and if it doesn't respond, try
 * to start it.
 */
 
void checkrvnamed(void)
{
    int execstat = 0;
    int resp;
    pid_t cpid = 0;
    int cstat;
    extern int errno;
    
    indicate("Trying to communicate with reverse lookup server");
    if (!rvnamedactive()) {
        indicate("Starting reverse lookup server");
        
        if ((cpid = fork()) == 0) {
     	    execstat = execl(RVNDFILE, NULL);      

	    /* 
	     * execl() never returns, so if we reach this point, we have
	     * a problem.
	     */
	    	
            _exit(1);
        } else if (cpid == -1)
            errbox("Can't spawn new process; lookups will block", ANYKEY_MSG, &resp);
        else {
	    while(waitpid(cpid, &cstat, 0) < 0)
	        if (errno != EINTR)
	            break;
	            
	    if (WEXITSTATUS(cstat) == 1)
                errbox("Can't start rvnamed; lookups will block", ANYKEY_MSG, &resp);
	}    
    }
}
    
/* 
 * The IP Traffic Monitor
 */

void ipmon(int rev_look, int logging, int filtered,
	   struct filterlist *fl, struct othpoptions *ofilter,
	   unsigned int tcptm)
{
    struct sockaddr_pkt fromaddr;
    int fd;
    char tpacket[8192];
    char *packet = NULL;
    struct iphdr *ippacket;
    struct tcphdr *transpacket;	/* encapsulated packet */

    unsigned int screen_idx = 1;
    
    time_t starttime = 0, now = 0;
    time_t timeint = 0;
    
    WINDOW *statwin;
    PANEL *statpanel;

    FILE *logfile = NULL;

    int curwin = 0;

    int isclosed;

    int readlen;
    char ifname[8];

    unsigned int iptotal = 0;
    unsigned int noniptotal = 0;
    unsigned int tcptotal = 0;
    unsigned int udptotal = 0;
    unsigned int icmptotal = 0;
    unsigned int othptotal = 0;
    unsigned int traftotal = 0;

    unsigned int nbplen;

    int hdrchk;			/* IP checksum values */
    int hdrcsum;

    unsigned int iphlen, tcphlen;
    unsigned int totalhlen;

    struct tcptable table;
    struct tcptableent *tcpentry;
    int mode = 0;

    struct othptable othptbl;
    struct othptabent *othpent;
    
    int p_sstat = 0, p_dstat = 0;       /* Reverse lookup statuses prior to */
					/* reattempt in updateentry(). */
    int nifltok = 0;			/* Non-IP filter ok */

    int ch;
    int endloop = 0;
    char msgstring[80];
    int nomem = 0;

    init_tcp_table(&table);
    init_othp_table(&othptbl);

    if (rev_look)
        checkrvnamed();

    ipmonhelp();
    statwin = newwin(1, 80, 23, 0);
    statpanel = new_panel(statwin);
    wattrset(statwin, FIELDATTR);
    wmove(statwin, 0, 0);
    wprintw(statwin, "%80c", 32);
    prepare_ip_statwin(statwin);
    show_ip_stats(statwin, 0, 0, 0, 0, 0);
    markactive(curwin, table.borderwin, othptbl.borderwin);
    update_panels();
    doupdate();

    open_socket(&fd);

    /*
     * Try to open log file if logging activated.  Turn off logging
     * (for this session only) if an error was discovered in opening
     * the log file.  Configuration setting is kept.  Who knows, the
     * situation may be corrected later.
     */

    if (logging)
	logfile = opentlog();

    if (logfile == NULL)
	logging = 0;

    writelog(logging, logfile, "******** IP traffic monitor started");

    if (fd > 0)
        starttime = timeint = time((time_t *) NULL);
        
	while (!endloop) {
	    getpacket(fd, tpacket, &fromaddr, &ch, &readlen, 5, table.tcpscreen);

	    if (ch != ERR) {
		switch (ch) {
		case KEY_UP:
		    if (!curwin)
			scrollupperwin(&table, SCROLLDOWN, &screen_idx, mode);
		    else
			scrolllowerwin(&othptbl, SCROLLDOWN);
		    break;
		case KEY_DOWN:
		    if (!curwin)
			scrollupperwin(&table, SCROLLUP, &screen_idx, mode);
		    else
			scrolllowerwin(&othptbl, SCROLLUP);
		    break;
		case KEY_PPAGE:
		    if (!curwin)
			pageupperwin(&table, SCROLLDOWN, &screen_idx, mode);
		    else
			pagelowerwin(&othptbl, SCROLLDOWN);
		    break;
		case KEY_NPAGE:
		    if (!curwin)
			pageupperwin(&table, SCROLLUP, &screen_idx, mode);
		    else
			pagelowerwin(&othptbl, SCROLLUP);
		    break;
		case KEY_F(6):
		case 'w':
		case 'W':
		    curwin = ~curwin;
		    markactive(curwin, table.borderwin, othptbl.borderwin);
		    break;
		case 'm':
		case 'M':
		    mode = ~mode;
		    refreshtcpwin(&table, screen_idx, mode);
		    break;
		case 'Q':
		case 'q':
		case 'X':
		case 'x':
		case 24:
		case 27:
		    endloop = 1;
		}
	    }
	    if (readlen > 0) {
		traftotal += readlen;
		strcpy(ifname, fromaddr.spkt_device);

		fromaddr.spkt_protocol = htons(fromaddr.spkt_protocol);

		if (fromaddr.spkt_protocol != ETH_P_IP) {
		    if ((fromaddr.spkt_protocol == ETH_P_ARP) ||
			(fromaddr.spkt_protocol == ETH_P_RARP))
			nifltok = othfilterok(ofilter, fromaddr.spkt_protocol,
					      0, 0, 0, 0);
		    else
			nifltok = 1;

		    /*
		     * Display non-IP entry in lower window if filter ok,
		     * and nomem not set
		     */

		    if ((nifltok) && (!nomem))
			othpent = add_othp_entry(&othptbl, &table,
				    0, 0, NOT_IP, fromaddr.spkt_protocol,
						 fromaddr.spkt_family, (char *) tpacket, ifname, 0, 0,
					       logging, logfile, &nomem);

		    noniptotal++;
		    continue;
		}
		/* 
		 * move past Ethernet header, if present
		 */

		adjustpacket(tpacket, fromaddr.spkt_family, &packet, &readlen);

		if (packet == NULL)
		    continue;

		ippacket = (struct iphdr *) packet;
		iphlen = ippacket->ihl * 4;
		transpacket = (struct tcphdr *) (packet + iphlen);

		hdrcsum = ippacket->check;
		ippacket->check = 0;
		hdrchk = in_cksum((u_short *) ippacket, iphlen);

		nbplen = ntohs(ippacket->tot_len);

		iptotal += nbplen;
		if (ippacket->protocol == IPPROTO_TCP) {
		    tcptotal += ntohs(ippacket->tot_len);

		    /*
		     * Apply TCP filter, if active.
		     */

		    if ((filtered) &&
			(!(utfilter(fl, ippacket->saddr, ippacket->daddr,
				    ntohs(transpacket->source), ntohs(transpacket->dest))))) {
			show_ip_stats(statwin, iptotal, tcptotal, udptotal, icmptotal, noniptotal);
			update_panels();
			doupdate();
			continue;
		    }
		    tcphlen = transpacket->doff * 4;

		    totalhlen = iphlen + tcphlen;

		    tcpentry = in_table(&table, ippacket->saddr, ippacket->daddr,
					ntohs(transpacket->source),
			                ntohs(transpacket->dest), &isclosed,
			                tcptm, logging, logfile, &nomem);

		    /* 
		     * Add a new entry if it doesn't exist, and if it is a 
		     * SYN packet or any packet with more than just headers,
		     * and, to reduce the chances of stales, not a FIN.
		     */

		    if ((tcpentry == NULL) && (!(transpacket->fin)) &&
			((totalhlen < ntohs(ippacket->tot_len)) ||
			 ((totalhlen == ntohs(ippacket->tot_len)) &&
			  (transpacket->syn))))
			/*
			 * Ok, so we have a packet.  Add it if this connection
			 * is not yet closed, or if it is a SYN packet.
			 */

			if ((!nomem) && ((!isclosed) || (isclosed && transpacket->syn))) {
			    tcpentry = addentry(&table,
					 (unsigned long) ippacket->saddr,
					 (unsigned long) ippacket->daddr,
				  transpacket->source, transpacket->dest,
				    ippacket->protocol, ifname, rev_look,
						&nomem);

			    if ((tcpentry != NULL) && (logging)) {
				sprintf(msgstring, "TCP: %s:%u to %s:%u first packet received ",
					tcpentry->s_fqdn, tcpentry->sport,
				      tcpentry->d_fqdn, tcpentry->dport);

				if (transpacket->syn)
				    strcat(msgstring, "(SYN)");

				writelog(logging, logfile, msgstring);
			    }
			    if (tcpentry != NULL)
				printentry(&table, tcpentry->oth_connection,
					   screen_idx, mode);
			}
		    /* 
		     * If we had an addentry() success, we should have no
		     * problem here.  Same thing if we had a table lookup
		     * success.
		     */

		    if (tcpentry != NULL) {

			/* 
			 * Don't bother updating the entry if the connection
			 * has been previously reset.  (Does this really
			 * happen in practice?)
			 */

			if (!(tcpentry->stat & FLAG_RST)) {
			    if (rev_look) {
			        p_sstat = tcpentry->s_fstat;
			        p_dstat = tcpentry->d_fstat;
			    }
			    
			    updateentry(&table, tcpentry, transpacket, nbplen,
					logging, rev_look, logfile, &nomem);
					
			    if ((rev_look) && (((p_sstat != RESOLVED) && (tcpentry->s_fstat == RESOLVED)) ||
			       ((p_dstat != RESOLVED) && (tcpentry->d_fstat == RESOLVED)))) {
			        clearaddr(&table, tcpentry, screen_idx);
			        clearaddr(&table, tcpentry->oth_connection, screen_idx);
			    }
			    
			    printentry(&table, tcpentry, screen_idx, mode);

			    /*
			     * Special cases: Update other direction if it's
			     * an ACK in response to a FIN. 
			     *
			     *         -- or --
			     *
			     * Addresses were just resolved for the other
			     * direction, so we should also do so here.
			     */             

			    if (((tcpentry->oth_connection->finsent == 2) &&	/* FINed and ACKed */
				(ntohl(transpacket->seq) == tcpentry->oth_connection->finack)) ||
				((rev_look) &&
				(((p_sstat != RESOLVED) && (tcpentry->s_fstat == RESOLVED)) ||
				((p_dstat != RESOLVED) && (tcpentry->d_fstat == RESOLVED)))))
				printentry(&table, tcpentry->oth_connection, screen_idx, mode);
			}
		    }
		} else {	/* now for the other IP protocols */
		    if (ippacket->protocol == IPPROTO_UDP)
			udptotal += nbplen;
		    else if (ippacket->protocol == IPPROTO_ICMP) {

			/*
			 * Cancel the corresponding TCP entry if an ICMP
			 * Destination Unreachable or TTL Exceeded message
			 * is received.
			 */

			if ((((struct icmphdr *) transpacket)->type == ICMP_DEST_UNREACH) ||
			    (((struct icmphdr *) transpacket)->type == ICMP_TIME_EXCEEDED))
			    process_dest_unreach(&table, (char *) transpacket, &nomem);

			icmptotal += nbplen;
		    } else
			othptotal += nbplen;

		    /*
		     * Add and display non-TCP packet entry if memory is ok,
		     * and if filter permits
		     */

		    if ((!nomem) && (othfilterok(ofilter, ippacket->protocol,
					ippacket->saddr, ippacket->daddr,
			  ntohs(((struct udphdr *) transpacket)->source),
			ntohs(((struct udphdr *) transpacket)->dest)))) {
			othpent = add_othp_entry(&othptbl, &table,
					ippacket->saddr, ippacket->daddr,
					    IS_IP, ippacket->protocol, 0,
					    (char *) transpacket, ifname,
				       rev_look, tcptm, logging, logfile,
						 &nomem);
		    }
		}

		show_ip_stats(statwin, iptotal, tcptotal, udptotal, icmptotal, noniptotal);
	    }
	    
	    now = time((time_t *) NULL);
	    if (now - timeint >= 5) {
	        printelapsedtime(starttime, now, 6, 15, othptbl.borderwin);
	        timeint = now;
	    }
	    
	    update_panels();
	    doupdate();
	}
	
    killrvnamed();
    del_panel(table.tcppanel);
    del_panel(table.borderpanel);
    del_panel(othptbl.othppanel);
    del_panel(othptbl.borderpanel);
    del_panel(statpanel);
    update_panels();
    doupdate();
    delwin(table.tcpscreen);
    delwin(table.borderwin);
    delwin(othptbl.othpwin);
    delwin(othptbl.borderwin);
    delwin(statwin);
    close(fd);
    destroytcptable(&table);
    destroyothptable(&othptbl);
    writelog(logging, logfile, "******** IP traffic monitor stopped\n");
    if (logfile != NULL)
        fclose(logfile);
    return;
}
