/***

IPTraf
An IP-oriented Network Statistics Utility for
Ethernet and SLIP/PPP networks
   		
Written by Gerard Paul Java
Copyright (c) Gerard Paul Java 1997, 1998
   	
Version 1.1.0

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

***/

#include <curses.h>
#include <panel.h>
#include <signal.h>
#include <unistd.h>
#include <stdlib.h>
#include <time.h>
#include "deskman.h"
#include "menurt.h"
#include "stdwinset.h"
#include "utfdefs.h"
#include "othfilter.h"
#include "utfilter.h"
#include "ifstats.h"
#include "serv.h"
#include "options.h"
#include "externs.h"
#include "log.h"
#include "attrs.h"

extern void srpromisc(int mode);
extern void about();
/*
 * Get the ball rolling: The program interface routine.
 */

void program_interface(struct OPTIONS *options, int argc,
		       int opt, char *optarg)
{
    struct MENU menu;
    int endloop = 0;
    int row = 1;
    int aborted;
    int faborted;
    int filtered = 0;
    struct filterlist fl;
    struct othpoptions ofilter;
    char ifname[10];
    struct porttab *ports;

    draw_desktop();
    attrset(STDATTR);
    move(0, 1);
    printw("IPTraf %s", VERSION);

    loadothfilter(&ofilter);
    loadaddports(&ports);

    if (argc == 1) {
	about();

	initmenu(&menu, 11, 41, 7, 18);

	additem(&menu, " IP traffic ^m^onitor", "Displays current IP traffic information");
	additem(&menu, " General interface ^s^tatistics", "Displays some statistics for attached interfaces");
	additem(&menu, " ^D^etailed interface statistics", "Displays more statistics for a selected interface");
	additem(&menu, " TCP/UDP ser^v^ice monitor", "Displays statistics on TCP/UDP traffic");
	additem(&menu, " ^E^thernet station monitor", "Displays statistics on detected Ethernet stations");
	additem(&menu, " ^T^CP display filters", "Manages TCP display filters");
	additem(&menu, " Other ^p^rotocol filters", "Select which non-TCP protocols to display");
	additem(&menu, " ^O^ptions", "Set various program options");
	additem(&menu, " E^x^it", "Exits program");

	endloop = 0;

	do {
	    showmenu(&menu);
	    operatemenu(&menu, &row, &aborted);

	    switch (row) {
	    case 1:
		ipmon(options->revlook, options->logging, filtered,
		      &fl, &ofilter, options->timeout);
		break;
	    case 2:
		ifstats(options->logging, options->logspan);
		break;
	    case 3:
		selectiface(ifname, &aborted);
		if (!aborted)
		    detstats(ifname, options->logging, options->logspan);
		break;
	    case 4:
		selectiface(ifname, &aborted);
		if (!aborted)
		    servmon(ifname, ports, options->logging, options->logspan);
		break;
	    case 5:
		hostmon(options->logging, options->logspan);
		break;
	    case 6:
		tcpfilterselect(&fl, &filtered, &faborted);
		break;
	    case 7:
		othfilterselect(&ofilter);
		break;
	    case 8:
		setoptions(options, &ports);
		saveoptions(options);
		break;
	    case 9:
		endloop = 1;
		break;
	    }
	} while (!endloop);

	destroymenu(&menu);
    } else {
	switch (opt) {
	case 'i':
	    ipmon(options->revlook, options->logging, filtered,
		  &fl, &ofilter, options->timeout);
	    break;
	case 'g':
	    ifstats(options->logging, options->logspan);
	    break;
	case 'd':
	    detstats(optarg, options->logging, options->logspan);
	    break;
	case 's':
	    servmon(optarg, ports, options->logging, options->logspan);
	    break;
	case 'e':
	    hostmon(options->logging, options->logspan);
	    break;
	}
    }

    destroyporttab(ports);
    erase();
    update_panels();
    doupdate();
}

/*
 * Handler for the TERM signal.  There's nothing we can do for the KILL.
 */

void termhandler()
{
    erase();
    refresh();
    endwin();
    fprintf(stderr, "Received TERM signal\n");
    fprintf(stderr, "IPTraf terminated\n\n");
    exit(0);
}

/* 
 * Handler for the SIGSEGV, Segmentation Fault.  Tries to clear the screen
 * and issue a better message than "Segmentation fault".
 */

void segvhandler()
{
    erase();
    refresh();
    endwin();
    fprintf(stderr, "Fatal error: unable to allocate memory for a critical function\n");
    fprintf(stderr, "IPTraf terminated abnormally\n\n");
    exit(2);
}

/*
 * Command-line help facility.
 */
 
void commandhelp()
{
    printf("\nHelp on the iptraf command\n\n");
    printf("Command syntax: iptraf [ { -i | -g | -d iface | -s iface | -e } ]\n\n");
    printf("If the iptraf command is issued without any command-line options,\n");
    printf("the program comes up in interactive mode, with the various facilities\n");
    printf("accessed through the main menu.\n\n");
    printf("These options can also be supplied to the command:\n\n");
    printf("	-i		- immediately start the IP traffic monitor\n");
    printf("	-g		- immediately start the general interface statistics\n");
    printf("	-d iface	- allows you to immediately start the detailed\n");
    printf("			  on the indicated interface (iface)\n");
    printf("	-s iface	- allows you to immediately monitor TCP and UDP\n");
    printf("			  traffic on the specified interface (iface)\n");
    printf("	-e		- start the Ethernet station monitor\n\n");
    printf("IPTraf %s Copyright (c) Gerard Paul Java 1997, 1998\n", VERSION);
    printf("This software may be distributed and/or modified under the terms of the\n");
    printf("GNU General Public License, Version 2 or later as published by the Free\n");
    printf("Software Foundation.\n\n");
}

int main(int argc, char **argv)
{
    struct OPTIONS options;
    extern char *optarg;
    extern int optind;
    extern int opterr;
    extern int optopt;
    int opt = 0;

#ifndef ALLOWUSERS
    if (getuid() != 0) {
	fprintf(stderr, "\nIPTraf Version %s\n", VERSION);
	fprintf(stderr, "Copyright (c) Gerard Paul Java 1997, 1998\n\n");
	fprintf(stderr, "This program can be run only by the system administrator\n\n");
	exit(1);
    }
#endif

    if (argc > 1) {
	opterr = 0;
	opt = getopt(argc, argv, "igd:s:eh");

	if (opt == 'h') {
	    commandhelp();
	    exit(0);
	} else if (opt == '?') {
	    fprintf(stderr, "\nInvalid option or missing parameter\n\n");
	    exit(1);
	}
    }
    signal(SIGTERM, (void *) termhandler);
    signal(SIGSEGV, (void *) segvhandler);
    signal(SIGTSTP, SIG_IGN);
    signal(SIGINT, SIG_IGN);
    signal(SIGHUP, SIG_IGN);

    if (getenv("TERM") == NULL) {
	fprintf(stderr, "Your TERM variable is not set.\n");
	fprintf(stderr, "Please set it to an appropriate value.\n");
	exit(1);
    }
    loadoptions(&options);

    initscr();

    if ((LINES < 25) || (COLS < 80)) {
	endwin();
	fprintf(stderr, "\nThis program requires a screen size of at least 80 columns by 25 lines\n");
	fprintf(stderr, "Please resize your window\n\n");
	exit(1);
    }
    start_color();
    standardcolors(options.color);
    noecho();
    nonl();
    cbreak();
    curs_set(0);

    srpromisc(options.promisc);

    program_interface(&options, argc, opt, optarg);

    srpromisc(0);

    endwin();
    
    return(0);
}
