%{
/*
     YABASIC --- a tiny integrated Basic Compiler/Interpreter

     BISON - part
     
     this Program is subject to the GNU General Public License;
     see the file yabasic.c for details.
*/

#undef WINDOWS
#include "yabasic.h"     /* definitions of yabasic */
#include <malloc.h>

#ifdef HAVE_ALLOCA_H /* this macro might be set by autoconf-package */
#include <alloca.h>
#endif
void __yy_bcopy(char *,char *,int); /* prototype missing */

int yylineno=1;
int yylex(void);
%}

%union {
  double number;        /* double number */
  int token;            /* token of command */
  char *string;         /* quoted string */
  char *symbol;         /* general symbol */
  char *strsym;         /* string symbol */
}

%type <number> step_part
%type <number> const
%type <number> hashed_number

%token <number> NUMBER
%token <symbol> SYMBOL
%token <strsym> STRSYM
%token <string> STRING
%token <sep> SEP

%token FOR TO STEP NEXT GOTO GOSUB LABEL ON 
%token IF THEN ELSE ENDIF
%token PRINT INPUT RETURN DIM END AT SCREEN
%token AND OR NOT
%token NE LE GE LT GT EQ
%token READ DATA RESTORE
%token OPEN CLOSE
%token WINDOW DOT LINE CIRCLE TEXT CLEAR PRINTER
%token WAIT BELL EOFILE

%token SIN ASIN COS ACOS TAN ATAN EXP LOG SQRT MYEOF
%token INT FRAC RAN LEN VAL LEFT RIGHT MID LEN MIN MAX
%token STR INKEY CHR ASC UPPER LOWER INSTR SYSTEM SYSTEM2

%left OR
%left AND
%left NOT

%left '-' '+'
%left '*' '/'
%left '^'
%nonassoc UMINUS

%%

program: statement_list EOFILE {yylineno=-2;YYACCEPT;}
  ;

statement_list: statement
  | statement {if (errorlevel<=ERROR) {YYABORT;}}
    SEP statement_list
  ;

statement:  /* empty */
  | string_assignment 
  | assignment
  | for_loop 
  | if_clause
  | GOTO SYMBOL {create_goto($2);}
  | GOSUB SYMBOL {create_gosub($2);}
  | ON SYMBOL {create_pushdblsym($2);} GOTO {create_skipper();}
    goto_list {create_nop();}
  | ON SYMBOL {create_pushdblsym($2);} GOSUB {create_skipper();} 
    gosub_list {create_nop();}
  | LABEL SYMBOL {create_label($2);}
  | OPEN hashed_number ',' string_expression ',' STRING
    {create_myopen($2,$6);}
  | OPEN hashed_number ',' string_expression
    {create_myopen($2,"a");}
  | CLOSE hashed_number {create_myclose($2);}
  | PRINT stream_or_pos printlist semicolon {}
  | INPUT stream_or_pos prompt inputlist 
    {lastcommand->args=-lastcommand->args;}
  | READ readlist
  | DATA datalist
  | RESTORE {create_restore("");}
  | RESTORE SYMBOL {create_restore($2);}
  | RETURN {create_return();}
  | DIM dimlist 
  | OPEN WINDOW {create_openwin(0);}
  | OPEN WINDOW expression ',' expression {create_openwin(2);}
  | OPEN WINDOW expression ',' expression ',' expression {create_openwin(3);}
  | DOT expression ',' expression {create_dot();}
  | LINE expression ',' expression TO expression ',' expression 
    {create_line();}
  | CIRCLE expression ',' expression ',' expression
    {create_circle()}
  | TEXT string_expression ',' expression ',' expression {create_text(TRUE);}
  | TEXT expression ',' expression ',' string_expression {create_text(FALSE);}
  | TEXT expression ',' expression ',' expression {
	error(ERROR,"Only string allowed as TEXT");}
  | CLOSE WINDOW {create_closewin();}
  | CLEAR WINDOW {create_clearwin();}
  | CLEAR SCREEN {create_clearscreen();}
  | OPEN PRINTER {create_openprinter(0);}
  | OPEN PRINTER string_expression {create_openprinter(1);}
  | CLOSE PRINTER {create_closeprinter();}
  | WAIT expression {create_mywait();}
  | BELL {create_bell();}
  | INKEY {create_function(MYINKEY); create_popstrsym(NULL);}
  | SYSTEM2 '(' string_expression ')' {create_function(MYSYSTEM2);
	create_popdblsym(NULL);}
  | END {create_myend();};

string_assignment: STRSYM EQ string_expression {create_popstrsym($1);}
  | MID '(' STRSYM {create_pushstrptr($3);} ',' expression ',' expression ')' EQ string_expression {create_changestring(MYMID);}
  | LEFT '(' STRSYM {create_pushstrptr($3);} ',' expression ')' EQ string_expression {create_changestring(MYLEFT);}
  | RIGHT '(' STRSYM {create_pushstrptr($3);} ',' expression ')' EQ string_expression {create_changestring(MYRIGHT);}
  | STRSYM '(' {pushcounter();} indexlist ')' EQ string_expression {create_doarray($1,ASSIGNSTRINGARRAY);}
  | MID '(' STRSYM '(' {pushcounter();} indexlist ')' {create_doarray($3,GETSTRINGPOINTER);} ',' expression ',' expression ')' EQ string_expression {create_changestring(MYMID);}
  | LEFT '(' STRSYM '(' {pushcounter();} indexlist ')' {create_doarray($3,GETSTRINGPOINTER);} ',' expression ')' EQ string_expression {create_changestring(MYLEFT);}
  | RIGHT '(' STRSYM '(' {pushcounter();} indexlist ')' {create_doarray($3,GETSTRINGPOINTER);} ',' expression ')' EQ string_expression {create_changestring(MYRIGHT);}
  ;

string_expression: STRSYM {create_pushstrsym($1);}
  | string_function;
  | STRING {create_pushstr($1);}
  | string_expression '+' string_expression {create_concat();}
  | STRSYM '(' {pushcounter();} indexlist ')' {create_doarray($1,CALLSTRINGARRAY);}
  | '(' string_expression ')'
  ;

string_function: LEFT '(' string_expression ',' expression ')' {create_function(MYLEFT);}
  | RIGHT '(' string_expression ',' expression ')' {create_function(MYRIGHT);}
  | MID '(' string_expression ',' expression ',' expression ')' {create_function(MYMID);}
  | STR '(' expression ')' {create_function(MYSTR);}
  | INKEY {create_function(MYINKEY);}
  | CHR '(' expression ')' {create_function(MYCHR);}
  | UPPER '(' string_expression ')' {create_function(MYUPPER);}
  | LOWER '(' string_expression ')' {create_function(MYLOWER);}
  | SYSTEM '(' string_expression ')' {create_function(MYSYSTEM);}
  ;

assignment: SYMBOL EQ expression {create_popdblsym($1);} 
  | SYMBOL '(' {pushcounter();} indexlist ')' EQ expression {create_doarray($1,ASSIGNARRAY);}
  ;

expression: NUMBER {create_pushdbl($1);}
  | function
  | SYMBOL {create_pushdblsym($1);}
  | SYMBOL '(' {pushcounter();} indexlist ')' {create_doarray($1,CALLARRAY);};
  | '(' expression ')' 
  | expression '+' expression {create_dblbin('+');}
  | expression '-' expression {create_dblbin('-');}
  | expression '*' expression {create_dblbin('*');}
  | expression '/' expression {create_dblbin('/');}
  | expression '^' expression {create_dblbin('^');}
  | '-' expression %prec UMINUS {create_negate();}
  ;

function: SIN '(' expression ')' {create_function(MYSIN);}
  | ASIN '(' expression ')' {create_function(MYASIN);}
  | COS '(' expression ')' {create_function(MYCOS);}
  | ACOS '(' expression ')' {create_function(MYACOS);}
  | TAN '(' expression ')' {create_function(MYTAN);}
  | ATAN '(' expression ')' {create_function(MYATAN);}
  | ATAN '(' expression ',' expression  ')' {create_function(MYATAN2);}
  | EXP '(' expression ')' {create_function(MYEXP);}
  | LOG '(' expression ')' {create_function(MYLOG);}
  | SQRT '(' expression ')' {create_function(MYSQRT);}
  | INT '(' expression ')' {create_function(MYINT);}
  | FRAC '(' expression ')' {create_function(MYFRAC);}
  | RAN '(' expression ')' {create_function(MYRAN);}
  | RAN '(' ')' {create_function(MYRAN2);}
  | MIN '(' expression ',' expression ')' {create_function(MYMIN);}
  | MAX '(' expression ',' expression ')' {create_function(MYMAX);}
  | LEN '(' string_expression ')' {create_function(MYLEN);}
  | VAL '(' string_expression ')' {create_function(MYVAL);}
  | ASC '(' string_expression ')' {create_function(MYASC);}
  | INSTR '(' string_expression ',' string_expression ')' {create_function(MYINSTR);}
  | SYSTEM2 '(' string_expression ')' {create_function(MYSYSTEM2);}
  ;

const: NUMBER {$$=$1}
  | '+' NUMBER {$$=$2}
  | '-' NUMBER {$$=-$2;};

dimlist: SYMBOL '(' {pushcounter();} indexlist ')' {create_dim($1,'d');}
  | dimlist ',' SYMBOL '(' {pushcounter();} indexlist ')' {create_dim($3,'d');}
  | STRSYM '(' {pushcounter();} indexlist ')' {create_dim($1,'s');}
  | dimlist ',' STRSYM '(' {pushcounter();} indexlist ')' {create_dim($3,'s');}
  ;

indexlist: expression {inccounter();}
  | indexlist ',' expression {inccounter();}
  ;
 
for_loop: FOR SYMBOL EQ expression 
            {pushname($2);create_popdblsym($2);pushgoto();create_pushdblsym($2);}
	  TO expression 
	  step_part {
	     create_dblrelop(($8>0)?'{':'}');
             create_decide();
             pushlabel();}
	  SEP
          statement_list {
             create_pushdbl($8);
	     create_pushdblsym($2);	
             create_dblbin('+');
	     create_popdblsym($2);
             swap();popgoto();poplabel();}
          next_or_eofile next_symbol
  ;

next_or_eofile: NEXT
  | EOFILE {yylineno=-2;error(ERROR,"'next'-statement is missing"); YYABORT;}

step_part: {$$=1.0;} /* can be omitted */
  | STEP const {$$=$2;}
  ;

next_symbol:  {pop();}/* can be omitted */
  | SYMBOL {if (strcmp(pop()->pointer,$1)) {error(ERROR,"'for' and 'next' do not match"); YYABORT;}}
  ;

if_clause: IF condition {create_decide();pushlabel();}
           THEN statement_list {pushlabel();swap();poplabel();}
           else_part {poplabel();}
           endif_or_eof
  ;

endif_or_eof: ENDIF
  | EOFILE {yylineno=-2;error(ERROR,"'endif'-statement is missing"); YYABORT;}

condition: '(' condition ')'
  | condition OR condition {create_boole('|');}
  | condition AND condition {create_boole('&');}
  | NOT condition {create_boole('!');}
  | string_expression EQ string_expression {create_strrelop('=');}
  | string_expression NE string_expression {create_strrelop('!');}
  | string_expression LT string_expression {create_strrelop('<');}
  | string_expression LE string_expression {create_strrelop('{');}
  | string_expression GT string_expression {create_strrelop('>');}
  | string_expression GE string_expression {create_strrelop('}');}
  | expression EQ expression {create_dblrelop('=');}
  | expression NE expression {create_dblrelop('!');}
  | expression LT expression {create_dblrelop('<');}
  | expression LE expression {create_dblrelop('{');}
  | expression GT expression {create_dblrelop('>');}
  | expression GE expression {create_dblrelop('}');}
  | MYEOF '(' hashed_number ')' {create_testeof($3);}
  ;

else_part: /* can be omitted */
  | ELSE statement_list
  ;

inputlist: input
  | inputlist ',' input
  ;

input: SYMBOL {create_myread('d');create_popdblsym($1);}
  | SYMBOL '(' {pushcounter();} indexlist ')' 
    {create_myread('d');create_doarray($1,ASSIGNARRAY);}
  | STRSYM {create_myread('s');create_popstrsym($1);}
  | STRSYM '(' {pushcounter();} indexlist ')' 
    {create_myread('s');create_doarray($1,ASSIGNSTRINGARRAY);}
  ;

readlist: readitem
  | readlist ',' readitem
  ;

readitem: SYMBOL {create_readdata('d');create_popdblsym($1);}
  | SYMBOL '(' {pushcounter();} indexlist ')' 
    {create_readdata('d');create_doarray($1,ASSIGNARRAY);}
  | STRSYM {create_readdata('s');create_popstrsym($1);}
  | STRSYM '(' {pushcounter();} indexlist ')' 
    {create_readdata('s');create_doarray($1,ASSIGNSTRINGARRAY);}
  ;

datalist: STRING {create_strdata($1);}
  | const {create_dbldata($1);}
  | datalist ','  STRING {create_strdata($3);}
  | datalist ',' const {create_dbldata($3);}
  ;

prompt: /* possible empty */ {create_prompt("?");}
  | STRING {create_prompt($1);}
  ;

printlist:  /* possible empty */
  | expression {create_print('d');}
  | printlist ',' expression {create_print('d');} 
  | string_expression {create_print('s');} 
  | printlist ',' string_expression {create_print('s');}
  ;

stream_or_pos: /* possible empty */ {create_myswitch(0.0);}
  | AT '(' expression ',' expression ')' {create_mymove(); create_myswitch(0.0);}
  | '#' NUMBER {create_myswitch($2);}
  ;

hashed_number: '#' NUMBER {$$=$2;}
  | NUMBER {$$=$1;} /* need not contain hash */
  ;

semicolon: /* can be left out */ {create_print('n');}
  | ';'
  ;

goto_list: SYMBOL {create_goto($1);}
  | goto_list ',' SYMBOL {create_goto($3);}
  ;

gosub_list: SYMBOL {create_gosub($1);}
  | gosub_list ',' SYMBOL {create_gosub($3);}
  ;
