
(define *debug-pstore* #f)

(define-syntax (debug/rstore msg . args)
  (if *debug-pstore*
      (format $console-output-port msg . args)))

;;
;; note: <persistent-addr> and <persistent-addr-vector>
;; use identical underlying representations.  The former
;; simply appears to be an immutable, single-element 
;; instantiation of the former
;;

(define-class <persistent-addr> (<object>) :bvec 
 image-mode: 20)

(define-class <persistent-addr-vector> (<object>) :bvec 
 image-mode: 20)

(define-class <persistent-store> (<object>)
  (rstore init-keyword: #f)
  path
  ;;
  ;; NOTE:  indirect-pages must be in SLOT(2), as 
  ;;        set_indirect_page() will add the indirect page's
  ;;        vector to this list
  ;;
  (indirect-pages init-value: '())
  (pivots init-value: '#uninit)
  (local-code-ptrs init-value: '#uninit)
  (local-fn-descrs init-value: '#uninit)
  ;;
  ;; user-overridable hook for copying objects into the
  ;; pstore.
  ;;
  ;; the default implementation will copy all objects
  ;; except <<class>> instances, which it will signal an
  ;; error for (since the most common error seems to be
  ;; to forget to specify a pivot class, which often winds up
  ;; sucking in the Universe)
  ;;
  ;; the procedure, if supplied, is called with two
  ;; arguments: the persistent store and the current
  ;; relocation table
  ;;
  ;; it should return two procedures -- see make-std-copy-iner
  ;; (the default) for more details
  ;;
  (make-copy-iner init-value: '#uninit))

#|
    /pivots/ are objects which are referred to directly from
    inside the persistent image.  The most notable examples
    are class objects, when the classes aren't to be in the
    image itself (the usual case)

    The /pivot-table/ is a hash table (object-table) mapping the
    local copies of these things to their persistent identifiers.

    The persistent identifier of a pivot consists of an indirect
    page number and an offset in that page.  The reverse mapping
    of pivots -- from persistent identifiers to local objects --
    are via /indirect pages/.  The local application makes available,
    for each indirect page, a vector of the local objects which
    are the pivots.


    Almost always, the /pivots/ are classes or symbols.

    Unfortunately, because indirect pages have to be available
    when a page is swizzled -- which could occur at essentially
    any memory instruction -- arbitrary scheme code cannot be
    responsible for providing the contents of indirect pages.

    Instead, specially crafted C functions called /indirect page
    constructors/ are responsible for creating the indirect pages
    when necessary.  Note that arbitrary objects can be pivots,
    but the indirect pages have to be available BEFORE the page
    requiring it is unswizzled.

    The following /indirect page constructors/ are available:
        SYMBOL()
        CODE-PTR()
        TABLE-LOOKUP(symbol-table)

     Currently, only SYMBOL is implemented.

     Each indirect page records the /page constructor type/ and
     /instantiation id/ which is to be used to load the indirect page.  
     Since SYMBOL and CODE-PTR have no data, there is only one
     unique instance of each constructor, so their instantiation id
     is always 0.

        SYMBOL-INDIRECT-PAGE-CONSTRUCTOR       (0)
        CODE-PTR-INDIRECT-PAGE-CONSTRUCTOR     (1)
        TABLE-LOOKUP-INDIRECT-PAGE-CONSTRUCTOR (2)

     The client program is responsible for filling the <persistent-store>'s
     indirect-page-constructor-data vector with appropriate information
     for the corresponding instantiation ids.  There is a common
     instantiation data id space, shared by all page constructor types.
|#

(define SYMBOL-INDIRECT-PAGE-CONSTRUCTOR 0)

(define *std-pivots* #f)

(define (get-standard-pivots)
 (if (not *std-pivots*)
     (bind ((p (vector <vector>
		       <pair>
		       <string>
		       0    ;; BCI code ptr [3]
		       0    ;; BCI fn descr [4]
		       <double-float>
		       <byte-vector>
		       <string-table>
		       <string-ci-table>
		       <eq-table>
		       <integer-table>
		       <hash-integer-table>
		       <generic-table>
		       <symbol-table>
		       <table-bucket>
		       <closure>
		       <function>
		       <template>
		       <byte-coded>
		       <file-name>
		       <directory-name>
		       <root-dir>
		       <time>
		       <interval>
		       <allocation-area>
		       <persistent-addr>
		       <persistent-addr-vector>
		       <persistent-object-table>
		       <persistent-object-table-bucket>
		       <long-int>))
	    (t (template get-standard-pivots)))
       (vector-set! p 3 (code-pointer t))
       (vector-set! p 4 (linkage-info t))
       (set! *std-pivots* p)))
 *std-pivots*)

#|
(define *bcinterp* (find-part-in-linked-module
		    (find-linked-module "bci")
		    8901))
|#

(define (setup-persistent-store (self <persistent-store>) rs default-pivots?)
  ;;
  ;; create the 3 pivot tables
  ;; (one for regular objects, and two for code pointers & c function
  ;; descriptors)
  ;;
  (let ((cp (make-object-table))
	(fd (make-object-table))
	(piv (make-object-table)))
    ;;
    (set-pivot-tables! self piv cp fd)
    ;;
    (set-pivots! self piv)
    (set-local-code-ptrs! self cp)
    (set-local-fn-descrs! self fd)
    ;;
    (if default-pivots?
	;;
	;; set up the standard pivots in page 0
	;;
	(let ((p (get-standard-pivots)))
	  ;;
	  ;; install code-pointer and linkage-info for BCI templates
	  ;;
	  (table-insert! cp (vector-ref p 3) 3)
	  (table-insert! fd (vector-ref p 4) 4)
	  ;;
	  (setup-indirect-page self 0 p)))
    ;;
    ;; install default make-copy-iner if they didn't
    ;; specify one, which is the std-copy-iner (which handles symbols)
    ;; parameterized with the default-copy-in-obj proc (which
    ;; copies everything except <<class>>'s, which it signals 
    ;; an error on)
    ;;
    (if (eq? (make-copy-iner self) '#uninit)
	(set-make-copy-iner! 
	 self 
	 (make-make-std-copy-iner default-copy-in-obj)))
    self))

;; does this do everything we need?

(define-rstore-glue (setup-indirect-page (ps <persistent-store>)
					 (page_num <raw-int>)
					 (itemv <vector>))
{
  setup_indirect_page( ps_store, page_num, itemv );
  RETURN0();
})

(define-method initialize ((self <persistent-store>) #rest keys)
  (let* ((kvv (keyword-value-list->vector keys))
	 (mode (get-keyword-value kvv 'mode: 'open))
	 (default-pivots (get-keyword-value kvv 'default-pivots: #t))
	 (commit-record (get-keyword-value kvv 'commit-record: '(0 . 0)))
	 (m (case mode
	      ((open) 0)
	      ((create) 1)
	      ((read) 2)
	      (else (error "~s: invalid persitent-store access mode" mode)))))
    (let ((rs (open-rstore self 
			   (path self)
			   m
			   (car commit-record) 
			   (cdr commit-record))))
      (if rs
	  (begin
	    (set-rstore! self rs)
	    (setup-persistent-store self rs default-pivots))
	  (error "~a: could not ~s persistent store" (path self) mode)))))

(define (open-persistent-store (path <string>))
  (make <persistent-store>
	path: path
	mode: 'open))

(define (create-persistent-store (path <string>))
  (let ((p (make <persistent-store>
		 path: path
		 mode: 'create)))
    (commit p #f)
    p))

(define (read-persistent-store (path <string>))
  (make <persistent-store>
	path: path
	mode: 'read))

(define (close-persistent-store (ps <persistent-store>))
  (if (eq? (rstore ps) 0)
      (error "~a: persistent store already closed!" (path ps)))
  (close-pstore ps)
  (set-rstore! ps 0)
  (values))

(define-rstore-glue (close-pstore (ps <persistent-store>))
{
   rstore_close( ps_store );
   RETURN0();
})

(define *keep-lss-log* #f)

(define (set-keep-lss-log! flag)
  (set! *keep-lss-log* flag))

(define-rstore-glue (open-rstore (owner <persistent-store>)
				 (path <raw-string>)
				 (modef <raw-int>)
				 (commit_seg_num <raw-int>)
				 (commit_in_seg <raw-int>))
 literals: ((& signal)
	    "open-rstore: file ~s is locked"
	    "open-rstore: file ~s: ~a"
	    "open-rstore: file ~s not a persistent store"
	    (& <allocation-area>)
	    (& *keep-lss-log*))
{
  RStore *store;
  int keeplog_q = 0;
  static int lssmode[3] = { LSS_WRITE|LSS_LOCK,
		  	    LSS_CREATE|LSS_WRITE|LSS_LOCK,
			    0 };

  if (truish(TLREF(5)))
     keeplog_q = LSS_KEEP_LOG;

  alloc_area_class = TLREF(4);
  errno = 0;
  store = rstore_open( raw_owner, 
		       path, lssmode[modef] | keeplog_q,
		       (((off_t)commit_seg_num) << 16)
		       + commit_in_seg );
  if (store)
    {
      REG0 = RAW_PTR_TO_OBJ(store);
      RETURN1();
    }
  else
    {
      unsigned n;

      REG1 = raw_path;
      n = 2;
      if (errno == LSSERR_LOCKED)
	{
	  REG0 = LITERAL(1);
	}
      else if (errno == LSSERR_NOT_LSS)
	{
	  REG0 = LITERAL(3);
	}
      else
	{
	  REG0 = LITERAL(2);
	  REG2 = make_string( strerror(errno) );
	  n = 3;
	}
      APPLY(n,TLREF(0));
    }
})

(define-rstore-glue (root-object (ps <persistent-store>))
{
  REG0 = ps_store->default_area->entry;
  RETURN1();
})

(define-rstore-glue (commit* (ps <persistent-store>) root)
{
  REG0 = rstore_commit( ps_store, root );
  RETURN1();
})

(define-rstore-glue (set-pivot-tables! (ps <persistent-store>)
				       pivots
				       code_ptrs
				       fn_descrs)
{
  ps_store->pivot_table = pivots;
  ps_store->local_code_ptrs = code_ptrs;
  ps_store->local_fn_descrs = fn_descrs;
  RETURN0();
})

;; returns 3 values:
;;  [0] commit version
;;  [1] commit time
;;  [2] creation time

(define-rstore-glue (commit-info (ps <persistent-store>))
  literals: ((& <time>))
{
  commit_info_t ci;

  lss_get_lss_commit_info( ps_store->lss, &ci );

  REG0 = int2fx( ci.commit_version );
  REG1 = make_time_sec( ci.commit_time, TLREF(0) );
  REG2 = make_time_sec( ci.create_time, TLREF(0) );
  REG3 = int2fx( ci.prev_commit_at >> 16 );
  REG4 = int2fx( ci.prev_commit_at & 0xFFFF );
  RETURN(5);
})

(define-rstore-glue (commit-record-locator (ps <persistent-store>))
{
  off_t spare_cr = ps_store->lss->spare_commit_at;

  REG0 = cons( int2fx( spare_cr >> 16 ), int2fx( spare_cr & 0xFFFF ) );
  RETURN1();
})

(define-rstore-glue (set-relocation-table! (ps <persistent-store>) tbl)
{
  ps_store->reloc_table = tbl;
  RETURN0();
})

(define-rstore-glue (pmake-gvec (area <allocation-area>) 
				(the_class <<class>>) #rest)
{
unsigned i;

    REG0 = make_gvec_in_area( raw_area, the_class, SLOT(arg_count_reg-2), FALSE_OBJ );

    for (i=2; i<arg_count_reg; i++)
      {
	/*  note that we don't use gvec_write_fresh, because
	    we need to trigger the write barrier
	*/
	gvec_write( REG0, SLOT(i-2), reg_ref(i) );
      }
    RETURN(1);
})

(define (pgvec-alloc area the-class num-slots)
  (gvec-alloc-in-area area the-class num-slots #f))

(define (pbvec-alloc area the-class num-bytes fill)
  (bvec-alloc-in-area area the-class num-bytes fill))

(define-rstore-glue (copy-into-pstore (area <allocation-area>) item)
{
  REG0 = copy_into_pstore( area, item );
  RETURN1();
})

(define-rstore-glue (need-to-copy (ps <persistent-store>) item more)
{
  REG0 = notice_object_refs( ps_store, item, more );
  RETURN1();
})

(define-rstore-glue (num-dirty-pages (ps <persistent-store>))
{
  REG0 = int2fx( ps_store->num_dirty );
  RETURN1();
})

(define-rstore-glue (add-image-mode-handler! (ps <persistent-store>)
					     (h <raw-ptr>))
{
  rstore_add_swiz_mode_handler( ps_store, (struct swiz_mode_handler *)h );
  RETURN0();
})

(define-rstore-glue (transient-cell-mode-handler)
{
  REG0 = RAW_PTR_TO_OBJ( &SWM_transient_cell );
  RETURN1();
})
