#|------------------------------------------------------------*-Scheme-*--|
 | File:    modules/iolib/read.scm
 |
 |          Copyright (C)1997 Donovan Kolbly <d.kolbly@rscheme.org>
 |          as part of the RScheme project, licensed for free use.
 |          See <http://www.rscheme.org/> for the latest information.
 |
 | File version:     1.7
 | File mod date:    1997.11.29 23:10:40
 | System build:     v0.7.2, 97.12.21
 | Owned by module:  iolib
 |
 | Purpose:          scheme reader
 `------------------------------------------------------------------------|#

(define (read:internal-error code info)
  (error "`read' internal error (code ~d: ~s)" code info))

(define (read:error scanner line msg . info-list)
  (if line
      (apply* (string-append "line ~d: " msg) line info-list error)
      (apply* msg info-list error)))

(define (read:parse-vector scanner first last start-line)
  (bind ((class item elem-line (read:parse-object* scanner)))
    (if (eof-object? item)
	(read:error scanner 
		    start-line
		    "<open-vector> missing <close-paren>")
	(if (eq? class 'end)
	    (case item
	      ((<close-paren>) (values 'result 
				       (list->vector first)
				       start-line))
	      ((<dot>)
	       (read:error scanner
			   elem-line
			   "<dot> not understood in vector datum"))
	      (else (read:internal-error 1 item)))
	    (let ((i (cons item '())))
	      (if (null? first)
		  (read:parse-vector scanner i i start-line)
		  (begin
		    (set-cdr! last i)
		    (read:parse-vector scanner first i start-line))))))))

(define (read:parse-list scanner first last start-line)
  (bind ((class item elem-line (read:parse-object* scanner)))
    (if (eof-object? item)
	(read:error scanner start-line "<open-paren> missing <close-paren>")
	(if (eq? class 'end)
	    (case item
	      ((<close-paren>) (values 'result
				       first
				       start-line))
	      ((<dot>)
	       (if (null? first)
		   (read:error scanner elem-line "<dot> before any items")
		   (bind ((class item tail-line (read:parse-object* scanner)))
		     (if (eq? class 'result)
			 (if (eof-object? item)
			     (read:error scanner
					 elem-line
					 "unexpected <eof> after <dot>")
			     (begin
			       (set-cdr! last item)
			       (bind ((class i l (read:parse-object* scanner)))
				 (if (and (eq? class 'end)
					  (eq? i '<close-paren>))
				     (values 'result
					     first
					     start-line)
				     (read:error
				      scanner
				      l
				      "unexpected ~s inside at end of ~s"
				      i 
				      first)))))
			 (read:error scanner
				     tail-line
				     "unexpected ~s after <dot>" 
				     item)))))
	      (else (read:internal-error 2 item)))
	    (let ((i (cons item '())))
	      (if (null? first)
		  (read:parse-list scanner i i start-line)
		  (begin
		    (set-cdr! last i)
		    (read:parse-list scanner first i start-line))))))))

(define (read:parse-object* scanner)
  (bind ((token-type token-data token-line (scanner)))
    (case token-type
      ((<number> <symbol> <literal>) 
       (values 'result token-data token-line))
      ((<open-paren>) 
       (read:parse-list scanner '() '() token-line))
      ((<close-paren> <dot>)
       (values 'end token-type token-line))
      ((<open-vector>)
       (read:parse-vector scanner '() '() token-line))
      ((<curly-braced>)
       (values 'result
	       (make <curly-braced>
		     text: token-data)
	       token-line))
      ((quote unquote unquote-splicing quasiquote)
       (bind ((meta datum line (read:parse-object* scanner)))
	 (if (eq? meta 'result)
	     (if (eof-object? datum)
		 (read:error scanner
			     token-line
			     "unexpected <eof> after ~s" token-type)
		 (values 'result (list token-type datum) token-line))
	     (read:error scanner 
			 token-line
			 "unexpected ~s after ~s" datum token-type))))
      (else
       (if (eof-object? token-type)
	   (values 'result token-type token-line)
	   (read:error scanner
		       token-line
		       "Strange token in input: ~s ~s" 
		       token-type
		       token-data))))))

(define (read:parse-object scanner)
  (bind ((type result line (read:parse-object* scanner)))
    (if (eq? type 'end)
	(case result
	  ((<close-paren>) 
	   (read:error scanner line "Unmatched <close-paren>"))
	  ((<dot>)
	   (read:error scanner line "Misplaced <dot>"))
	  (else
	   (read:internal-error 3 result)))
	(values result line))))

;;
;; we use closure introspection to avoid having a flag or
;; extra argument.  Specifically, we look at the environment
;; of the "scanner" closure to find the port that's being read
;; from.
;;

(define (scanner->port scanner)
  (gvec-ref (gvec-ref scanner 0) 2))

(define (scanner-for-port port)
  (let ((meth (find-method input-port-scan-token (list port)))
	(p port)) ;; copy of port for scanner->port to use
    (lambda ()
      (meth p))))

(define-method input-port-read ((self <input-port>))
  (read:parse-object (scanner-for-port self)))

