#!/bin/sh
# This is the script "update-rc.d" used manipulating the runlevel setup.
# This version handles a configuration file for the SysV-init instead 
# of dealing with links in /etc/rc?.d/*
#
# Author: Winfried Trmper <winni@xpilot.org>
#
# Misc fixes to make "remove" work like the "real" one - should work if the
# file doesn't exist - Tom Lees <tom@lpsg.demon.co.uk>.
#
# 27.12.96 v0.2 (corresponds to v0.2 of "new-rc")

CFGFILE="/etc/runlevel.conf"
BAKCFG="/etc/runlevel.fallback"
LOCKFILE="/etc/runlevel.lock"
TMPFILE="/etc/runlevel.tmp"

valid_runlevels="0 1 2 3 4 5 6 7 8 9 S"
valid_min_seq=1
valid_max_seq=99

true=0
false=1

function print_usage() {
    cat <<EOF
Usage:   update-rc.d [-f] <basename> <action>

Actions:
	remove
        start|stop NN runlevel runlevel  ...  .  start|stop NN runlevel runlevel ...  .  ...
        defaults  [NN  |  NN-start  NN-stop]
                  (eqivalent to   start <NN-start> 2,3,4,5 . stop  <NN-stop>  0,1,6 .
                   <NN-*> defaults to 20)
EOF
}

function is_valid_runlevel() {
    if [ $# -ne 1 ]
    then
	return $false
    fi

    for lev in $valid_runlevels
    do
	if [ "$1" = "$lev" ]
	then
	    return $true
	fi
    done
    return $false
}

function is_valid_sequence() {
    if [ $# -ne 1 ]
    then
	return $false
    fi

    if [ $1 -ge $valid_min_seq -a $1 -le $valid_max_seq ]
    then
	return $true
    fi
    return $false
}

function get_runlevels_for_sequence() {
    local seq s i list
    seq=$1; shift
    for i in $*
    do
	r=${i%%:*}
	s=${i##*:}
	if [ $seq -eq $s ]
	then
	    list="$list$r,"
	fi
    done
    list=${list%%,}
    echo $list
}

function greater_sequence() {
    s=$1; shift
    for k in $*
    do
	i=${k##*:}
	[ $s -gt $i ] && return $true
    done
    return $false
}

function remove_sequence() {
    seq=$1; shift
    for i in $*
    do
	s=${i##*:}
        if [ "$s" != "$seq" ]
	then
	    echo -n "$i "
	fi
    done
    echo
}
function get_shortest_sequence() {
    level=99
    for i in $*
    do
	i=${i##*:}
	[ $i -lt $level ] && level=$i
    done
    echo $level
}

function print_config_line() {
    if [ $startseq -eq $stopseq ]
    then
	NEW_START=`get_runlevels_for_sequence $startseq $STARTLIST`
	NEW_STOP=`get_runlevels_for_sequence $stopseq $STOPLIST`
	NEW_SEQ=$startseq
	STARTLIST=`remove_sequence $startseq $STARTLIST`
	STOPLIST=`remove_sequence $stopseq $STOPLIST`
	startseq=`get_shortest_sequence $STARTLIST`
	stopseq=`get_shortest_sequence $STOPLIST`
    else
	if [ $startseq -lt $stopseq ]
	then
	    NEW_START=`get_runlevels_for_sequence $startseq $STARTLIST`
	    NEW_STOP="-"
	    NEW_SEQ=$startseq
	    STARTLIST=`remove_sequence $startseq $STARTLIST`
	    startseq=`get_shortest_sequence $STARTLIST`
	else
	    NEW_START="-"
	    NEW_STOP=`get_runlevels_for_sequence $stopseq $STOPLIST`
	    NEW_SEQ=$stopseq
	    STOPLIST=`remove_sequence $stopseq $STOPLIST`
	    stopseq=`get_shortest_sequence $STOPLIST`
	fi
    fi
    echo -e "$NEW_SEQ\t$NEW_STOP\t$NEW_START\t\t/etc/init.d/$basename" >> "$TMPFILE"
    modified="1"
}

do_while=1
opt_force=0
opt_simulate=0
while [ $# -gt 0 -a $do_while -eq 1 ]
do
    case $1 in
    -h|--help)
	print_usage
	exit 0
	;;
    -f|--force)
	opt_force=1
	shift
	;;
    -n)
	opt_simulate=1
	shift
	;;
    *)
	do_while=0
	;;
    esac
done

if [ $# -lt 2 ]
then
    echo "update-rc.d: too few arguments." >&2
    print_usage >&2
    exit 1
fi

basename="$1"; shift
if [ $opt_force -eq 0 ]; then
    if [ "$1" = "remove" ]; then
	if [ -f "/etc/init.d/$basename" ]
	then
	    echo "update-rc.d: warning /etc/init.d/$basename still exist. Terminating" >&2
	    exit 1
	fi
    else
	if ! [ -f "/etc/init.d/$basename" ]
	then
	    echo "update-rc.d: warning /etc/init.d/$basename doesn't exist. Terminating" >&2
	    exit 1
	fi

	if ! [ -x "/etc/init.d/$basename" ]
	then
	    echo "update-rc.d: warning /etc/init.d/$basename is not executable. Terminating" >&2
	    exit 1
	fi
    fi
fi

function element() {
    local element list IFS

    element="$1"
    case "$element" in
	reboot | R) element=0 ;;
	single | S) element=1 ;;
	halt   | H) element=6 ;;
    esac
	
    [ "$2" = "in" ] && shift
    list="$2"
    [ "$list" = "-" ] && return 1
    [ "$list" = "*" ] && return 0

    IFS=","
    set -- $list
    case $element in
	"$1" | "$2" | "$3" | "$4" | "$5" | "$6" | "$7" | "$8" | "$9")
	    return 0
    esac
    return 1
}


function unique() {
    local IFS level_set
    level_set="$2"
    echo -n "$level_set"

    IFS=","
    set -- $1
    for i in $@
    do
	if ! element $i in $level_set
	then
	    echo ",$i"
	fi
    done
}


START_SORT_NO=""
STOP_SORT_NO=""
STARTLEVELS=""
STOPLEVELS=""

STARTLIST=
STOPLIST=
action="$1"
case "$action" in
    defaults)
	STARTLEVELS="2 3 4 5"
	STOPLEVELS="0 1 6"
	case "$#" in
	    "1")
		START_SORT_NO="20"
		STOP_SORT_NO="20"
		;;
	    "2")
		START_SORT_NO="$2"
		STOP_SORT_NO="$2"
		;;
	    "3")
		START_SORT_NO="$2"
		STOP_SORT_NO="$3"
		;;
	esac

	if ! is_valid_sequence $START_SORT_NO || ! is_valid_sequence $STOP_SORT_NO
	then
	    echo "Invalid sequence $START_SORT_NO or $STOP_SORT_NO."
	    exit 1
	fi
	for lev in $STARTLEVELS; do
	    STARTLIST="$STARTLIST$lev:$START_SORT_NO "
	done
	for lev in $STOPLEVELS; do
	    STOPLIST="$STOPLIST$lev:$STOP_SORT_NO "
	done
	action=add
	;;
    remove)
	START_SORT_NO="*"
	STOP_SORT_NO="*"
	;;
    start|stop)
	# Loop over the remaining arguments
	while [ $# -gt 0 ]
	do
 	    if [ $# -gt 2 ]
 	    then
 		type="$1"; shift
 		seq="$1"; shift
 		levels=
		if [ "$type" != "start" -a "$type" != "stop" ]
		then
		    echo "Invalid type $type."
		    exit 1
		fi
 		if ! is_valid_sequence $seq
 		then
 		    echo "Invalid sequence $seq."
 		    exit 1
 		fi
 		while [ $# -gt 0 -a "$1" != "." ]
 		do
		    if ! is_valid_runlevel $1
		    then
			echo "Invalid runlevel $1."
			exit 1
		    fi
 		    levels="$levels$1 "; shift
 		done
 		if [ $# -gt 0 -a "$1" = "." ]
 		then
 		    shift
 		fi
		case "$type" in
		"start")
		    for lev in $levels; do
			STARTLIST="$STARTLIST$lev:$seq "
		    done
		    ;;
		"stop")
		    for lev in $levels; do
			STOPLIST="$STOPLIST$lev:$seq "
		    done
		    ;;
		esac
 	    else
 		echo "Too few arguments."
 		print_usage
 		exit 1
 	    fi
	done
	action=add
	;;
    *)
	print_usage
	;;
esac

function remove_lock() {
    rm -f "$LOCKFILE"
}

  # wait for any lock to vanish
i=0
while [ -f "$LOCKFILE" ]
do
    read pid < "$LOCKFILE"
    if ! kill -0 $pid &> /dev/null
    then
        remove_lock
        break
    fi
    if [ "$i" -gt "5" ]
    then
        echo "Process no. '$pid' is locking the configuration database. Terminating." >&2
        exit 1
    fi
    sleep 2
    let i+=1
done

  # lock the configuration file
echo -e "$$\n" > "$LOCKFILE"


skip=""
rm -f $TMPFILE
touch $TMPFILE

stopseq=`get_shortest_sequence $STOPLIST`
startseq=`get_shortest_sequence $STARTLIST`
SORT_NO=0
seen=0
while read LINE
do
    remove=0

    case $LINE in
	\#* | "" )
	    echo "$LINE" >> "$TMPFILE"
	    continue
    esac

    set -- $LINE
    SORT_NO="$1"; STOP="$2"; START="$3"; CMD="$4"

    if [ "$CMD" = "/etc/init.d/$basename" ] 
    then
	[ "$action" = "remove" ] && remove=1
	[ "$action" = "add" -a $opt_force -eq 1 ] && remove=1
	[ $opt_force -eq 0 ] && seen=1
    fi

    if [ $seen -eq 0 ]
    then
	if [ $SORT_NO -gt $stopseq -o $SORT_NO -gt $startseq ]
	then
	    print_config_line
	fi
    fi

    if [ $remove -ne 1 ]
    then
	echo -e "$SORT_NO\t$STOP\t$START\t\t$CMD" >> "$TMPFILE"
    else
	modified="1"
    fi

done < "$CFGFILE"

if [ $seen -eq 0 ]
then
    while [ -n "$STARTLIST" -o  -n "$STOPLIST" ]
    do
	print_config_line
    done
fi

remove_lock

if [ -z "$modified" ]
then
    echo "Nothing to do."
    rm -f "$TMPFILE"
else
    umask=022
    mv "$TMPFILE" "$CFGFILE"
fi
exit 0
