#include <stdio.h>
#include <fcntl.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "dinstall.h"
#include "lang.h"
#ifdef _TESTING_
  #include <newt.h>
#endif

/*
 * Strings and other constants
 */
#ifndef _TESTING_
#define NC_ETC_DIR       "/target/etc"
#define NC_ETC_INIT_DIR  "/target/etc/init.d"
#define NC_HOSTNAME_FILE "/target/etc/hostname"
#define NC_NETWORKS_FILE "/target/etc/networks"
#define NC_RESOLV_FILE   "/target/etc/resolv.conf"
#define NC_NETWORK_FILE  "/target/etc/init.d/network"
#define NC_HOSTS_FILE    "/target/etc/hosts"
#define NC_PCMCIA_DIR    "/target/etc/pcmcia"
#define NC_PCMCIA_FILE   "/target/etc/pcmcia/network.opts"
#define NC_ROOT_DIR      "/target/root"
#define NC_PPP_FILE      "/target/root/ppp"
#else
#define NC_ETC_DIR       "etc"
#define NC_ETC_INIT_DIR  "etc/init.d"
#define NC_HOSTNAME_FILE "etc/hostname"
#define NC_NETWORKS_FILE "etc/networks"
#define NC_RESOLV_FILE   "etc/resolv.conf"
#define NC_NETWORK_FILE  "etc/init.d/network"
#define NC_HOSTS_FILE    "etc/hosts"
#define NC_PCMCIA_DIR    "etc/pcmcia"
#define NC_PCMCIA_FILE   "etc/pcmcia/network.opts"
#define NC_ROOT_DIR      "root"
#define NC_PPP_FILE      "root/ppp"
#endif

#define OPT_LIM    (100)
#define NC_NI_NONE (0)
#define NC_HOST_TEXT MSG_CHOOSE_HOST_L
#define NC_HOST_TITLE MSG_CHOOSE_HOST
#define NC_DOMAIN_TEXT MSG_CHOOSE_DOMAIN_L
#define NC_DOMAIN_TITLE MSG_CHOOSE_DOMAIN
#define NC_IPADDR_ERROR MSG_IPADDR_ERROR 
#define NC_NETMASK_ERROR MSG_NETMASK_ERROR

#define NC_GW_INTRO MSG_CHOOSE_GW_INTRO_L
#define NC_GW_INTRO_TITLE MSG_CHOOSE_GW_INTRO
#define NC_GATEWAY_ERROR MSG_GATEWAY_ERROR 
#define NC_BROADCAST_ERROR MSG_BROADCAST_ERROR

#define IP_ADDR_SZ (4)
#define IP_ADDR_LN (3*IP_ADDR_SZ+4)
typedef struct ip_addr
{
  int i[IP_ADDR_SZ];
} ip_addr_t;


/*
 * Global variables for this module
 */
char* host=NULL;
char* domain=NULL;
char* hostname=NULL;
char* transceiver=NULL;
char* netinterface=NULL;
ip_addr_t ipaddr={{192,168,1,1}}; 
ip_addr_t netmask={{255,255,255,0}};
ip_addr_t network;
int has_gw=0;
ip_addr_t gateway={{0,0,0,0}};
ip_addr_t broadcast;
char *nameserver=NULL;

/*
 * Function prototypes
 */
extern void input_parameter(char*, char*, char**);
int check_dir(char*);
int get_host();                      /* Used to be get_required_parameters() */
int atoIP(char*,ip_addr_t*);
char* IPtoa(ip_addr_t);
char* IPtostr(char*,ip_addr_t);
int activate_network();
extern int activate_pcmcia();

/*
 * Checks a string with an IP-address for validity and converts it into an
 * ip_addr_t type;
 */
int atoIP(char* addr, ip_addr_t* iaddr)
{
  char* end;
  char* tmp_a;
  char* current=strdup(addr);
  char* next=NULL;
  int ix=0;
  int tmp;

  tmp_a=current;
  end=current+strlen(current)+1;
  while(next!=end)
    {
      next=strchr(current,'.');
      if(next==NULL)
	next=end;
      else
	{
	  *next='\0';
	  next++;
	}
      
      if(ix==IP_ADDR_SZ)
	{
	  free(tmp_a);
	  return 255;
	}
      else
	{
	  tmp=atoi(current);
	  if((tmp<0)||(tmp>255))
	    {
	      free(tmp_a);
	      return 255;
	    }
	  iaddr->i[ix++]=tmp;
	  current=next;
	}
    }
  free(tmp_a);
  if(ix!=IP_ADDR_SZ)
    return 255;
  return 0;
}

char* IPtoa(ip_addr_t iaddr)
{
  int length=3*IP_ADDR_SZ+4;
  char* addr=calloc(length,sizeof(char));
  snprintf(addr,length,"%d.%d.%d.%d",
	   iaddr.i[0],
	   iaddr.i[1],
	   iaddr.i[2],
	   iaddr.i[3]);
  return addr;
}

char* IPtostr(char *s, ip_addr_t iaddr)
{
  snprintf(s,3*IP_ADDR_SZ+4,"%d.%d.%d.%d",
	   iaddr.i[0],
	   iaddr.i[1],
	   iaddr.i[2],
	   iaddr.i[3]);
  return s;
}

/* Ask a generic IP to user */
int ask_for_IP(ip_addr_t *ip, char *title, char *mesg, char *err)
{
  char *def;
  char *addr=IPtoa(*ip);
  for(;;)
    {
      def=addr;
      addr=inputBox(title,mesg,def);
      free(def);
      if(addr==NULL)
       return -1;
      if(atoIP(addr,ip)!=0)
       problemBox(err,MSG_PROBLEM);
      else
       break;
    }
  free(addr);
  return 0;
}

/* Used to be get_required_parameters() */
int get_host()
{
  char *def;

  if(host)
    def=host;
  else
    def=strdup("debian");
  snprintf(prtbuf,PRTBUFSIZE,NC_HOST_TEXT,def);
  host=inputBox(prtbuf,NC_HOST_TITLE,def);
  free(def);
  if(host==NULL)
    return 255;

  if(strchr(host,' ')!=NULL)
    {
      problemBox(MSG_HOST_ERROR_1WORD,MSG_PROBLEM);
      return 1;
    }
  else if(strchr(host,'.')!=NULL)
    {
      problemBox(MSG_HOST_ERROR_DOT,MSG_PROBLEM);
      return 1;
    }

#ifdef _TESTING_
  snprintf(prtbuf,PRTBUFSIZE,MSG_YOU_SELECTED,host);
  problemBox(prtbuf,MSG_SELECTION);
#endif
  return 0;
}

/* Ask for the broadcast IP address */
int get_broadcast()
{
  struct d_choices *opt;
  int ix,rs;
  ip_addr_t alt;

  for(ix=0;ix<IP_ADDR_SZ;ix++)
      alt.i[ix]=netmask.i[ix]&ipaddr.i[ix];
  opt=calloc(3,sizeof(struct d_choices));
  opt[0].tag=IPtoa(broadcast);
  opt[0].string=strdup(MSG_LAST_BITS_2ONE);
  opt[1].tag=IPtoa(alt);
  opt[1].string=strdup(MSG_LAST_BITS_2ZERO);
  opt[2].tag=strdup(MSG_ANOTHER);
  opt[2].string=strdup(MSG_CHOOSE_ANOTHER_BROADCAST);
  rs=menus(MSG_CHOOSE_BROADCAST_L,MSG_CHOOSE_BROADCAST,opt,3);
  switch(rs)
    {
    case 1:
      for(ix=0;ix<IP_ADDR_SZ;ix++)
	broadcast.i[ix]=alt.i[ix];
      break;
    case 2:
      if (ask_for_IP(&broadcast,MSG_CHOOSE_BROADCAST_L2,MSG_CHOOSE_BROADCAST2
               ,NC_BROADCAST_ERROR))
	return -1;
      break;
    default:
    }
  free(opt[0].tag);
  free(opt[0].string);
  free(opt[1].tag);
  free(opt[1].string);
  free(opt[2].tag);
  free(opt[2].string);
  return 0;
}

/* Check for valid IPs in nameservers' string */
int check_ip_nameserver(char *ns)
{
  ip_addr_t ip;
  char msg[100];
  char *buf=strdup(ns);
  char *cur=buf;
  char *tmp,*end;
  end=buf+strlen(ns);
  do
    {
    if ((tmp=strstr(cur," "))!=NULL)
      *tmp='\0';
    if (atoIP(cur,&ip)!=0)
      {
      snprintf(msg,100,MSG_IP_INVALID,cur);
      problemBox(msg,MSG_PROBLEM);
      free(buf);
      return -1;
      }
    cur=(tmp==NULL)?end:tmp+1;  
    }
  while (cur<end);
  free(buf);
  return 0;
}

int get_net_parameters()
{
  char tmpstr[IP_ADDR_LN];
  char tmpstr2[IP_ADDR_LN];
  char tmpstr3[IP_ADDR_LN];
  char tmpstr4[IP_ADDR_LN];
  char* def;
  char* addr=NULL;
  int slength;
  struct d_choices *opt;
  int rs;
  int ix;
  
  /*
   * Get domain name.
   */
  for(;;)
    {
      if(domain)
	{
	  def=strdup(domain);
	  free(domain);
	}
      else
	def=strdup("");
      domain=inputBox(NC_DOMAIN_TEXT,NC_DOMAIN_TITLE,def);
      free(def);
      if(domain==NULL)
	return 255;
      else
	break;
    }

  /*
   * Confirm the fully qualified domain name
   */
  slength=strlen(host)+strlen(domain)+2;
  hostname=calloc(slength,sizeof(char));
  snprintf(hostname,slength,"%s.%s",host,domain);

  snprintf(prtbuf,PRTBUFSIZE,MSG_CONFIRM_FULL_NAME,
	   hostname);

  if(yesNoBox(prtbuf,MSG_CONFIRM)==0)
    return 255;

  /*
   * Getting the IP address
   */
  snprintf(prtbuf,PRTBUFSIZE,MSG_CHOOSE_IP_L,host);
  if (ask_for_IP(&ipaddr,prtbuf,MSG_CHOOSE_IP,NC_IPADDR_ERROR))
	return -1;

  /*
   * Getting the Netmask
   */
  if (ask_for_IP(&netmask,MSG_CHOOSE_NETMASK_L,
		MSG_CHOOSE_NETMASK_L,NC_NETMASK_ERROR))
	return -1;

  /*
   * Generating the network, broadcast and default gateway address
   * Default gateway address is network with last number set to "1", 
   * or "2" if "1" is the local address.
   */
  for(ix=0;ix<IP_ADDR_SZ;ix++)
    {
      gateway.i[ix]=network.i[ix]=ipaddr.i[ix]&netmask.i[ix];
      broadcast.i[ix]=(~netmask.i[ix]&255)|ipaddr.i[ix];
    }
  gateway.i[IP_ADDR_SZ-1]|=
    (ipaddr.i[IP_ADDR_SZ-1]==(network.i[IP_ADDR_SZ-1]|1))?2:1;


  /*
   * Getting the broadcast address
   */
  if (get_broadcast())
    return -1;


  /*
   * Getting the Gateway
   */
  if(yesNoBox(NC_GW_INTRO,NC_GW_INTRO_TITLE)!=0)
    {
      has_gw=1;
      if (ask_for_IP(&gateway,MSG_CHOOSE_IP_GW_L,
		MSG_CHOOSE_IP_GW,NC_GATEWAY_ERROR))
	return -1;
    }
  else
    has_gw=0;

  /*
   * Getting the DNS server list.
   */
  opt=calloc(3,sizeof(struct d_choices));
  opt[0].string=strdup("1");
  snprintf(prtbuf,PRTBUFSIZE,MSG_OWN_DNS,host);
  opt[0].tag=strdup(prtbuf);
  opt[1].string=strdup("2");
  snprintf(prtbuf,PRTBUFSIZE,MSG_ANOTHER_DNS,host);
  opt[1].tag=strdup(prtbuf);
  opt[2].string=strdup("3");
  opt[2].tag=strdup(MSG_NO_DNS);
  rs=menus(MSG_CHOOSE_DNS_L,MSG_CHOOSE_DNS,opt,3);
  switch(rs)
    {
    case 0:
      nameserver=strdup("127.0.0.1");
      break;
    case 1:
      if(addr)
	free(addr);
      addr=(nameserver==NULL)?IPtoa(gateway):strdup(nameserver);
      for(;;)
	{
	  def=addr;
	  addr=inputBox(MSG_CHOOSE_DNS_L2,MSG_CHOOSE_DNS2,def);
	  free(def);
	  if(addr==NULL)
	    return 255;
	  if (check_ip_nameserver(addr)==0)
            break;
	}
      nameserver=strdup(addr);
      break;
    case 2:
      nameserver=strdup("none");
    }
  
  free(opt[0].tag);
  free(opt[0].string);
  free(opt[1].tag);
  free(opt[1].string);
  free(opt[2].tag);
  free(opt[2].string);
  /*
   * Confirmation
   */
  snprintf(prtbuf,PRTBUFSIZE,MSG_NET_CONFIRM_L1,
		IPtostr(tmpstr,ipaddr),
		IPtostr(tmpstr2,netmask),
		IPtostr(tmpstr3,network),
		IPtostr(tmpstr4,broadcast));
  if(has_gw)
    snprintf(prtbuf+strlen(prtbuf),PRTBUFSIZE-strlen(prtbuf),
    		MSG_NET_CONFIRM_L2, IPtostr(tmpstr,gateway));
  snprintf(prtbuf+strlen(prtbuf),PRTBUFSIZE-strlen(prtbuf),
    		MSG_NET_CONFIRM_L3, nameserver);
  if(yesNoBox(prtbuf,MSG_CONFIRMATION)==0)
    return 1;
  
  free(addr);
  return 0;
}

int write_network_configuration ()
{
  FILE *p_file, *pcmcia_file;
  char *tmp,*ns,*aux;
  int ix;

  /*
   * Create /target/etc and /target/etc/init.d if they don't exist
   */
  if(check_dir(NC_ETC_DIR)==-1)
    mkdir(NC_ETC_DIR,0755);
  if(check_dir(NC_ETC_INIT_DIR)==-1)
    mkdir(NC_ETC_INIT_DIR,0755);
  
  p_file=fopen(NC_HOSTNAME_FILE,"w");
  fprintf(p_file,"%s\n",host);
  fclose(p_file);
  
  p_file=fopen(NC_NETWORKS_FILE,"w");
  fprintf(p_file,"localnet %s\n",IPtostr(prtbuf,network));
  fclose(p_file);
  
  if(strcmp(domain,"none"))
    {
      p_file=fopen(NC_RESOLV_FILE,"w");
      fprintf(p_file,"search %s\n",domain);
      
      if(strcmp(nameserver,"none"))
	{
	  aux=ns=strdup(nameserver);
	  while((tmp=strchr(ns,' '))!=NULL)
	    {
	      *tmp='\0';
	      tmp++;
	      fprintf(p_file,"nameserver %s\n",ns);
	      ns=tmp;
	    }
	  fprintf(p_file,"nameserver %s\n",ns);
	  free(aux);
	}
      fclose(p_file);
    }

  p_file=fopen(NC_NETWORK_FILE,"w");
  fprintf(p_file,
	  "#! /bin/sh\nifconfig lo 127.0.0.1\nroute add -net 127.0.0.0\n");
  fclose(p_file);
  chmod(NC_NETWORK_FILE,0755);
  
  
  if(strcmp(netinterface,"none"))
    if(strcmp(netinterface,"pcmcia")) /* PCMCIA uses a different file */
      {
	p_file=fopen(NC_NETWORK_FILE,"a");
	  {
	    fprintf(p_file,"IPADDR=%s\n",IPtostr(prtbuf,ipaddr));
	    fprintf(p_file,"NETMASK=%s\n",IPtostr(prtbuf,netmask));
	    fprintf(p_file,"NETWORK=%s\n",IPtostr(prtbuf,network));
	    fprintf(p_file,"BROADCAST=%s\n",IPtostr(prtbuf,broadcast));
	    if(has_gw)
	      fprintf(p_file,"GATEWAY=%s\n",IPtostr(prtbuf,gateway));
	    else
	      fprintf(p_file,"GATEWAY=\n");

	    fprintf(p_file,
          "ifconfig %s ${IPADDR} netmask ${NETMASK} broadcast ${BROADCAST}\n",
		    netinterface);
	    fprintf(p_file,"route add -net ${NETWORK}\n");
	    fprintf(p_file,
	  "[ \"${GATEWAY}\" ] && route add default gw ${GATEWAY} metric 1\n");
	  }
	fclose(p_file);
      }

  p_file=fopen(NC_HOSTS_FILE,"w");
  
  if((!strcmp(netinterface,"eth0"))||
     (!strcmp(netinterface,"tr0")))
    {
      fprintf(p_file,"127.0.0.1\tlocalhost\n");
      fprintf(p_file,"%s\t%s\t%s\n",
	      IPtostr(prtbuf,ipaddr),
	      hostname,
	      host);
    }
  else if(!strcmp(netinterface,"none"))
    {
      fprintf(p_file,"127.0.0.1\t%s\tlocalhost\n",host);
    }
  else if((!strcmp(netinterface,"slip"))||
	  (!strcmp(netinterface,"ppp"))||
	  (!strncmp(netinterface,"plip",4)))
    {
      fprintf(p_file,"127.0.0.1\tlocalhost\n");
      fprintf(p_file,"%s\t%s\t%s\n",
	      IPtostr(prtbuf,ipaddr),
	      hostname,
	      host);
      problemBox(MSG_NET_INFO,MSG_INFORMATION);
    }
  else if(!strcmp(netinterface,"pcmcia"))
    {
      fprintf(p_file,"127.0.0.1\tlocalhost\n");
      fprintf(p_file,"%s\t%s\t%s\n",
	      IPtostr(prtbuf,ipaddr),
	      hostname,
	      host);
      
      if(check_dir(NC_PCMCIA_DIR)==-1)
	mkdir(NC_PCMCIA_DIR,0755);
      
      pcmcia_file=fopen(NC_PCMCIA_FILE,"w");
      fprintf(pcmcia_file,"# Network adapter configuration\n#\n# Automatically generated during the Debian Installation\n#\ncase \"$ADDRESS\" in\n*,*,*,*)\n");
      fprintf(pcmcia_file,"\tIF_PORT=\"%s\"\n",transceiver);
      fprintf(pcmcia_file,"\t# Use BOOTP [y/n]\n\tBOOTP=\"n\"\n");
      fprintf(pcmcia_file,"\tIPADDR=\"%s\"\n",IPtostr(prtbuf,ipaddr));
      fprintf(pcmcia_file,"\tNETMASK=\"%s\"\n",IPtostr(prtbuf,netmask));
      fprintf(pcmcia_file,"\tNETWORK=\"%s\"\n",IPtostr(prtbuf,network));
      fprintf(pcmcia_file,"\tBROADCAST=\"%s\"\n",IPtostr(prtbuf,broadcast));
      if(has_gw)
	fprintf(pcmcia_file,"\tGATEWAY=\"%s\"\n",IPtostr(prtbuf,gateway));
      else
	fprintf(pcmcia_file,"\tGATEWAY=\"\"\n");
      fprintf(pcmcia_file,"\tDOMAIN=\"%s\"\n",domain);
      if(strcmp(nameserver,"none"))
	{
	  aux=ns=strdup(nameserver);
	  ix=1;
	  while((tmp=strchr(ns,' '))!=NULL)
	    {
	      *tmp='\0';
	      tmp++;
	      fprintf(pcmcia_file,"\tDNS_%d=\"%s\"\n",ix,ns);
	      ns=tmp;
	      ix++;
	    }
	  fprintf(pcmcia_file,"\tDNS_%d=\"%s\"\n",ix,ns);
	  free(aux);
	}
      else
	fprintf(pcmcia_file,"\tDNS_1=\"\"\n");
      fprintf(pcmcia_file,"\t;;\nesac\n");
      fclose(pcmcia_file);
    }
  fclose(p_file);
  
  return 0;
}
 
/*
 * Main configuration routine
 */
int configure_network ()
{
  int rtn;
  struct d_choices* opt;
  FILE *ppp_file;

  host=NULL;
  domain=NULL;
  hostname=NULL;
  netinterface=NULL;
  has_gw=0;
  nameserver=NULL;
  transceiver=strdup("auto");
  
  do
    if((rtn=get_host())==255)
      return rtn;
  while(rtn!=0);
  
  if(yesNoBox(MSG_NETWORK_CONNECTION_L,MSG_NETWORK_CONNECTION)==0)
    {
      broadcast.i[0]=127;broadcast.i[1]=0;broadcast.i[2]=0;broadcast.i[0]=255;
      domain=strdup("none");
      has_gw=0;
      hostname=calloc(strlen(host),sizeof(char));
      strcpy(hostname,host);
      ipaddr.i[0]=127;ipaddr.i[1]=0;ipaddr.i[2]=0;ipaddr.i[0]=1;
      nameserver=strdup("none");
      netmask.i[0]=255;netmask.i[1]=255;netmask.i[2]=255;netmask.i[3]=0;
      network.i[0]=127;network.i[1]=0;network.i[2]=0;network.i[3]=0;
      netinterface=strdup("none");
      if(check_dir(NC_ROOT_DIR)==-1)
        mkdir(NC_ROOT_DIR,0755);
      if (NULL==(ppp_file=fopen(NC_PPP_FILE,"w"))){
	fprintf(stderr,"Error creating ppp file.\n");
	return -1;
      }
      fprintf(ppp_file,"%s\n",host);
      fclose(ppp_file);
    }
  else
   {
     do
       if((rtn=get_net_parameters())==255)
	 return rtn;
     while(rtn!=0);
     
     /*
      * Net interface menu
      */
     opt=calloc(6,sizeof(struct d_choices));
     opt[0].tag=strdup("eth0");
     opt[0].string=strdup(MSG_ETH_NETWORK);
     opt[1].tag=strdup("tr0");
     opt[1].string=strdup(MSG_TR_NETWORK);
     opt[2].tag=strdup("ppp");
     opt[2].string=strdup(MSG_PPP_NETWORK);
     opt[3].tag=strdup("slip");
     opt[3].string=strdup(MSG_SLIP_NETWORK);
     opt[4].tag=strdup("plip");
     opt[4].string=strdup(MSG_PLIP_NETWORK);
     opt[5].tag=strdup("pcmcia");
     opt[5].string=strdup(MSG_PCMCIA_NETWORK);
     rtn=menus(MSG_CHOOSE_NET_INTERFACE_L,MSG_CHOOSE_NET_INTERFACE,opt,6);
     
     netinterface=strdup(opt[rtn].tag);
     
     free(opt[0].tag);
     free(opt[0].string);
     free(opt[1].tag);
     free(opt[1].string);
     free(opt[2].tag);
     free(opt[2].string);
     free(opt[3].tag);
     free(opt[3].string);
     free(opt[4].tag);
     free(opt[4].string);
     free(opt[5].tag);
     free(opt[5].string);
   }

  if(!(strcmp(netinterface,"pcmcia")))
    for(;;)
      {
	input_parameter(MSG_PCMCIA_TRANSCEIVER_L,MSG_PCMCIA_TRANSCEIVER,&transceiver);

	if(!(strcmp(transceiver,"auto")&&
	     strcmp(transceiver,"10baseT")&&
	     strcmp(transceiver,"10base2")&&
	     strcmp(transceiver,"aui")&&
	     strcmp(transceiver,"100baseT")))
	  break;
      } 
  write_network_configuration();
  free(nameserver);
  free(hostname);
  free(host);
  free(domain);
  
  activate_network();

  if(!(strcmp(netinterface,"pcmcia")))
    {
      free(transceiver);
      if(activate_pcmcia())
	problemBox(MSG_PCMCIA_NET_L,MSG_PCMCIA_NET);
    }
  free(netinterface);
  return 0;
}

/*
 * Activates the network interface
 */
int activate_network()
{
  struct stat tmp;
  if(stat("/sbin/ifconfig",&tmp)==0)
    {
      getcwd(prtbuf,PRTBUFSIZE);
      chdir("/etc");
      unlink("resolv.conf");
      symlink(NC_RESOLV_FILE,"resolv.conf");
      chdir(prtbuf);
      system(NC_NETWORK_FILE);
    }
  return 0;
}

/*
 * check_dir
 * Stats some dirname and checks whether it exists and whether it's a 
 * directory.
 */
int check_dir(char* dirname)
{
  struct stat check;
  if(stat(dirname,&check)==-1)
    return -1;
  else
    return S_ISDIR(check.st_mode);
}

#ifdef _TESTING_
/*
 * To test, compile using:
 * cc -o ./the_test -D_TESTING_ -g netconfig.c boxes.c -lnewt -lslang
 */
void main(void)
{
  newtInit();
  newtCls();

  newtDrawRootText(0, 0, "Network Config test program");
  newtOpenWindow(7, 2, 65, 18, "Root Window");
  newtPushHelpLine(NULL);
  configure_network();
  newtPopWindow();
  newtFinished();
  free(netinterface);
}
#endif
