
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif	 // HAVE_CONFIG_H

/////////////////// QProgressCtrl widget class //////////////////////
// 
// for making Win95-style progress bars, as well as options for
// blocked/solid bar style, vertical/horizontal alignment, 
// bar/background colors
// 
// Copyright (C) 1996 Tim D. Gilman
// This version of the class is the first release.  Please
// send comments/suggestions/bug reports to <tdgilman@best.com>
// 
// This class was written using Qt 1.0 (http://www.troll.no)
// 
// You may freely use this source code without permission for 
// non-commercial development.

#include <qwidget.h>
#include <qframe.h>
#include <qpainter.h>
#include <qcolor.h>

#include "qprogres.h"

#define BRD 2					// pixel width between frame and progress bar and between blocks

#define min(a,b) ((a)<(b)?(a):(b))
#define max(a,b) ((a)>(b)?(a):(b))
#define abs(a)   ((a)>0?(a):(-1*a))

/////////////// Construction //////////////

QProgressCtrl::QProgressCtrl(QWidget * parent,
			  ProgressStyle style,
			  const QColor & barColor,
			  const QColor & backColor,
			  ProgressOrient orient,
			  const char *name)
:       QFrame(parent, name)
{
	// default behavior
	m_nLower = 0;
	m_nUpper = 100;
	m_nPos = 0;
	m_nStep = 10;
	m_nSolidLen = 0;			// current width of the bar

	m_nNumBlocks = 0;			// current number of bar blocks (for blockedBar style)

	m_barColor = new QColor(barColor);
	m_orient = orient;
	m_style = style;

	setBackgroundColor(backColor);

	init();
}

void    QProgressCtrl::init()
{
	initMetaObject();
	int     fs;

	switch (style())
	{
	case WindowsStyle:
	case MotifStyle:
		fs = QFrame::Panel | QFrame::Sunken;
		break;
		/* 
		 * I have noticed that any style other than Windows or Motif
		 * breaks the behavior of QProgressCtrl.  I think it is from
		 * lack of support from Qt.  Do not try it yet.
		 */
	default:
		fs = QFrame::Panel | QFrame::Plain;
	}
	setLineWidth(2);
	setMidLineWidth(3);
	setFrameStyle(fs);

	if (m_style == blockedBar)
	{
		m_nBlockWidth = calcBlockLen();
	}

}

///////////////  Attributes ////////////////

/*
 * Parameters
 *  nLower Specifies the lower limit of the range (default is zero)
 *  nUpper Specifies the upper limit of the range (default is 100)
 *
 * Description
 *  Sets the min and max ranges for a progress bar control and
 *  redraws the bar to reflect the new changes.
 */
void    QProgressCtrl::setRange(int nLower, int nUpper)
{
	int     nOldSolidLen = m_nSolidLen;
	int     nOldNumBlocks = m_nNumBlocks;

	m_nLower = nLower;
	m_nUpper = nUpper;

	switch (m_style)
	{
	case solidBar:
		m_nSolidLen = calcSolidBarLen(m_nPos);
		paintSolidBar(nOldSolidLen, m_nSolidLen);
		break;
	case blockedBar:
		m_nNumBlocks = calcNumBlocks(m_nPos);
		paintBlockedBars(nOldNumBlocks, m_nNumBlocks);
		break;
	}
}

/*
 * Returns
 *  The previous position in the progress bar control.
 *
 * Parameters
 *  nPos  New position of the progress bar control.
 *
 * Description
 *  Sets the current position for a progress bar control and redraws
 *  the bar to reflect the new position.
 */
int     QProgressCtrl::setPos(int nPos)
{
	int     nOldPos = m_nPos;
	int     nOldSolidLen = m_nSolidLen;
	int     nOldNumBlocks = m_nNumBlocks;

	m_nPos = nPos;

	switch (m_style)
	{
	case solidBar:
		m_nSolidLen = calcSolidBarLen(m_nPos);
		paintSolidBar(nOldSolidLen, m_nSolidLen);
		break;
	case blockedBar:
		m_nNumBlocks = calcNumBlocks(m_nPos);
		paintBlockedBars(nOldNumBlocks, m_nNumBlocks);
		break;
	}

	return nOldPos;
}

/*
 * Returns
 *  The previous position in the progress bar control.
 *
 * Parameters
 *  nPos  Amount to advance the position.
 *
 * Description
 *  Advances the current position of a progress bar control by a specified
 *  increment and redraws the bar to reflect the new position.
 */
int     QProgressCtrl::offsetPos(int nPos)
{
	int     nOldPos = m_nPos;
	int     nOldSolidLen = m_nSolidLen;
	int     nOldNumBlocks = m_nNumBlocks;

	m_nPos += nPos;

	switch (m_style)
	{
	case solidBar:
		m_nSolidLen = calcSolidBarLen(m_nPos);
		paintSolidBar(nOldSolidLen, m_nSolidLen);
		break;
	case blockedBar:
		m_nNumBlocks = calcNumBlocks(m_nPos);
		paintBlockedBars(nOldNumBlocks, m_nNumBlocks);
		break;
	}

	return nOldPos;
}

/*
 * Returns
 * The previous step increment.
 *
 * Parameters
 *  nStep  New step increment.
 *
 * Description
 *  Specifies the step increment for a progress bar control.  The step
 *  increment is used by stepIt() to increase the progress bar's position.
 *
 *  The default step increment is 10.
 */
int     QProgressCtrl::setStep(int nStep)
{
	int     oldStep = m_nStep;

	m_nStep = nStep;
	return oldStep;
}

///////////////// Operations //////////////////

/*
 * Returns
 *  The previous position of the progress bar control.
 *
 * Description
 *  Advances the current position for a progress bar control by the
 *  step increment (see setStep()) and redraws the bar to reflect
 *  the new position.
 */
int     QProgressCtrl::stepIt()
{
	int     nOldPos = m_nPos;
	int     nOldSolidLen = m_nSolidLen;
	int     nOldNumBlocks = m_nNumBlocks;

	m_nPos += m_nStep;

	switch (m_style)
	{
	case solidBar:
		m_nSolidLen = calcSolidBarLen(m_nPos);
		paintSolidBar(nOldSolidLen, m_nSolidLen);
		break;
	case blockedBar:
		m_nNumBlocks = calcNumBlocks(m_nPos);
		paintBlockedBars(nOldNumBlocks, m_nNumBlocks);
		break;
	}

	return nOldPos;
}

//////////////// Private Stuff //////////////

/*
 * calculates the solid bar length based on the position nPos
 */
int     QProgressCtrl::calcSolidBarLen(int nPos)
{
	int     len = 0;

	// what total length are we dealing with here?
	switch (m_orient)
	{
	case horizBar:
		len = width() - 2 * frameWidth();
		break;
	case vertBar:
		len = height() - 2 * frameWidth();
		break;
	}

	// calculate the pixel length by scaling the position to the 
	// range along length
	if (nPos <= m_nLower)
		return 0;
	else if (nPos >= m_nUpper)
		return (len);
	else
		return ((int) ((float) (m_nPos - m_nLower) / (float) (m_nUpper - m_nLower)
					   * (float) len));

}

/*
 * calculate the number of blocks that should be drawn based on
 * the position nPos.  That is, the number of blocks that first
 * exceeds or equals the exact length.
 */
int     QProgressCtrl::calcNumBlocks(int nPos)
{
	int     totLen = 0,
	        pLen,
	        nBlocks,
	        nSep,
	        sumLen;
	int     edge = frameWidth() + BRD;

	switch (m_orient)
	{
	case horizBar:
		totLen = width() - 2 * edge;
		break;
	case vertBar:
		totLen = height() - 2 * edge;
		break;
	}

	// calculate the pixel length that is a percentage of len based on nPos
	if (nPos <= m_nLower)
		pLen = 0;
	else if (nPos >= m_nUpper)
		pLen = totLen;
	else
		pLen = ((int) ((float) (m_nPos - m_nLower) / (float) (m_nUpper - m_nLower)
					   * (float) totLen));

	// find out how many blocks it takes to exceed or equal this pixel length pLen

	if (pLen == 0)
		return 0;

	nBlocks = 0;
	do
	{
		nBlocks++;
		nSep = nBlocks - 1;
		sumLen = nBlocks * m_nBlockWidth + nSep * BRD;
	}
	while (sumLen < pLen);

	return nBlocks;
}

/*
 * calculates the individual block length based on the total available
 * width of the frame.
 * 
 * To emulate the Win95 common progress control there will be a
 * border to separate the frame from the blocks and also a border
 * between each of the blocks themselves (BRD).
 */
int     QProgressCtrl::calcBlockLen()
{
	const float aspect = 2.f / 3.f;		// approximate aspect ratio for blocks

	int     blockHeight = 1,
	        blockWidth = 1;
	int     edge = frameWidth() + BRD;

	switch (m_orient)
	{
	case horizBar:
		blockHeight = height() - 2 * edge;
		break;
	case vertBar:
		blockHeight = width() - 2 * edge;
		break;
	}

	blockWidth = (int) (aspect * (float) blockHeight);

	// never use zero length
	if (blockWidth < 1)
		blockWidth = 1;

	return ((int) (aspect * (float) blockHeight));
}

/*
 * updates the solid bar when the length has changed
 * this only repaints the portion needed and only erases
 * that part of the background if it moves backward
 */
void    QProgressCtrl::paintSolidBar(int nOldLen, int nLen)
{
	switch (m_orient)
	{
	case horizBar:
		if (nLen > nOldLen)
		{						// do not erase, repaint needed area

			repaint(frameWidth() + nOldLen, frameWidth(),
					nLen - nOldLen, height() - 2 * frameWidth(), FALSE);
		}
		else if (nLen < nOldLen)
		{						// erase and repaint needed area

			repaint(frameWidth() + nLen, frameWidth(),
					nOldLen - nLen, height() - 2 * frameWidth(), TRUE);
		}
		// else do not repaint rectangle, it did not change
		break;
	case vertBar:
		if (nLen > nOldLen)
		{
			repaint(frameWidth(), height() - (frameWidth() + nLen),
					width() - 2 * frameWidth(), nLen - nOldLen, FALSE);
		}
		else if (nLen < nOldLen)
		{
			repaint(frameWidth(), height() - (frameWidth() + nOldLen),
					width() - 2 * frameWidth(), nOldLen - nLen, TRUE);
		}
		break;
	}
}

void    QProgressCtrl::paintBlockedBars(int nOldNumBlocks, int nNumBlocks)
{
	int     x = 0,
	        y = 0,
	        w = 0,
	        h = 0;
	int     edge = frameWidth() + BRD;

	// only repaint if the number of blocks changes and calculate the specific rectangle to erase
	// to avoid flickering.
	if (nOldNumBlocks != nNumBlocks)
	{
		switch (m_orient)
		{
		case horizBar:
			x = edge + min(nOldNumBlocks, nNumBlocks) * (m_nBlockWidth + BRD);
			y = edge;
			w = abs(nNumBlocks - nOldNumBlocks) * m_nBlockWidth + abs(nNumBlocks - nOldNumBlocks - 1) * BRD;
			h = height() - 2 * edge;
			if (w > 0)
			{
				if ((x + w) > (width() - frameWidth() - BRD))
					w = (width() - frameWidth() - BRD) - x;
			}
			break;
		case vertBar:
			x = edge;
			// y is top of the largest block number
			y = height()		// bottom of the vertical bar
					- edge		// subtract edge width
					 - max(nNumBlocks, nOldNumBlocks) * m_nBlockWidth	// subtract sum of blocks width
					 - (max(nNumBlocks, nOldNumBlocks) - 1) * BRD;	// subtract sum of separators width

			w = width() - 2 * edge;
			h = abs(nNumBlocks - nOldNumBlocks) * m_nBlockWidth + abs(nNumBlocks - nOldNumBlocks - 1) * BRD;
			if (h > 0)
			{
				if (y < edge)
					y = edge;
			}
		}

		// if adding blocks, repaint invalid rectangle, but do not erase
		if (nNumBlocks > nOldNumBlocks)
			repaint(x, y, w, h, FALSE);
		// if removing blocks, erase invalid rectangle and repaint
		else if (nNumBlocks < nOldNumBlocks)
			repaint(x, y, w, h, TRUE);
	}

}

void    QProgressCtrl::paintEvent(QPaintEvent *)
{
	QPainter p;
	int     x,
	        y,
	        w,
	        h,
	        edge;

	p.begin(this);

	p.setBrush(*m_barColor);
	p.setPen(*m_barColor);

	switch (m_style)
	{
	case solidBar:
		if (m_nSolidLen > 0)
		{						// no point to drawing a zero-length rectangle

			switch (m_orient)
			{
			case horizBar:
				p.drawRect(frameWidth(), frameWidth(),
						   m_nSolidLen, height() - 2 * frameWidth());
				break;
			case vertBar:
				p.drawRect(frameWidth(), height() - (frameWidth() + m_nSolidLen),
						   width() - 2 * frameWidth(), m_nSolidLen);
				break;
			}
		}
		break;
	case blockedBar:
		{
			int     i;

			edge = frameWidth() + BRD;
			switch (m_orient)
			{
			case horizBar:
				for (i = 0; i < m_nNumBlocks; i++)
				{
					x = edge + i * (m_nBlockWidth + BRD);
					y = edge;
					w = m_nBlockWidth;
					h = height() - 2 * edge;

					if (x < (width() - edge))
					{
						if ((x + w) > (width() - edge))
							w = (width() - edge) - x;
						p.drawRect(x, y, w, h);
					}
				}
				break;
			case vertBar:
				for (i = 0; i < m_nNumBlocks; i++)
				{
					x = edge;
					y = height() - edge - i * (m_nBlockWidth + BRD) - m_nBlockWidth;
					w = width() - 2 * edge;
					h = m_nBlockWidth;
					// do not draw beyond the boundary
					if (y < edge)
					{
						h = y + m_nBlockWidth - edge;
						y = edge;
					}
					p.drawRect(x, y, w, h);
				}
				break;
			}
		}

	}

	// paint the frame since we are overriding QWidget::paintEvent()
	drawFrame(&p);

	// add a black shadow on the top and left side of the frame.
	// this makes it look more like the Win95 style
	if (m_style == blockedBar)
	{

		p.setPen(black);
		p.drawLine(frameWidth(), height() - frameWidth() - 2,
				   frameWidth(), frameWidth());
		p.drawLine(frameWidth(), frameWidth(),
				   width() - frameWidth() - 2, frameWidth());
	}

	p.end();
}

/*
 * On a resize event the bar length needs to be recalculated
 * for a solid bar style and the block length and number of blocks to
 * paint needs to be calculated for a blocked bar style.
 */
void    QProgressCtrl::resizeEvent(QResizeEvent *)
{
	switch (m_style)
	{
	case solidBar:
		m_nSolidLen = calcSolidBarLen(m_nPos);
		break;
	case blockedBar:
		m_nBlockWidth = calcBlockLen();
		m_nNumBlocks = calcNumBlocks(m_nPos);
		break;
	}
}
