// QWeb - An SGML Web Browser
// Copyright (C) 1997  Sean Vyain
// svyain@mail.tds.net
// smvyain@softart.com
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#include <qchkbox.h>
#include <qcombo.h>
#include <qfiledlg.h>
#include <qgrpbox.h>
#include <qlabel.h>
#include <qlined.h>
#include <qlistbox.h>
#include <qmenubar.h>
#include <qmsgbox.h>
#include <qpushbt.h>
#include "Options.h"
#include "Request.h"
#include "ScrolledWindow.h"
#include "StyleEditor.h"
#include "StyleParser.h"
#include "StyleSheet.h"
#include "YesNoCancelDlg.h"

class SgmlAttrItem : public QListBoxText {
public:
    QString name;
    QString value;

    SgmlAttrItem( const char* text, const char* _name, const char* _value )
            : QListBoxText( text ), name( _name ), value( _value ) {}
};

class StyleItem : public QListBoxText {
public:
    Style* style;

    StyleItem( const char* text, Style* _style )
            : QListBoxText( text ), style( _style ) {}
};

class StyleListBox : public QListBox {
public:
    StyleListBox( QWidget* parent ) : QListBox( parent ) {}
    StyleItem* getItem( int index ) { return (StyleItem*)item( index ); }
};

class SgmlAttrListBox : public QListBox {
public:
    SgmlAttrListBox( QWidget* parent ) : QListBox( parent ) {}
    SgmlAttrItem* getItem( int index ) { return (SgmlAttrItem*)item( index ); }
};

StyleEditor::AttrInput::AttrInput( QWidget*        parent,
                                   StyleAttrProto* proto,
                                   Style*          style )
        : QWidget( parent ),
          _proto( proto ),
          _style( style )
{
    _label = new QLabel( _proto->name(), this );
}

void StyleEditor::AttrInput::resizeEvent( QResizeEvent* )
{
    _label->setGeometry( 0, 0, 100, height() );
    widget()->setGeometry( 100, 0, width() - 100, height() );
}

StyleEditor::StringInput::StringInput( QWidget*        parent,
                                       StyleAttrProto* proto,
                                       Style*          style )
        : StyleEditor::AttrInput( parent, proto, style )
{
    _entry = new QLineEdit( this );

    QString tmp;
    if ( ( style ) && ( style->stringValue( proto->token(), tmp ) ) ) {
        _entry->setText( tmp );
    }
}

QWidget* StyleEditor::StringInput::widget()
{
    return _entry;
}

StyleAttr* StyleEditor::StringInput::apply()
{
    QString value = _entry->text();
    if ( value.length() ) {
        return new StringStyleAttr( proto(), value );
    }

    return 0;
}

StyleEditor::NumberInput::NumberInput( QWidget*        parent,
                                       StyleAttrProto* proto,
                                       Style*          style )
        : StyleEditor::AttrInput( parent, proto, style )
{
    _entry = new QLineEdit( this );
    int tmp;
    if ( ( style ) && ( style->numberValue( proto->token(), tmp ) ) ) {
        QString str;
        str.sprintf( "%d", tmp );
        _entry->setText( str );
    }
}

QWidget* StyleEditor::NumberInput::widget()
{
    return _entry;
}

StyleAttr* StyleEditor::NumberInput::apply()
{
    QString value = _entry->text();
    if ( value.length() ) {
        return new NumberStyleAttr( proto(), value.toInt(), FALSE );
    }

    return 0;
}

StyleEditor::EnumeratedInput::EnumeratedInput( QWidget*        parent,
                                               StyleAttrProto* proto,
                                               Style*          style )
        : StyleEditor::AttrInput( parent, proto, style )
{
    _combo = new QComboBox( this );

    if ( proto->inherit() ) {
        _combo->insertItem( "Inherit" );
    }

    int tmp;
    bool found = FALSE;
    if ( style ) {
        found = style->enumValue( proto->token(), tmp );
    }
    
    QListIterator<StyleAttrProto::EnumMap> i( proto->enumMap() );
    for ( i.toFirst(); i.current(); ++i ) {
        _combo->insertItem( i.current()->name );
        if ( ( found ) && ( tmp == i.current()->value ) ) {
            _combo->setCurrentItem( _combo->count() - 1 );
        }
    }
}

QWidget* StyleEditor::EnumeratedInput::widget()
{
    return _combo;
}

StyleAttr* StyleEditor::EnumeratedInput::apply()
{
    int index = _combo->currentItem();
    
    if ( proto()->inherit() ) {
        index--;
    }

    // Check for inherit.
    if ( index < 0 ) {
        return 0;
    }

    QListIterator<StyleAttrProto::EnumMap> i( proto()->enumMap() );
    for ( i.toFirst(); index; ++i, index-- );
    return new EnumeratedStyleAttr( proto(), i.current()->value );
}

StyleEditor::ListInput::ListInput( QWidget*        parent,
                                   StyleAttrProto* proto,
                                   Style*          style )
        : StyleEditor::AttrInput( parent, proto, style )
{
    _combo = new QComboBox( FALSE, this );

    if ( proto->inherit() ) {
        _combo->insertItem( "Inherit" );
    }

    QString tmp;
    bool found = FALSE;
    if ( style ) {
        found = style->listValue( proto->token(), tmp );
    }
    
    QListIterator<StyleAttrProto::EnumMap> i( proto->enumMap() );
    for ( i.toFirst(); i.current(); ++i ) {
        _combo->insertItem( i.current()->name );
        if ( ( found ) && ( tmp == i.current()->name ) ) {
            _combo->setCurrentItem( _combo->count() - 1 );
        }
    }
}

QWidget* StyleEditor::ListInput::widget()
{
    return _combo;
}

StyleAttr* StyleEditor::ListInput::apply()
{
    int index = _combo->currentItem();
    
    if ( proto()->inherit() ) {
        index--;
    }

    // Check for inherit.
    if ( index < 0 ) {
        return 0;
    }

    QString value = _combo->currentText();
    return new ListStyleAttr( proto(), value );
}

StyleEditor::FlagInput::FlagInput( QWidget*        parent,
                                   StyleAttrProto* proto,
                                   Style*          style )
        : StyleEditor::AttrInput( parent, proto, style )
{
    _check = new QCheckBox( "True", this );
    if ( style ) {
        _check->setChecked( style->flagValue( proto->token() ) );
    }
}

QWidget* StyleEditor::FlagInput::widget()
{
    return _check;
}

StyleAttr* StyleEditor::FlagInput::apply()
{
    if ( _check->isChecked() ) {
        return new StyleAttr( proto() );
    }

    return 0;
}

StyleEditor::StyleEditor( QWidget* parent, const char* name )
        : QWidget( parent, name ),
          _styleSheet( 0 ),
          _modified( FALSE ),
          _lastSgmlAttrIndex( -1 )
{
    setCaption( "QWeb: Style Sheet Editor" );
    setIconText( "QWeb: Style Sheet Editor" );

    QFont menuFont( options->menuFontName(),
                    options->menuFontSize(),
                    options->menuFontWeight(),
                    options->menuFontItalic() );
    
    _fileMenu = new QPopupMenu;
    _fileMenu->setFont( menuFont );
    _fileMenu->insertItem( "New"       , this, SLOT( fileNew() ) );
    _fileMenu->insertItem( "Open..."   , this, SLOT( fileOpen() ) );
    _fileMenu->insertItem( "Save"      , this, SLOT( fileSave() ) );
    _fileMenu->insertItem( "Save As...", this, SLOT( fileSaveAs() ) );
    _fileMenu->insertItem( "Close"     , this, SLOT( fileClose() ) );
    _fileMenu->insertItem( "Dismiss"   , this, SLOT( fileDismiss() ) );

    _menu = new QMenuBar( this );
    _menu->setFont( menuFont );
    _menu->insertItem( "File", _fileMenu );

    _fileNameLabel = new QLabel( "Filename:", this );

    _fileNameValue = new QLabel( this );

    _styleListGroup = new QGroupBox( "Style List", this );
    _styleListGroup->setFrameStyle( QFrame::Panel | QFrame::Sunken );

    _styleList = new StyleListBox( _styleListGroup );
    connect( _styleList, SIGNAL( highlighted( int ) ), this, SLOT( styleHighlighted( int ) ) );

    _styleSelectorGroup = new QGroupBox( "Style Selector", this );
    _styleSelectorGroup->setFrameStyle( QFrame::Panel | QFrame::Sunken );

    _tagLabel = new QLabel( "Tag:", _styleSelectorGroup );
    _tagLabel->setAlignment( AlignVCenter | AlignRight );

    _tagEntry = new QLineEdit( _styleSelectorGroup );

    _sgmlAttrsGroup = new QGroupBox( "SGML Attributes", _styleSelectorGroup );
    _sgmlAttrsGroup->setFrameStyle( QFrame::Panel | QFrame::Sunken );

    _sgmlAttrsList = new SgmlAttrListBox( _sgmlAttrsGroup );
    connect( _sgmlAttrsList, SIGNAL( highlighted( int ) ), this, SLOT( sgmlAttrHighlighted( int ) ) );

    _attrNameEntry = new QLineEdit( _sgmlAttrsGroup );

    _equalLabel = new QLabel( "=", _sgmlAttrsGroup );
    _equalLabel->setAlignment( AlignCenter );

    _attrValueEntry = new QLineEdit( _sgmlAttrsGroup );

    _sgmlAttrAddButton = new QPushButton( "Add", _sgmlAttrsGroup );
    connect( _sgmlAttrAddButton, SIGNAL( clicked() ), this, SLOT( sgmlAttrAdd() ) );

    _sgmlAttrUpdateButton = new QPushButton( "Update", _sgmlAttrsGroup );
    connect( _sgmlAttrUpdateButton, SIGNAL( clicked() ), this, SLOT( sgmlAttrUpdate() ) );

    _sgmlAttrDeleteButton = new QPushButton( "Delete", _sgmlAttrsGroup );
    connect( _sgmlAttrDeleteButton, SIGNAL( clicked() ), this, SLOT( sgmlAttrDelete() ) );

    _styleAttrsGroup = new QGroupBox( "Style Attributes", this );
    _styleAttrsGroup->setFrameStyle( QFrame::Panel | QFrame::Sunken );

    _displayLabel = new QLabel( "Display:", _styleAttrsGroup );
    _displayLabel->setAlignment( AlignVCenter | AlignRight );

    _displayCombo = new QComboBox( FALSE, _styleAttrsGroup );
    connect( _displayCombo, SIGNAL( activated( int ) ), this, SLOT( displayActivated( int ) ) );
    QListIterator<StyleSheetProto::DisplayProto> i( styleSheetProto->displayProtos() );
    for ( i.toFirst(); i.current(); ++i ) {
        _displayCombo->insertItem( i.current()->name );
    }

    _styleAttrsWindow = new ScrolledWindow( _styleAttrsGroup );

    _styleAttrs = new QWidget( _styleAttrsWindow->clip() );
    _styleAttrsWindow->setWindow( _styleAttrs );

    _addButton = new QPushButton( "Add", this );
    connect( _addButton, SIGNAL( clicked() ), this, SLOT( styleAdd() ) );

    _updateButton = new QPushButton( "Update", this );
    connect( _updateButton, SIGNAL( clicked() ), this, SLOT( styleUpdate() ) );

    _deleteButton = new QPushButton( "Delete", this );
    connect( _deleteButton, SIGNAL( clicked() ), this, SLOT( styleDelete() ) );

    setStyleSheet( 0 );
    
    resize( 500, 450 );
}

StyleEditor::~StyleEditor()
{
}

void StyleEditor::setFileName( const char* fileName )
{
    _fileName = fileName;

    QString tmp = "file:";
    tmp += fileName;
    if ( _styleSheet ) {
        _styleSheet->soi( tmp );
    }

    if ( _fileName.length() ) {
        _fileNameValue->setText( _fileName );
    } else {
        _fileNameValue->setText( "<Untitled>" );
    }
}

void StyleEditor::setStyleSheet( StyleSheet* styleSheet )
{
    _styleSheet = styleSheet;

    if ( !_styleSheet ) {
        _lastSgmlAttrIndex = -1;
    }
}

void StyleEditor::constrain()
{
    if ( _styleSheet ) {
        _fileMenu->setItemEnabled( 2, TRUE );
        _fileMenu->setItemEnabled( 3, TRUE );
        _fileMenu->setItemEnabled( 4, TRUE );
        _styleListGroup->setEnabled( TRUE );
        _styleList->setEnabled( TRUE );
        _styleSelectorGroup->setEnabled( TRUE );
        _tagLabel->setEnabled( TRUE );
        _tagEntry->setEnabled( TRUE );
        _sgmlAttrsGroup->setEnabled( TRUE );
        _sgmlAttrsList->setEnabled( TRUE );
        _attrNameEntry->setEnabled( TRUE );
        _equalLabel->setEnabled( TRUE );
        _attrValueEntry->setEnabled( TRUE );
        _sgmlAttrAddButton->setEnabled( TRUE );
        _sgmlAttrUpdateButton->setEnabled( TRUE );
        _sgmlAttrDeleteButton->setEnabled( TRUE );
        _styleAttrsGroup->setEnabled( TRUE );
        _displayLabel->setEnabled( TRUE );
        _displayCombo->setEnabled( TRUE );
        _styleAttrsWindow->setEnabled( TRUE );
        _styleAttrs->setEnabled( TRUE );
        _addButton->setEnabled( TRUE );
        _updateButton->setEnabled( TRUE );
        _deleteButton->setEnabled( TRUE );
    } else {
        _fileMenu->setItemEnabled( 2, FALSE );
        _fileMenu->setItemEnabled( 3, FALSE );
        _fileMenu->setItemEnabled( 4, FALSE );
        _styleListGroup->setEnabled( FALSE );
        _styleList->setEnabled( FALSE );
        _styleSelectorGroup->setEnabled( FALSE );
        _tagLabel->setEnabled( FALSE );
        _tagEntry->setEnabled( FALSE );
        _sgmlAttrsGroup->setEnabled( FALSE );
        _sgmlAttrsList->setEnabled( FALSE );
        _attrNameEntry->setEnabled( FALSE );
        _equalLabel->setEnabled( FALSE );
        _attrValueEntry->setEnabled( FALSE );
        _sgmlAttrAddButton->setEnabled( FALSE );
        _sgmlAttrUpdateButton->setEnabled( FALSE );
        _sgmlAttrDeleteButton->setEnabled( FALSE );
        _styleAttrsGroup->setEnabled( FALSE );
        _displayLabel->setEnabled( FALSE );
        _displayCombo->setEnabled( FALSE );
        _styleAttrsWindow->setEnabled( FALSE );
        _styleAttrs->setEnabled( FALSE );
        _addButton->setEnabled( FALSE );
        _updateButton->setEnabled( FALSE );
        _deleteButton->setEnabled( FALSE );
    }
}

void StyleEditor::resizeEvent( QResizeEvent* )
{
    const int pad = 4;
    const int bh  = 24;
    const int bw  = 100;
    const int h1  = ( height() - _menu->height() - 2 * bh - 5 * pad ) * 30 / 100;
    const int h2  = ( height() - _menu->height() - 2 * bh - 5 * pad ) - h1;
    const int w1  = ( width() - 3 * pad ) * 50 / 100;
    const int w2  = ( width() - 3 * pad ) - w1;
    int w, h;

    _fileNameLabel->setGeometry( pad, _menu->height() + pad, 60, bh );

    _fileNameValue->setGeometry( 60 + 2 * pad, _menu->height() + pad, width() - 60 - 3 * pad, bh );
    
    _styleListGroup->setGeometry( pad, _menu->height() + bh + 2 * pad, width() - 2 * pad, h1 );
    w = _styleListGroup->width();
    h = _styleListGroup->height();

    _styleList->setGeometry( pad, 4 * pad, w - 2 * pad, h - 5 * pad );
    
    _styleSelectorGroup->setGeometry( pad, _menu->height() + h1 + bh + 3 * pad, w1, h2 );
    w = _styleSelectorGroup->width();
    h = _styleSelectorGroup->height();

    _tagLabel->setGeometry( pad, 4 * pad, 50, 24 );

    _tagEntry->setGeometry( 50 + 2 * pad, 4 * pad, w - 50 - 3 * pad, 24 );

    _sgmlAttrsGroup->setGeometry( pad, 24 + 5 * pad, w - 2 * pad, h - 24 - 6 * pad );
    w = _sgmlAttrsGroup->width();
    h = _sgmlAttrsGroup->height();

    _sgmlAttrsList->setGeometry( pad, 4 * pad, w - 2 * pad, h - 48 - 7 * pad );

    _attrNameEntry->setGeometry( pad, h - 48 - 2 * pad, ( w - 20 - 4 * pad ) / 2, 24 );

    _equalLabel->setGeometry( _attrNameEntry->x() + _attrNameEntry->width() + pad, _attrNameEntry->y(), 20, 24 );

    _attrValueEntry->setGeometry( w - ( w - 20 - 4 * pad ) / 2 - pad, _equalLabel->y(), ( w - 20 - 4 * pad ) / 2, 24 );

    _sgmlAttrAddButton->setGeometry( pad, h - 24 - pad, ( w - 4 * pad ) / 3, 24 );

    _sgmlAttrUpdateButton->setGeometry( ( w - 4 * pad ) / 3 + 2 * pad, h - 24 - pad, ( w - 4 * pad ) / 3, 24 );

    _sgmlAttrDeleteButton->setGeometry( 2 * ( w - 4 * pad ) / 3 + 3 * pad, h - 24 - pad, ( w - 4 * pad ) / 3, 24 );
    
    _styleAttrsGroup->setGeometry( w1 + 2 * pad, _menu->height() + h1 + bh + 3 * pad, w2, h2 );
    w = _styleAttrsGroup->width();
    h = _styleAttrsGroup->height();

    _displayLabel->setGeometry( pad, 4 * pad, 50, 24 );

    _displayCombo->setGeometry( 50 + 2 * pad, 4 * pad, w - 50 - 3 * pad, 24 );

    _styleAttrsWindow->setGeometry( pad, 24 + 5 * pad, w - 2 * pad, h - 24 - 6 * pad );
    w = _styleAttrsWindow->width() - 16;
    h = _styleAttrsWindow->height() - 16;

    _styleAttrs->setGeometry( 0, 0, w, 24 * _attrInputs.count() );
    _styleAttrsWindow->windowResized();
    
    QListIterator<AttrInput> i( _attrInputs );
    int y;
    for ( i.toFirst(), y = 0; i.current(); ++i, y += 24 ) {
        i.current()->setGeometry( 0, y, w, 24 );
    }

    _addButton->setGeometry( pad, _menu->height() + h1 + h2 + bh + 4 * pad, bw, bh );
    _updateButton->setGeometry( bw + 2 * pad, _menu->height() + h1 + h2 + bh + 4 * pad, bw, bh );
    _deleteButton->setGeometry( 2 * bw + 3 * pad, _menu->height() + h1 + h2 + bh + 4 * pad, bw, bh );
}

void StyleEditor::closeEvent( QCloseEvent* e )
{
    if ( fileClose() ) {
        e->accept();
    } else {
        e->ignore();
    }
}

void StyleEditor::fileNew()
{
    if ( fileClose() ) {
        setStyleSheet( new StyleSheet );
        setFileName( 0 );
        _modified = FALSE;
    }
}

void StyleEditor::fileOpen()
{
    QString fn( QFileDialog::getOpenFileName( 0, "*.sty" ) );

    if ( fn.length() ) {
        if ( fileClose() ) {
            fn.prepend( "file:" );
            Url url( fn );
            styleSheetManager->requestBySOI( this, url.url() );
        }
    }
}

bool StyleEditor::fileSave()
{
    if ( _fileName.length() == 0 ) {
        return fileSaveAs();
    }
    _styleSheet->save();
    _modified = FALSE;
    return TRUE;
}

bool StyleEditor::fileSaveAs()
{
    QString fn = QFileDialog::getSaveFileName( 0, 0, this );

    if ( fn.length() ) {
        if ( _fileName.length() ) {
            setStyleSheet( new StyleSheet( *_styleSheet ) );
        }
        setFileName( fn );
        return fileSave();
    }

    return FALSE;
}

bool StyleEditor::fileClose()
{
    if ( _modified ) {
        YesNoCancelDlg dlg( "QWeb: File Modified", "Style Sheet has been modified.\nSave?", this );

        switch ( dlg.exec() ) {
            case YesNoCancelDlg::Yes:
                if ( !fileSave() ) {
                    return FALSE;
                }
                break;
                
            case YesNoCancelDlg::No:
                break;
                
            case YesNoCancelDlg::Cancel:
                return FALSE;
                break;
        }
    }

    if ( _fileName.length() == 0 ) {
        delete _styleSheet;
    }
    setStyleSheet( 0 );
    setFileName( 0 );
    _modified = FALSE;

    _styleList->clear();
    _sgmlAttrsList->clear();
    _tagEntry->setText( 0 );
    _attrNameEntry->setText( 0 );
    _attrValueEntry->setText( 0 );
    while ( _attrInputs.first() ) {
        delete _attrInputs.first();
        _attrInputs.remove();
    }
    return TRUE;
}

void StyleEditor::fileDismiss()
{
    if ( fileClose() ) {
        hide();
    }
}

void StyleEditor::styleSheet( StyleSheet* styleSheet )
{
    if ( !fileClose() ) {
        return;
    }
    
    setStyleSheet( styleSheet );
    setFileName( Url( _styleSheet->soi() ).path().data() );
    
    QListIterator<Style> i( _styleSheet->_styles );
    QString selector;
    for ( i.toFirst(); i.current(); ++i ) {
        _styleList->inSort( new StyleItem( i.current()->selectorString(), i.current() ) );
    }
}

void StyleEditor::styleHighlighted( int index )
{
    Style* style = _styleList->getItem( index )->style;
    
    _tagEntry->setText( style->_name );

    _sgmlAttrsList->clear();
    QListIterator<Style::Attr> i( style->_sgmlAttrs );
    for ( i.toFirst(); i.current(); ++i ) {
        QString tmp = i.current()->name.copy();
        if ( i.current()->value.length() ) {
            tmp += "=";
            tmp += i.current()->value;
        }
        _sgmlAttrsList->inSort( new SgmlAttrItem( tmp, i.current()->name, i.current()->value ) );
    }

    QListIterator<StyleSheetProto::DisplayProto> j( styleSheetProto->displayProtos() );
    int k;
    for ( j.toFirst(), k = 0; j.current(); ++j, k++ ) {
        if ( style->_display == j.current() ) {
            _displayCombo->setCurrentItem( k );
            displayActivated( k );
            break;
        }
    }

    _lastSgmlAttrIndex = -1;
    _attrNameEntry->setText( 0 );
    _attrValueEntry->setText( 0 );
}

void StyleEditor::sgmlAttrHighlighted( int index )
{
    // Check for a modifed/new SGML attribute.
    if ( _lastSgmlAttrIndex > -1 ) {
        if ( ( _sgmlAttrsList->getItem( _lastSgmlAttrIndex )->name != _attrNameEntry->text() ) ||
             ( _sgmlAttrsList->getItem( _lastSgmlAttrIndex )->value != _attrNameEntry->text() ) ) {
            if ( !QMessageBox::query( "QWeb: Discard Changes?", "You have modified the SGML attribute.\nDiscard changes?" ) ) {
                return;
            }
        }
    }
    
    _attrNameEntry->setText( _sgmlAttrsList->getItem( index )->name );
    _attrValueEntry->setText( _sgmlAttrsList->getItem( index )->value );

    _lastSgmlAttrIndex = index;
}

void StyleEditor::sgmlAttrAdd()
{
    QString name = _attrNameEntry->text();
    if ( name.stripWhiteSpace().length() == 0 ) {
        QMessageBox::message( "QWeb: Error", "You must specify an SGML attribute." );
        return;
    }
    
    // Search for a duplicate.
    uint i;
    for ( i = 0; i < _sgmlAttrsList->count(); i++ ) {
        if ( _sgmlAttrsList->getItem( i )->name == _attrNameEntry->text() ) {
            QString q;
            q.sprintf( "SGML attribute '%s' exists.\nUpdate instead?", _attrNameEntry->text() );
            bool ok = QMessageBox::query( "QWeb: Duplicate SGML Attribute", q );
            if ( ok ) {
                sgmlAttrUpdate();
            }
            return;
        }
    }

    // Add the entry to the list.
    QString tmp = _attrNameEntry->text();
    if ( _attrValueEntry->text() ) {
        tmp += "=";
        tmp += _attrValueEntry->text();
    }
    _sgmlAttrsList->inSort( new SgmlAttrItem( tmp, _attrNameEntry->text(), _attrValueEntry->text() ) );
}

void StyleEditor::sgmlAttrUpdate()
{
    QString name = _attrNameEntry->text();
    if ( name.stripWhiteSpace().length() == 0 ) {
        QMessageBox::message( "QWeb: Error", "You must specify an SGML attribute." );
        return;
    }
    
    // Search for an existing entry.
    uint i;
    for ( i = 0; i < _sgmlAttrsList->count(); i++ ) {
        if ( _sgmlAttrsList->getItem( i )->name == _attrNameEntry->text() ) {
            _sgmlAttrsList->removeItem( i );
            QString tmp = _attrNameEntry->text();
            if ( _attrValueEntry->text() ) {
                tmp += "=";
                tmp += _attrValueEntry->text();
            }
            _sgmlAttrsList->inSort( new SgmlAttrItem( tmp, _attrNameEntry->text(), _attrValueEntry->text() ) );
            return;
        }
    }

    // Add the entry?
    QString q;
    q.sprintf( "SGML attribute '%s' does not exist.\nAdd instead?", _attrNameEntry->text() );
    bool ok = QMessageBox::query( "QWeb: Nonexistant SGML Attribute", q );
    if ( ok ) {
        sgmlAttrAdd();
    }
}

void StyleEditor::sgmlAttrDelete()
{
    // Check for a selected item.
    int index = _sgmlAttrsList->currentItem();
    if ( index == -1 ) {
        QMessageBox::message( "QWeb: Error", "You must select an item to delete." );
        return;
    }

    _sgmlAttrsList->removeItem( index );
}

void StyleEditor::styleAdd()
{
    if ( !_styleSheet ) {
        fileNew();
    }
    
    // Search for a duplicate.
    uint i, j;
    Style* style;
    SgmlAttrItem* attr;
    for ( i = 0; i < _styleList->count(); i++ ) {
        style = _styleList->getItem( i )->style;
        if ( ( style->_name == _tagEntry->text() ) && ( style->_sgmlAttrs.count() == _sgmlAttrsList->count() ) ) {
            for ( j = 0; j < _sgmlAttrsList->count(); j++ ) {
                attr = _sgmlAttrsList->getItem( j );
                QListIterator<Style::Attr> k( style->_sgmlAttrs );
                for ( k.toFirst(); k.current(); ++k ) {
                    if ( ( k.current()->name != attr->name ) || ( k.current()->value != attr->value ) ) {
                        break;
                    }
                }
                if ( k.current() ) {
                    break;
                }
            }
            if ( j >= _sgmlAttrsList->count() ) {
                break;
            }
        }
    }

    if ( i < _styleList->count() ) {
        // Found it.
        QString q;
        q.sprintf( "Style '%s' exists.\nUpdate instead?", _styleList->text( i ) );
        bool ok = QMessageBox::query( "QWeb: Duplicate Style Selector", q );
        if ( ok ) {
            styleUpdate();
        }
        return;
    }

    // Create a new style, and add it to the style sheet.
    style = new Style( _tagEntry->text() );
    QListIterator<StyleSheetProto::DisplayProto> a( styleSheetProto->displayProtos() );
    int index = _displayCombo->currentItem();
    for ( a.toFirst(); index && a.current(); ++a, index-- );
    style->setDisplay( a.current() );
    for ( i = 0; i < _sgmlAttrsList->count(); i++ ) {
        attr = _sgmlAttrsList->getItem( i );
        style->_sgmlAttrs.append( new Style::Attr( attr->name, attr->value ) );
    }
    StyleAttr* styleAttr;
    for ( _attrInputs.first(); _attrInputs.current(); _attrInputs.next() ) {
        if ( ( styleAttr = _attrInputs.current()->apply() ) ) {
            style->_attrs.append( styleAttr );
        }
    }
    _styleSheet->addStyle( style );
    _styleList->inSort( new StyleItem( style->selectorString(), style ) );
    for ( i = 0; i < _styleList->count(); i++ ) {
        if ( _styleList->getItem( i )->style == style ) {
            _styleList->setCurrentItem( i );
            break;
        }
    }

    _modified = TRUE;
}

void StyleEditor::styleUpdate()
{
    // Search for a duplicate.
    uint i, j;
    Style* style;
    SgmlAttrItem* attr;
    for ( i = 0; i < _styleList->count(); i++ ) {
        style = _styleList->getItem( i )->style;
        if ( ( style->_name == _tagEntry->text() ) && ( style->_sgmlAttrs.count() == _sgmlAttrsList->count() ) ) {
            for ( j = 0; j < _sgmlAttrsList->count(); j++ ) {
                attr = _sgmlAttrsList->getItem( j );
                QListIterator<Style::Attr> k( style->_sgmlAttrs );
                for ( k.toFirst(); k.current(); ++k ) {
                    if ( ( k.current()->name != attr->name ) || ( k.current()->value != attr->value ) ) {
                        break;
                    }
                }
                if ( k.current() ) {
                    break;
                }
            }
            if ( j >= _sgmlAttrsList->count() ) {
                break;
            }
        }
    }

    if ( i >= _styleList->count() ) {
        // Not found.
        bool ok = QMessageBox::query( "QWeb: Nonexistant Style Selector", "Style does not exist.\nAdd instead?" );
        if ( ok ) {
            styleAdd();
        }
        return;
    }

    // Delete old style, then add new one.
    styleDelete();
    styleAdd();

    _modified = TRUE;
}

void StyleEditor::styleDelete()
{
    // Check for a selected item.
    int index = _styleList->currentItem();
    if ( index == -1 ) {
        QMessageBox::message( "QWeb: Error", "You must select an item to delete." );
        return;
    }

    // Remove the style from the stylesheet.
    _styleSheet->deleteStyle( _styleList->getItem( index )->style );
    _styleList->removeItem( index );

    _modified = TRUE;
}

void StyleEditor::displayActivated( int index )
{
    while ( _attrInputs.first() ) {
        delete _attrInputs.first();
        _attrInputs.remove();
    }

    resizeEvent( 0 );

    QListIterator<StyleSheetProto::DisplayProto> j( styleSheetProto->displayProtos() );
    for ( j.toFirst(); index && j.current(); ++j, index-- );

    Style* style = 0;
    if ( _styleList->currentItem() > -1 ) {
        style = _styleList->getItem( _styleList->currentItem() )->style;
    }

    QListIterator<StyleAttrProto> i( j.current()->attrProtos );
    for ( i.toFirst(); i.current(); ++i ) {
        switch ( i.current()->type() ) {
            case StyleAttrProto::String:
                _attrInputs.append( new StringInput( _styleAttrs, i.current(), style ) );
                break;

            case StyleAttrProto::Number:
                _attrInputs.append( new NumberInput( _styleAttrs, i.current(), style ) );
                break;

            case StyleAttrProto::Enumerated:
                _attrInputs.append( new EnumeratedInput( _styleAttrs, i.current(), style ) );
                break;

            case StyleAttrProto::List:
                _attrInputs.append( new ListInput( _styleAttrs, i.current(), style ) );
                break;

            case StyleAttrProto::Flag:
                _attrInputs.append( new FlagInput( _styleAttrs, i.current(), style ) );
                break;
        }
        _attrInputs.last()->show();
    }

    resizeEvent( 0 );
}
