// QWeb - An SGML Web Browser
// Copyright (C) 1997  Sean Vyain
// svyain@mail.tds.net
// smvyain@softart.com
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
extern "C" {
#include <ctype.h>
}
#include <qfontmet.h>
#include "BlockRenderer.h"
#include "ButtonRenderer.h"
#include "Canvas.h"
#include "CheckBoxRenderer.h"
#include "EntryRenderer.h"
#include "ImageRenderer.h"
#include "ListBoxRenderer.h"
#include "OptionRenderer.h"
#include "RadioRenderer.h"
#include "Style.h"
#include "StyleSheet.h"
#include "TextAreaRenderer.h"

BlockRenderer::BlockRenderer( Canvas*     canvas,
                              SgmlParser* parser,
                              int         clipWidth,
                              QObject*    parent,
                              const char* name )
        : TextRenderer( canvas, parser, clipWidth, parent, name ),
          _alignment( Style::Left )
{
    style()->enumValue( "alignment", _alignment );

    QListIterator<Style> i( styleStack() );
    for ( i.toLast(); i.current(); --i ) {
        if ( i.current()->enumValue( "prefix", _prefix ) ) {
            break;
        }
    }
}

void BlockRenderer::content( QString text )
{
    if ( _textArea ) {
        _textArea->content( text );
        return;
    } else if ( ( _option ) || ( _listBox ) ) {
        _optionText += text;
        return;
    }
    
    bool          newWords = FALSE;
    int           idx1;
    WordRenderer* word;
    QString       url;

    _leftover += text;

    // Each word is an individual word, with no spaces at all.
    for ( idx1 = 0; ( idx1 < int( _leftover.length() ) ) && ( isspace( _leftover[idx1] ) ); idx1++ );

    if ( idx1 ) {
        // Remove leading white space.
        _leftover.remove( 0, idx1 );
        _spaceBefore = TRUE;
    }
    
    while ( _leftover.length() ) {
        for ( idx1 = 0; ( idx1 < int( _leftover.length() ) ) && ( !isspace( _leftover[idx1] ) ); idx1++ );

        _spaceAfter = ( idx1 < int( _leftover.length() ) );
        
        if ( ( idx1 >= int( _leftover.length() ) ) && ( !_done ) ) {
            // Need more data.
            if ( newWords ) needRedraw();
            return;
        } else {
//            printf( "new word = '%s'\n", _leftover.left( idx1 ).data() );
            word = new WordRenderer( _leftover.left( idx1 ), _spaceBefore, _spaceAfter, FALSE, canvas(), 0, this );
            url = findHyperlink( FALSE );
            if ( url.length() ) {
                canvas()->registerMouseZone( word, url );
            }
            _leftover.remove( 0, idx1 + 1 );

            _renderers.append( word );
            newWords = TRUE;
            _spaceBefore = TRUE;
        }
        
        for ( idx1 = 0; ( idx1 < int( _leftover.length() ) ) && ( isspace( _leftover[idx1] ) ); idx1++ );
        
        if ( idx1 ) {
            // Remove trailing white space.
            _leftover.remove( 0, idx1 );
        }
    }

    if ( newWords ) needRedraw();
}

bool BlockRenderer::redraw()
{
    // Give my kids (inlined images) a chance to redraw.
    bool b = FALSE;
    Renderer* r;
    for ( r = _renderers.first(); r; r = _renderers.next() ) {
        if ( r->redraw() ) {
            b = TRUE;
        }
    }
    
    // Do I really need to do anything?
    if ( !_needRedraw ) return b;
    
    int lineHeight = 0;
    int lineWidth = 0;
    int h = 0;
    bool lastSpace = FALSE;
    StyleRenderer* style = 0;

    // Destroy any existing lines.
    while ( _lines.first() ) {
        delete _lines.first();
        _lines.removeFirst();
    }

    QFontMetrics* fm = 0;

    // Draw the text.
    _maximumWidth = 0;
    _lines.append( new Line );
    _lines.last()->style = (StyleRenderer*)_renderers.first();

    int prefixWidth = 0;
    if ( _prefix == Style::Bullet ) {
        prefixWidth = 10;
    }

    for ( r = _renderers.first(); r; r = _renderers.next() ) {
        if ( !strcmp( r->className(), "StyleRenderer" ) ) {
            style = (StyleRenderer*)r;
            delete fm;
            fm = new QFontMetrics( style->font() );
        }

        if ( !strcmp( r->className(), "WordRenderer" ) ) {
//            printf( "word = '%s'\n", ((WordRenderer*)r)->word().data() );
            r->resize( fm->width( ((WordRenderer*)r)->word() ), fm->height() );
        }

        if ( _minimumWidth < r->width() ) {
            _minimumWidth = r->width();
        }
        _maximumWidth += r->width();

        if ( lastSpace || ( ( !strcmp( r->className(), "WordRenderer" ) ) && ( ((WordRenderer*)r)->spaceBefore() ) ) ) {
            _maximumWidth += style->spaceWidth();
            if ( lineWidth ) {
                lineWidth += style->spaceWidth();
            }
        }
            
        if ( lineWidth && ( lineWidth + r->width() > ( clipWidth() - prefixWidth ) ) ) {
            // Put the inline element on the next line.
            _lines.last()->setRect( 0, h, lineWidth, h + lineHeight - 1 );
            _lines.append( new Line );
            _lines.last()->style = style;
            h += lineHeight;
            lineWidth = 0;
            lineHeight = r->height();
            if ( !strcmp( r->className(), "WordRenderer" ) ) {
                lastSpace = ((WordRenderer*)r)->spaceAfter();
            } else {
                lastSpace = FALSE;
            }

            r->move( lineWidth, h );

            lineWidth = r->width();

            _lines.last()->renderers.append( r );
        } else {
            // Put the inline element on this line.
            if ( lineHeight < r->height() ) {
                lineHeight = r->height();
            }

            r->move( lineWidth, h );

            lineWidth += r->width();
            if ( !strcmp( r->className(), "WordRenderer" ) ) {
                lastSpace = ((WordRenderer*)r)->spaceAfter();
            } else {
                lastSpace = FALSE;
            }

            _lines.last()->renderers.append( r );
        }
    }
    _lines.last()->setRect( 0, h, lineWidth, h + lineHeight - 1 );
    h += lineHeight;

    delete fm;

    // Handle alignment styles for each line.
    Line*     l;
    int       dx;
    for ( l = _lines.first(); l; l = _lines.next() ) {
        if ( _alignment == Style::Right ) {
            dx = clipWidth() - l->width();
        } else if ( _alignment == Style::Center ) {
            dx = ( clipWidth() - l->width() - prefixWidth ) / 2 + prefixWidth;
        } else {
            dx = prefixWidth;
        }
        for ( r = l->renderers.first(); r; r = l->renderers.next() ) {
            r->moveBy( dx, 0 );
        }
        l->moveBy( dx, 0 );
    }

    // Resize the widget.
    _minimumWidth += prefixWidth;
    _maximumWidth += prefixWidth;
    if ( clipWidth() < _minimumWidth ) {
        resize( _minimumWidth, h );
    } else {
        resize( clipWidth(), h );
    }

    _needRedraw = FALSE;

    return TRUE;
}
