// $Id: canvas.cpp,v 1.12 1998/01/13 19:01:51 rich Exp $

#include <kdebug.h>
#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <strings.h>
#include <qpainter.h>
#include <klocale.h>
#include <kapp.h>
#include <math.h>
#include "canvas.h"
#include "tool.h"

Canvas::Canvas(int width, int height,
	       QWidget *parent, const char *name)
  : QWidget(parent, name)
{
  currentTool= 0;
  s= INACTIVE;
  matrix= new QWMatrix;
  zoomed= 0;

  // Create pixmap
  pix= new QPixmap(width, height);
  if (!pix) {
KDEBUG(KDEBUG_INFO, 3000, "Canvas::Canvas(): Cannot create pixmap\n");
    exit(1);
  }

  pix->fill(QColor("white"));

  setZoom(100);

  // Set keyboard focus policy
  setFocusPolicy(QWidget::StrongFocus);
  emit sizeChanged();
}

Canvas::Canvas(const char *filename, QWidget *parent, const char *name)
  : QWidget(parent, name)
{
  currentTool= 0;
  s= INACTIVE;
  zoomed= 0;
  matrix= new QWMatrix;

  // Create pixmap
  pix= new QPixmap(filename);
  if (!pix) {
KDEBUG(KDEBUG_INFO, 3000, "Canvas::Canvas(): Cannot create pixmap\n");
    exit(1);
  }

  resize(pix->width(), pix->height());

  setZoom(100);

  emit sizeChanged();

  // Set keyboard focus policy
  setFocusPolicy(QWidget::StrongFocus);
}

void Canvas::setSelection(const QRect &rect)
{
  selection_= rect;
  haveSelection_= true;
}

const QRect &Canvas::selection()
{
  return selection_;
}

void Canvas::clearSelection()
{
  haveSelection_= false;
  selection_= QRect(0,0,0,0);
}

QPixmap *Canvas::selectionData()
{
  QPixmap *p;

  if (haveSelection_) {
    p= new QPixmap(selection_.width(), selection_.height());
    bitBlt(p, 0, 0, pix,
	   selection_.left(), selection_.top(),
	   selection_.width(), selection_.height(),
	   CopyROP, true);
  }
  else
    p= 0;

  return p;
}

void Canvas::setZoom(int z)
{
  QPainter p;
  int w, h;

  zoomFactor= z;
  matrix->reset();
  matrix->scale((float) z/100, (float) z/100);

  if (zoomed != 0)
    delete zoomed;

  w= (int) (pix->width()* ((float) zoomFactor/100));
  h= (int) (pix->height()*((float) zoomFactor/100));

  zoomed= new QPixmap(w, h);
  zoomed->fill(QColor("white"));

  p.begin(zoomed);
  p.setWorldMatrix(*matrix);
  p.drawPixmap(0,0,*pix);
  p.end();

  if ((w != width()) || (h != height())) {
    resize(w,h);
    emit sizeChanged();
  }
  repaint(0);
}

void Canvas::updateZoomed()
{
  QPainter p;
/*  int w, h; */

  zoomed->fill(QColor("white"));

  p.begin(zoomed);
  p.setWorldMatrix(*matrix);
  p.drawPixmap(0,0,*pix);
  p.end();

  repaint(0);
}

int Canvas::zoom()
{
  return zoomFactor;
}

void Canvas::activate(Tool *t)
{
  assert(!isActive());
  currentTool= t;
  s= ACTIVE;
}

void Canvas::deactivate()
{
  assert(isActive());
  s= INACTIVE;
  currentTool= 0;
}

QPixmap *Canvas::pixmap()
{
  return pix;
}

QPixmap *Canvas::zoomedPixmap()
{
  return zoomed;
}

void Canvas::setPixmap(QPixmap *px)
{
  QPainter p;
  int w, h;

  *pix= *px;

  delete zoomed;

  w= (int) (px->width()* ((float) zoomFactor/100));
  h= (int) (px->height()*((float) zoomFactor/100));

  zoomed= new QPixmap(w, h);

  p.begin(zoomed);
  p.setWorldMatrix(*matrix);
  p.drawPixmap(0,0,*pix);
  p.end();

  if ((w != width()) || (h != height())) {
    resize(w,h);
    emit sizeChanged();
  }
  repaint(0);
}

void Canvas::paintEvent(QPaintEvent * /*e*/ )
{
  bitBlt(this, 0, 0, zoomed);
}

void Canvas::mousePressEvent(QMouseEvent *e)
{
KDEBUG(KDEBUG_INFO, 3000, "Canvas::mousePressEvent() redirector called\n");
  if (isActive())
    currentTool->mousePressEvent(e);
} 

void Canvas::mouseMoveEvent(QMouseEvent *e)
{
  if (isActive())
    currentTool->mouseMoveEvent(e);
}

void Canvas::mouseReleaseEvent(QMouseEvent *e)
{
KDEBUG(KDEBUG_INFO, 3000, "Canvas::mouseReleaseEvent() redirector called\n");
  if (isActive())
    currentTool->mouseReleaseEvent(e);
}

bool Canvas::isActive()
{
  if (s == ACTIVE)
    return true;
  else
    return false;
}

bool Canvas::load(const char *filename, const char *format)
{
  bool s;
  QPixmap p;
  QPixmap q; // Fix UMR when reading transparent pixels (they hold junk)

  if (!format)
    s= p.load(filename);
  else
    s= p.load(filename, format);

  if (s) {
    q.resize(p.size());
    q.fill(QColor("white"));
    bitBlt(&q, 0, 0, &p);
    setPixmap(&q);
  }

  repaint(0);

  return s;
}

bool Canvas::save(const char *filename, const char *format)
{
  bool s;

KDEBUG2(KDEBUG_INFO, 3000, "Canvas::save() file= %s, format= %s\n", filename, format);

  s= pix->save(filename, format);

KDEBUG1(KDEBUG_INFO, 3000, "Canvas::save() returning %d\n", s);

  return s;
}

void Canvas::keyPressEvent(QKeyEvent *e)
{
KDEBUG(KDEBUG_INFO, 3000, "Canvas::keyPressEvent() redirector called\n");
  if (isActive())
    currentTool->keyPressEvent(e);
}


void Canvas::keyReleaseEvent(QKeyEvent *e)
{
KDEBUG(KDEBUG_INFO, 3000, "Canvas::keyReleaseEvent() redirector called\n");
  if (isActive())
    currentTool->keyReleaseEvent(e);
}


#include "canvas.moc"
