/*
 * dfile.c - HP-UX file processing functions for lsof
 */


/*
 * Copyright 1994 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */

#ifndef lint
static char copyright[] =
"@(#) Copyright 1994 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: dfile.c,v 1.5 98/07/17 08:54:44 abe Exp $";
#endif

#if	defined(HPUXKERNBITS) && HPUXKERNBITS>=64
#define _TIME_T
typedef int time_t;
/*
 * CAUTION!!! CAUTION!!! CAUTION!!! CAUTION!!! CAUTION!!! CAUTION!!!
 *
 * Do NOT:
 *
 *	#define INO_T
 *	typedef int ino_t;
 *
 * in this source file for HP-UX >= 10.30.  Doing so will cause the kernel's
 * ino_t type to be erroneously used instead of the application's.
 *
 * CAUTION!!! CAUTION!!! CAUTION!!! CAUTION!!! CAUTION!!! CAUTION!!!
 */
#endif	/* defined(HPUXKERNBITS) && HPUXKERNBITS>=64 */

#include "lsof.h"


/*
 * get_max_fd() - get maximum file descriptor plus one
 */

int
get_max_fd()
{
	struct rlimit r;

	if (getrlimit(RLIMIT_NOFILE, &r))
	    return(-1);
	return(r.rlim_cur);
}


/*
 * print_dev() - print device
 */

char *
print_dev(lf)
	struct lfile *lf;		/* file whose device is to be printed */
{
	static char buf[128];

	(void) sprintf(buf, "%d,%#x", major(lf->dev), minor(lf->dev));
	return(buf);
}


/*
 * process_file() - process file
 */

void
process_file(fp)
	KA_T fp;			/* kernel file structure address */
{
	struct file f;
	int flag;

	if (kread((KA_T)fp, (char *)&f, sizeof(f))) {
	    (void) sprintf(Namech, "can't read file struct from %s",
		print_kptr(fp, (char *)NULL));
	    enter_nm(Namech);
	    return;
	}
	Lf->off = (SZOFFTYPE)f.f_offset;

	if (f.f_count) {

	/*
	 * Construct access code.
	 */
	    if ((flag = (f.f_flag & (FREAD | FWRITE))) == FREAD)
		Lf->access = 'r';
	    else if (flag == FWRITE)
		Lf->access = 'w';
	    else if (flag == (FREAD | FWRITE))
		Lf->access = 'u';
	/*
	 * Process structure by its type.
	 */
	    switch (f.f_type) {

#if	defined(DTYPE_LLA)
	    case DTYPE_LLA:
		process_lla((KA_T)f.f_data);
		return;
#endif	/* DTYPE_LLA */

	    case DTYPE_VNODE:
		process_node((KA_T)f.f_data);
		return;
	    case DTYPE_SOCKET:
		process_socket((KA_T)f.f_data);
		return;
	    default:
		if (!f.f_type || (f.f_ops && (KA_T)f.f_ops != Vnfops)) {
		    (void) sprintf(Namech, "%s file struct, ty=%#x, op=%#x",
			print_kptr(fp, (char *)NULL), f.f_type, f.f_ops);
		    enter_nm(Namech);
		    return;
		}
	    }
	}
	enter_nm("no more information");
}


#if	HPUXV>=1030
/*
 * read_mi() - read stream's module information
 *
 * Note: this function is included in this module, because ino_t is not
 *	 redfined to the kernel's type, but is left at the application's tupe.
 *	 See the CAUTION statement inside the HPUXKERNBITS>=64 #if/#endif
 *	 block at the beginning of this file.
 */

int
read_mi(sh, ip, pcb, pn)
	KA_T sh;			/* stream head address */
	KA_T *ip;			/* returned IP q_ptr */
	KA_T *pcb;			/* returned TCP or UDP q_ptr */
	char **pn;			/* returned protocol name */
{
	struct l_dev *dp;
	struct sth_s hd;
	int i;
	size_t len, ml, nl;
	char mn[32];
	KA_T ka, qa;
	struct module_info mi;
	char *np = Namech;
	struct queue q;
	struct qinit qi;

	if (!sh
	||  kread(sh, (char *)&hd, sizeof(hd))) {
	    (void) sprintf(Namech, "can't read stream head: %s",
		print_kptr(sh, (char *)NULL));
	    return(1);
	}
	if (!Lf->dev_def)
	    dp = (struct l_dev *)NULL;
	else
	    dp = lkupdev(&Lf->dev, 1, 0);
	len = dp ? strlen(dp->name) : 0;
	if (len && len <= (MAXPATHLEN - 1)) {
	    (void) strcpy(np, dp->name);
	    nl = len;
	    np += len;
	} else {
	    nl = 0;
	    *np = '\0';
	}
/*
 * Follow the stream head to each of its queue structures, retrieving the
 * module names for each queue's q_info->qi_minfo->mi_idname chain of
 * structures.  Separate each additional name from the previos one with
 * "->".
 *
 * Ignore failures to read all but queue structure chain entries.
 *
 * Ignore module names that end in "head".
 *
 * Save the q_ptr value for "tcp" and "udp" modules.
 */
	ml = sizeof(mn) - 1;
	mn[ml] = '\0';
	*ip = *pcb = (KA_T)NULL;
	qa = (KA_T)hd.sth_wq;
	for (i = 0; i < 20; i++, qa = (KA_T)q.q_next) {
	    if (!qa || kread(qa, (char *)&q, sizeof(q)))
		break;
	    if (!(ka = (KA_T)q.q_qinfo) || kread(ka, (char *)&qi, sizeof(qi)))
		continue;
	    if (!(ka = (KA_T)qi.qi_minfo) || kread(ka, (char *)&mi, sizeof(mi)))
		continue;
	    if (!(ka = (KA_T)mi.mi_idname) || kread(ka, mn, ml))
		continue;
	    if ((len = strlen(mn)) < 1)
		continue;
	    if (len >= 3 && !strcmp(&mn[len - 3], "sth"))
		continue;
	    if (nl) {
		if ((nl + 2) > (MAXPATHLEN - 1))
		    break;
		(void) strcpy(np, "->");
		nl += 2;
		np += 2;
	    }
	    if ((nl + len) > (MAXPATHLEN - 1))
		break;
	    (void) strcpy(np, mn);
	    nl += len;
	    np += len;
	    if (!q.q_ptr)
		continue;
	    if (!*ip && !strcmp(mn, "ip")) {
		*ip = (KA_T)q.q_ptr;
		continue;
	    }
	    if (!*pcb && !strcmp(mn, "tcpm")) {
		*pcb = (KA_T)q.q_ptr;
		*pn = "TCP";
		continue;
	    }
	    if (!*pcb && !strcmp(mn, "udpm")) {
		*pcb = (KA_T)q.q_ptr;
		*pn = "UDP";
	    }
	}
	return(0);
}
#endif	/* HPUXV>=1030 */
