;*---------------------------------------------------------------------*/
;*    Copyright (c) 1996 by Manuel Serrano. All rights reserved.       */
;*                                                                     */
;*                                     ,--^,                           */
;*                               _ ___/ /|/                            */
;*                           ,;'( )__, ) '                             */
;*                          ;;  //   L__.                              */
;*                          '   \    /  '                              */
;*                               ^   ^                                 */
;*                                                                     */
;*                                                                     */
;*    This program is distributed in the hope that it will be useful.  */
;*    Use and copying of this software and preparation of derivative   */
;*    works based upon this software are permitted, so long as the     */
;*    following conditions are met:                                    */
;*           o credit to the authors is acknowledged following         */
;*             current academic behaviour                              */
;*           o no fees or compensation are charged for use, copies,    */
;*             or access to this software                              */
;*           o this copyright notice is included intact.               */
;*      This software is made available AS IS, and no warranty is made */
;*      about the software or its performance.                         */
;*                                                                     */
;*      Bug descriptions, use reports, comments or suggestions are     */
;*      welcome Send them to                                           */
;*        <Manuel.Serrano@inria.fr>                                    */
;*        Manuel Serrano                                               */
;*        INRIA -- Rocquencourt                                        */
;*        Domaine de Voluceau, BP 105                                  */
;*        78153 Le Chesnay Cedex                                       */
;*        France                                                       */
;*---------------------------------------------------------------------*/


;*=====================================================================*/
;*    serrano/prgm/project/bigloo/comptime1.9/Ast/build.scm            */
;*    -------------------------------------------------------------    */
;*    Author      :  Manuel Serrano                                    */
;*    Creation    :  Thu Dec 29 16:37:53 1994                          */
;*    Last change :  Sat Apr  6 11:23:43 1996 (serrano)                */
;*    -------------------------------------------------------------    */
;*    The construction of the `Ast'                                    */
;*=====================================================================*/

;*---------------------------------------------------------------------*/
;*    The module                                                       */
;*---------------------------------------------------------------------*/
(module ast_build
   (include "Ast/ast.sch"
	    "Tools/trace.sch"
	    "Type/type.sch")
   (import  tools_error
	    tools_shape
	    tools_misc
	    tools_location
	    tools_progn
	    tools_module
	    tools_args
	    engine_param
	    parse_definition
	    type_env
	    ast_env
	    ast_global
	    ast_local
	    ast_sexp)
   (export  (sexp*->ast    <s-exp>*)
	    (toplevel->ast <s-exp>)))

;*---------------------------------------------------------------------*/
;*    sexp*->ast ...                                                   */
;*---------------------------------------------------------------------*/
(define (sexp*->ast sexp*)
   (let ((values '())
	 (defs   '()))
      (let loop ((sexp* sexp*))
	 ;; we always add a form to a module. This added form is used to
	 ;; prevent loop in module initialization stage.
	 (if (null? sexp*)
	     (let* ((l1 (toplevel->ast
			 ;; we add the real initialization function.
			 `(define (,(module-init-name *module-name*))
			     (if require-initialization?
				 (begin
				    ;; the module _is_initialized
				    (set! require-initialization? #f)
				    ;; we set the debugging mode
				    ,(if (and (integer? *compiler-debug*)
					      (>fx *compiler-debug* 0))
					  `(set! *debug* ,*compiler-debug*)
					  #unspecified)
				    ;; we declare global variables as GC roots
				    ,(if (eq? *garbage-collector* 'bumpy)
					 '(gc-global-variables-declarations!)
					 #unspecified)
				    ;; we initialize imported modules
				    (initialize-imported-modules!)
				    ;; we initialize tvector
				    (tvectors-declarations!)
				    ;; we initialize constants
				    (initialize-constants!)
				    ;; we initialize eval values
				    ,(if *eval?*
					 '(initialize-eval-primop!)
					 #unspecified)
				    ;; we introduce the version checking
				    ,(if (not *unsafe-version*)
					 `(check-version! ',*module-name*
							  ,*bigloo-name*
							  ,*bigloo-level*)
					 #t)
				    ;; we compute top-level forms
				    ,@(reverse! values)
				    #unspecified)
				 #unspecified))))
		    (l2 (if (eq? *garbage-collector* 'bumpy)
			    (toplevel->ast
			     `(define (gc-global-variables-declarations!)
				 #unspecified))))
		    (l3 (toplevel->ast
			 ;; we define the function that initialize
			 ;; imported module. This function have to be
			 ;; defined after the initialization function.
			 ;; For now this function is filled with a dummy
			 ;; value. Its real body is computed at the very
			 ;; end of the compilation (in the Cgen pass).
			 `(define (initialize-imported-modules!)
			     #unspecified)))
		    (l4 (toplevel->ast
			 '(define (initialize-constants!)
			     #unspecified))))
		(if (eq? *garbage-collector* 'bumpy)
		    (append l1 l2 l3 l4 (reverse! defs))
		    (append l1 l3 l4 (reverse! defs))))
	     (begin
		(for-each (lambda (exp)
			     (if (not (global? exp))
				 (set! values (cons exp values))
				 (set! defs (cons exp defs))))
			  (toplevel->ast (car sexp*)))
		(loop (cdr sexp*)))))))

;*---------------------------------------------------------------------*/
;*    toplevel->ast ...                                                */
;*    -------------------------------------------------------------    */
;*    This function returns a list of ast-nodes.                       */
;*---------------------------------------------------------------------*/
(define (toplevel->ast exp)
   (trace init "toplevel->ast: " exp #\Newline)
   (match-case exp
      ((begin . ?exps)
       ;; when encountering toplevel begin we just loop...
       (let loop ((exps exps))
	  (if (null? exps)
	      '()
	      (append (toplevel->ast (car exps)) (loop (cdr exps))))))
      ((define (and (? symbol?) ?var) (lambda ?args . ?-))
       ;; it is a function definition
       (fix-function-definition! var `(,var ,@args) exp))
      ((define (and (? symbol?) ?var) (and (? symbol?) ?alias))
       (trace init "   un alias... ")
       (let ((alias (find-global alias)))
	  (trace init (global? alias) " "
		 (and (global? alias)
		      (function? (global-value alias)))
		 #\Newline)
	  (if (and (global? alias) (function? (global-value alias)))
	      (let* ((arity (function-arity (global-value alias)))
		     (args  (make-n-proto arity)))
		 (cond
		    ((>=fx arity 0)  
		     (toplevel->ast `(define ,(cons var args)
					(,(global-name alias) ,@args))))
		    ((=fx arity -1)
		     (toplevel->ast `(define ,(cons var args)
					(apply ,(global-name alias) ,args))))
		    (else
		     (toplevel->ast `(define ,(cons var args)
					(apply
					 ,(global-name alias)
					 (cons*
					  ,@(args*->args-list args))))))))
	      (fix-variable-definition! var exp))))
      ((define (and (? symbol?) ?var) . ?-)
       (fix-variable-definition! var exp))
      ((define (and ?proto (?var . ?args)) . ?-)
       (fix-function-definition! var proto exp))
      ((define-inline (and ?proto (?var . ?args)) . ?-)
       (if (global? var)
	   (fix-imported-inline-definition! var proto exp)
	   (fix-inline-definition! var proto exp)))
      (else
       (list exp))))
 
;*---------------------------------------------------------------------*/
;*    fix-function-definition! ...                                     */
;*    -------------------------------------------------------------    */
;*    We look if `var' already was a function or if we can change      */
;*    it to a function.                                                */
;*---------------------------------------------------------------------*/
(define (fix-function-definition! var proto exp)
   (let ((global (find-global (car (parse-formal-ident var)))))
      (cond
	 ((not (global? global))
	  (internal-error "fix-function-definition!"
			  "Unbound variable"
			  var))
	 ((eq? (global-class global) 'procedure)
	  (fix-variable-type!  global)
	  (fix-function-args!  var
			       (global-value global)
			       (function-args (global-value global))
			       (cddr (parse-definition proto)))
	  (fix-function-types! (global-value global))
	  (list (function-definition->ast global exp)))
	 ((and (eq? (global-import global) 'static)
	       (eq? (global-access global) 'read))
	  ;; yes, we are able to change for a function
	  (let* ((proto (parse-definition proto)))
	     (set-global-procedure-slot! global
					 'static
					 (car proto)
					 (cdr proto))
	     (toplevel->ast exp)))
	 (else
	  ;; no, we can't change the variable definition so we just
	  ;; produce a `set!' expression instead of a `define' one.
	  (match-case exp
	     ((define (and ?def (?- . ?args)) . ?body)
	      (let ((fun `(lambda ,args ,@body)))
		 (set-car! exp 'set!)
		 (replace! def fun)
		 (set-car! (cdr exp) var)
		 (set-car! (cddr exp) def)
		 (toplevel->ast exp)))
	     ((define ?- . ?-)
	      (set-car! exp 'set!)
	      (toplevel->ast exp)))))))

;*---------------------------------------------------------------------*/
;*    fix-variable-definition! ...                                     */
;*---------------------------------------------------------------------*/
(define (fix-variable-definition! var exp)
   (let* ((ident  (car (parse-formal-ident var)))
	  (global (find-global ident)))
      (if (not (global? global))
	  (internal-error "fix-variable-definition!"
			  "Unbound variable"
			  var)
	  (begin
	     (fix-variable-type! global)
	     (set-car! exp 'set!)
	     (set-car! (cdr exp) ident)
	     (list exp)))))

;*---------------------------------------------------------------------*/
;*    fix-imported-inline-definition! ...                              */
;*---------------------------------------------------------------------*/
(define (fix-imported-inline-definition! var proto exp)
   (let ((global var))
      (cond
	 ((not (global? global))
	  (internal-error "fix-inline-definition!"
			  "Unbound variable"
			  var))
	 (else
	  (fix-variable-type!  global)
	  (fix-function-args!  var
			       (global-value global)
			       (function-args (global-value global))
			       (cddr (parse-definition
				      (cons (global-name var)
					    (cdr proto)))))
	  (fix-function-types! (global-value global))
	  (list (inline-definition->ast global exp))))))
   
;*---------------------------------------------------------------------*/
;*    fix-inline-definition! ...                                       */
;*---------------------------------------------------------------------*/
(define (fix-inline-definition! var proto exp)
   (let ((global (find-global (car (parse-formal-ident var)))))
      (cond
	 ((not (global? global))
	  (internal-error "fix-inline-definition!"
			  "Unbound variable"
			  var))
	 (else
	  (fix-variable-type!  global)
	  (fix-function-args!  var
			       (global-value global)
			       (function-args (global-value global))
			       (cddr (parse-definition proto)))
	  (fix-function-types! (global-value global))
	  (list (inline-definition->ast global exp))))))
   
;*---------------------------------------------------------------------*/
;*    function-definition->ast ...                                     */
;*---------------------------------------------------------------------*/
(define (function-definition->ast global exp)
   (trace init "function-definition->ast: " exp #\Newline)
   (match-case exp
      ((or (define (?var . ?args) . ?body)
	   (define ?var (lambda ?args . ?body)))
       (enter-function (global-name global))
       (let ((ast-body (sexp->ast (normalize-progn body)
				  (function-args (global-value global))
				  global
				  (find-location exp)
				  '())))
	  (function-body-set! (global-value global) ast-body)
	  (leave-function)
	  global))
      (else
       (internal-error "function-definition->ast"
		       "Illegal expression"
		       exp))))

;*---------------------------------------------------------------------*/
;*    inline-definition->ast ...                                       */
;*---------------------------------------------------------------------*/
(define (inline-definition->ast global exp)
   (trace init "inline-definition->ast: " exp #\Newline)
   (match-case exp
      ((define-inline (?var . ?args) . ?body)
       (enter-function (global-name global))
       (let ((ast-body (sexp->ast (normalize-progn body)
				  (function-args (global-value global))
				  global
				  (find-location exp)
				  '())))
	  (function-body-set! (global-value global) ast-body)
	  (leave-function)
	  global))
      (else
       (internal-error "inline-definition->ast"
		       "Illegal expression"
		       exp))))
 
;*---------------------------------------------------------------------*/
;*    fix-variable-type! ...                                           */
;*---------------------------------------------------------------------*/
(define (fix-variable-type! variable)
   (cond
      ((null? (variable-type variable))
       'done)
      ((type? (variable-type variable))
       'done)
      (else
       (variable-type-set! variable (find-type (variable-type variable))))))

;*---------------------------------------------------------------------*/
;*    fix-function-types! ...                                          */
;*---------------------------------------------------------------------*/
(define (fix-function-types! function)
   (cond
      ((null? (function-type-res function))
       'ok)
      ((type? (function-type-res function))
       'ok)
      (else
       (function-type-res-set! function
			       (find-type (function-type-res function)))))
   (let ((args (function-args function)))
      (for-each fix-variable-type! args))
   'done)

;*---------------------------------------------------------------------*/
;*    fix-function-args! ...                                           */
;*---------------------------------------------------------------------*/
(define (fix-function-args! var function old-args args)
   (function-args-set!
    function
    (let loop ((args args)
	       (olds  old-args)
	       (res  '()))
       (cond
	  ((null? args)
	   (if (null? olds)
	       (reverse! res)
	       (error (shape var)
		      "Prototype and definition don't match"
		      (shape olds))))
	  ((null? olds)
	   (error (shape var)
		  "Prototype and definition don't match"
		  (shape args)))
	  ((not (pair? (car args)))
	   ;; it is the last formal of an n-ary
	   ;; function
	   (reverse! (cons (make-local-variable (car args) (find-type 'obj))
			   res)))
	  (else
	   (let* ((arg (car args))
		  (old (car olds))
		  (var (make-local-variable (car arg)
					    (find-type (cdr arg)))))
	      (if (null? (cdr arg))
		  (local-type-set! var (local-type old)))
	      (loop (cdr args)
		    (cdr olds)
		    (cons var res))))))))


