;**************************************************************************
;*
;* Boot-ROM-Code to load an operating system across a TCP/IP network.
;*
;* Module:  rom.asm
;* Purpose: Decompress the rom image
;* Entries: Called by BIOS at offset 0, malloc, getbyte, putbyte
;*
;**************************************************************************
;*
;* Copyright (C) 1995,1996 Gero Kuhlmann <gero@gkminix.han.de>
;*
;*  This program is free software; you can redistribute it and/or modify
;*  it under the terms of the GNU General Public License as published by
;*  the Free Software Foundation; either version 2 of the License, or
;*  any later version.
;*
;*  This program is distributed in the hope that it will be useful,
;*  but WITHOUT ANY WARRANTY; without even the implied warranty of
;*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;*  GNU General Public License for more details.
;*
;*  You should have received a copy of the GNU General Public License
;*  along with this program; if not, write to the Free Software
;*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
;*


;
;**************************************************************************
;
; Load the boot rom segment layout and assembler macros.
;
INITSEG		equ	1		; create end-of-segment labels

include ..\headers\asm\macros.inc
include ..\headers\asm\layout.inc
include ..\headers\asm\memory.inc
include ..\headers\asm\version.inc
include .\loadpriv.inc


;
;**************************************************************************
;
; Equates for this module:
;
; maximum size of floppy image
FLOSIZE		equ	((FLOPEND - FLOPMEM) shl 4) - 1

; maximum size of decompressed rom image
IMGSIZE		equ	((FLOPMEM - LOWMEM) shl 4) - 1

; first free memory segment
FREEMEM		equ	FLOPEND

; an uncompressed floppy image has to be able to go into the
; destination memory area
if FLOSIZE gt IMGSIZE
	.err	Invalid memory specification - check memory.inc
endif


;
;**************************************************************************
;
; Define BIOS entry point for rebooting the system.
;
% bootseg	segment	seguse at 0FFFFh

	org	0
doboot	label	far

bootseg		ends


;
;**************************************************************************
;
; Define the point where the executable rom image (decompressed) starts.
;
% imageseg	segment	seguse at LOWMEM

		org	0
startimage	label	far

imageseg	ends


;
;**************************************************************************
;
; Define the data segment
;
data_start

outptr		dw	0	; ptr into decompressed memory (rel. to LOWMEM)
inptr		dw	0	; ptr into compressed memory (rel. to CS)
freeseg		dw	FREEMEM	; segment of next free memory
endseg		dw	0	; last usable segment
insize		dw	0	; size of compressed image

		even		; align to word boundary

data_end


;
;**************************************************************************
;
; The physical rom starts here. At the beginning there is some
; miscellaneous data.
;
text_start

	public	start
	public	malloc
	public	getbyte
	public	putbyte

	org	0			; the rom always starts at offset 0

	dw	0AA55h			; rom signature
romlen	db	0			; rom size (filled in by checksum prg)

start:	jmp	short romstart		; the BIOS calls us here


; The following variables contain the size of the bootrom kernel image and
; the program version number.

	db	(_text:DATAOFS - $) dup (0)	; Fill up
romsiz	dw	0				; rom size
	db	MAJOR_VER			; major version number
	db	MINOR_VER			; minor version number
	dw	0				; serial number


;
;**************************************************************************
;
; Now start with the actual boot rom code. The startup code just installs
; a new bootrom interrupt vector, and then returns to the BIOS. This is
; necessary so that the BIOS gets a chance to also initialize other installed
; roms. The bootrom vector is used for the ROM BASIC in original IBM PC's
; and gets called by the BIOS whenever there is no bootable disk in any
; of the drives.
;
romstart	proc	far

	cli				; do not allow interrupts when changing
	push	ds			; the interrupt vector table
	xor	ax,ax			; copy old interrupt vector
	mov	ds,ax
ifdef IS386
	mov	eax,ds:[BOOTROM * 4]
	mov	ds:[OLDBOOTROM * 4],eax
else
	mov	ax,ds:[BOOTROM*4 + 0]
	mov	ds:[OLDBOOTROM*4 + 0],ax
	mov	ax,ds:[BOOTROM*4 + 2]
	mov	ds:[OLDBOOTROM*4 + 2],ax
endif
	mov	word ptr ds:[BOOTROM*4 + 0],offset cgroup:int18
	mov	word ptr ds:[BOOTROM*4 + 2],cs
	pop	ds
	sti				; that's it
	ret

romstart	endp


;
;**************************************************************************
;
; The bootrom interrupt routine decompresses the rom image and starts it.
;
int18		proc	far

	assume	ds:dgroup
	assume	es:dgroup

	extrn	inflate:near

	cli
	cld
	mov	bx,NEWDATA		; set temporary stack pointer
	mov	ss,bx
if (LOWMEM - NEWDATA) lt 1000h
	mov	sp,((LOWMEM - NEWDATA) shl 4) - 1
else
	mov	sp,0FFFFh
endif

	mov	ax,cs
	mov	ds,ax			; move data segment into lower ram
	mov	es,bx
	mov	si,offset cgroup:_end_of_code + 1
	and	si,0FFFEh
	mov	di,si			; start the data segment at the right
	and	di,000Fh		; offset
	mov	cx,offset dgroup:_end_of_bss
	rep	movsb
	mov	ds,bx			; set the new data segment
	sti

; Compute last usable segment and check if we have enough memory

	int	12h			; get number of base kB using BIOS
	shift	shl,ax,6		; convert it into number of segments
	mov	endseg,ax		; and save it for malloc
	cmp	ax,HIGHMEM		; check if it's enough for us
	jae	short memok
	mov	si,offset cgroup:memmsg	; not enough memory
	jmp	short hderr		; avoid far jump (takes too much space)
memok:	mov	si,offset cgroup:_end_of_bss	; note: rel to cgroup!!
	mov	cx,cs:romsiz

; If the image is not compressed just copy it to it's new place

norom:	cmp	word ptr cs:[si],COMPR_MAGIC
	je	short iscomp
	mov	ax,cs
	mov	ds,ax
	mov	ax,LOWMEM
	mov	es,ax			; copy the image to it's new place
	xor	di,di
	rep	movsb
	jmp	short doimg1		; start the image

; The image is compressed, so uncompress it.

iscomp:	add	si,2
	mov	inptr,si		; set input ptr to first data byte
	sub	cx,2
	mov	insize,cx		; set maximum input size
	mov	si,offset cgroup:inimsg
	call	prnstr
	call	inflate			; decompress the block
	or	al,al			; check for error
	jz	short doimg
	mov	si,offset cgroup:infmsg
hderr:	jmp	error

; Finally print the end message, restore the bootrom vector and start
; the new image

doimg:	mov	si,offset cgroup:endmsg
	call	prnstr
doimg1:	cli
	push	ds
	xor	ax,ax			; copy old interrupt vector
	mov	ds,ax
ifdef IS386
	mov	eax,ds:[OLDBOOTROM * 4]
	mov	ds:[BOOTROM * 4],eax
else
	mov	ax,ds:[OLDBOOTROM*4 + 0]
	mov	ds:[BOOTROM*4 + 0],ax
	mov	ax,ds:[OLDBOOTROM*4 + 2]
	mov	ds:[BOOTROM*4 + 2],ax
endif
	pop	ds
	sti
	jmp	startimage		; call decompressed image


; Messages

inimsg	db	'Uncompressing... ',0
endmsg	db	'done',0Dh,0Ah,0

infmsg	db	'format',0
memmsg	db	'memory',0

int18		endp


;
;**************************************************************************
;
; Allocate memory for the decompression tables. This is a very simple
; implementation in that the allocated memory cannot be freed anymore.
; Input:  AX  -  number of paragraphs required
; Output: ES  -  segment of memory block
; Changed registers: AX, ES
;
malloc		proc	near

	push	bx
	mov	bx,freeseg
	add	ax,bx			; determine segment of next free
	cmp	ax,endseg		; memory block
	jb	short mall1		; enough memory?
	mov	si,offset cgroup:memmsg	; no, print error message
	jmp	short error
mall1:	mov	freeseg,ax		; save next free segment
	mov	es,bx			; return current free segment
	pop	bx
	ret

malloc		endp


;
;**************************************************************************
;
; Return the next byte from the compressed image.
; Input:  none
; Output: AL  -  next byte
;         AH  -  not 0, if end of file
; Changed registers: AX, SI
;
getbyte		proc	near

	mov	ax,insize
	or	ax,ax			; check if at end of file
	jz	short getb9
	dec	insize
	mov	si,inptr
ifdef IS386
	movzx	ax,byte ptr cs:[si]
else
	mov	al,cs:[si]		; no, get next byte
	xor	ah,ah
endif
	inc	inptr
	ret

getb9:	mov	ah,1			; indicate end of file
	ret

getbyte		endp


;
;**************************************************************************
;
; Write a byte to the decompressed image.
; Input:  AL  -  output byte
; Output: none
; Registers changed: AX
;
putbyte		proc	near

	push	di
	push	es
	mov	di,LOWMEM
	mov	es,di
	mov	di,outptr		; simply write the byte into the
	stosb				; output buffer area
	inc	outptr
	pop	es
	pop	di
	ret

putbyte		endp


;
;**************************************************************************
;
; Print an error message onto the console screen
; Input:  CS:SI  -  Pointer to message
; Output: none
; Registers changed: routine does not return
;
error		proc	near

	push	si
	mov	si,offset cgroup:er1msg
	call	prnstr
	pop	si
	call	prnstr
	mov	si,offset cgroup:er2msg
	call	prnstr
	xor	ax,ax			; wait until user presses a key
	int	16h
	mov	ax,0040h
	mov	ds,ax
	mov	ds:[0072h],1234h	; indicates warm boot
	jmp	doboot			; reboot the system


; Messages:

er1msg	db	0Dh,0Ah,'Error: ',0
er2msg	db	0Dh,0Ah,'Press a key to reboot',0

error		endp


;
;**************************************************************************
;
; Print a string onto the console screen
; Input:  CS:SI  -  Pointer to string
; Output: none
; Registers changed: AX, BX, SI
;
prnstr		proc	near

prnst1:	mov	al,cs:[si]		; get next character
	or	al,al
	jz	short prnst2
	mov	ah,0Eh			; BIOS command for character output
	mov	bx,0007h		; screen page and colors
	int	10h			; send character to screen
	inc	si
	jmp	short prnst1
prnst2:	ret

prnstr		endp


;
;**************************************************************************
;
text_end			; end text segment

	end
