;**************************************************************************
;*
;* Boot-ROM-Code to load an operating system across a TCP/IP network.
;*
;* Module:  floppy.asm
;* Purpose: Floppy boot sector to load the rest of the boot rom code
;* Entries: called by BIOS at offset 0
;*
;**************************************************************************
;*
;* Copyright (C) 1995,1996 Gero Kuhlmann <gero@gkminix.han.de>
;*
;*  This program is free software; you can redistribute it and/or modify
;*  it under the terms of the GNU General Public License as published by
;*  the Free Software Foundation; either version 2 of the License, or
;*  any later version.
;*
;*  This program is distributed in the hope that it will be useful,
;*  but WITHOUT ANY WARRANTY; without even the implied warranty of
;*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;*  GNU General Public License for more details.
;*
;*  You should have received a copy of the GNU General Public License
;*  along with this program; if not, write to the Free Software
;*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
;*


;
;**************************************************************************
;
; Load the boot rom segment layout and assembler macros:
;
include ..\headers\asm\macros.inc
include ..\headers\asm\layout.inc
include ..\headers\asm\memory.inc
include .\loadpriv.inc


;
;**************************************************************************
;
; Equates local for this module:
;
STACKTOP	equ	0FFFEh		; top of stack in code segment
HIGHADR		equ	MEMEND		; end of physical base memory
LOADADR		equ	FLOPMEM		; segment where to load the boot code
LOADSIZE	equ	(FLOPEND-FLOPMEM)	; required load memory size
DPTVECTOR	equ	0078h		; address of pointer to DPT
SECTSIZE	equ	512		; size of one floppy sector


;
;**************************************************************************
;
; Define BIOS entry point for rebooting the system.
;
% bootseg	segment	seguse at 0FFFFh

	org	0
doboot	label	far

bootseg		ends


;
;**************************************************************************
;
; Structure defining the layout of a disk parameter table
;
dptab		struc

dp_spec1	db	0
dp_spec2	db	0
dp_moffc	db	0
dp_byt_sec	db	0	; bytes per sector
dp_sec_track	db	0	; sectors per track
dp_gap		db	0
dp_dtl		db	0
dp_gap3		db	0
dp_fil_byte	db	0
dp_hd_time	db	0
dp_most		db	0
dp_maxtrack	db	0
dp_other	db	0

dptab		ends


;
;**************************************************************************
;
; Here the loader program starts. Some ideas are taken from the Linux
; boot sector. Note that this program has to be completely seperate
; from all other segments of the boot rom code. It can contain data
; since it's getting loaded into the ram.
; The loader works by first patching the floppy disk parameter table,
; and then trying to read the last sector of the floppy in order to
; find out about the disk size. Then it will load 64 kB from the
; diskette into the OS loading area (see memory.h for explanation).

load_start				; begin loader text segment

	extrn	start:far		; this will include the rom code

	org	0

	cli
	call	near ptr dobot1
dobot3:

; The following variable gets written by the pass2 binary patch program
; and contains the length of the bootrom kernel image. It has to be passed
; to the actual rom loader.

	db	(load_text:DATAOFS - $) dup (0)	; Fill up
romsiz	dw	0

dobot1:	pop	ax			; determine the offset
	sub	ax,offset load_text:dobot3

	mov	cl,4
	mov	bx,cs
	mov	dx,bx			; convert the base address from
	and	dx,0F000h		; seg:ofs format into sort of a
	shl	bx,cl			; linear address. this conversion
	add	bx,ax			; assumes that no overflow can occur.
					; compute a code segment address
	shr	bx,cl			; which lets this code start at
	add	bx,dx			; offset 0. this is necessary so
	push	bx			; all offsets computed by masm work.
	mov	ax,offset load_text:dobot2
	push	ax
	retf				; this will jump to dobot2 with new
					; code segment register
dobot2:	mov	ax,cs
	mov	ds,ax			; load segment registers
	mov	es,ax
	mov	ss,ax
	mov	sp,STACKTOP		; finally set the stack
	sti

; now it's time to print some really important messages :-))

	mov	si,offset load_text:sigmsg
	call	prnstr

; Next determine the number of sectors per track on the floppy. To do
; this the disk parameter table has first to be changed, so that the
; BIOS is able to access at least 18 sectors in a track.
; Note that we don't check if the system really has enough base memory
; for loading everything. Is there any system on the market which still
; doesn't have 640kB in the lower memory area?

	cld
	push	ds
	mov	dx,ds
	xor	ax,ax
	mov	es,ax			; first save the old disk parameter
	lds	si,es:[DPTVECTOR]	; table
	mov	es,dx
	mov	word ptr es:[oldvect+0],si	; save old vector
	mov	word ptr es:[oldvect+2],ds
	mov	di,offset load_text:newdpt
	mov	cx,size dptab
	rep	movsb
	pop	ds
	mov	al,18
	mov	newdpt.dp_sec_track,al	; set new number of sectors per track

	xor	ax,ax			; set pointer to new DPT
	mov	es,ax
	mov	word ptr es:[DPTVECTOR + 0],offset load_text:newdpt
	mov	word ptr es:[DPTVECTOR + 2],ds

; Next try to find the number of sectors per track. To do this, the
; last sector for each disk format is read. This idea has been taken
; from the Linux boot sector.

	xor	ah,ah			; reset drive 0
	xor	dl,dl
	int 	13h

	mov	ax,LOADADR
	mov	es,ax			; set load address 9800:0
	xor	bx,bx

	xor	dx, dx			; drive 0, head 0
	mov	cx,18			; sector 18, track 0
	mov	ax,0201h		; read one sector
	int	13h
	jnc	short gotit
	mov	cx,15			; sector 15
	mov	ax,0201h		; read one sector
	int	13h
	jnc	short gotit
	mov	cx,9
gotit:	mov	sects,cx		; save number of sector per track

; Now load the next 64kB starting at the second sector of track 0.

	xor	bx,bx			; starting offset of load memory
if SECTSIZE eq 512
	mov	dx,(LOADSIZE shr 5)	; number of sectors to load
elseif SECTSIZE eq 256
	mov	dx,(LOADSIZE shr 4)	; number of sectors to load
else
	.ERR
	%OUT Invalid sector size
endif
	mov	ax,1			; starting sector
rdloop:	or	dx,dx
	jz	short loaded
	push	ax
	push	dx
	call	cvtsec			; compute sector information
	mov	ax,0201h		; load one sector
	int	13h
	pop	dx
	pop	ax
	jc	short lderr
	dec	dx			; decrement sector count
	inc	ax			; increment sector number
	add	bx,SECTSIZE		; increment load pointer
	jmp	short rdloop

lderr:	mov	si,offset load_text:dskmsg
	call	prnstr
	jmp	short reboot

; The boot code is in the memory now. It has to be called exactly
; like the BIOS would call it in a real rom. First just call the
; entry routine, which should setup int 18h. Then call the boot-
; strap vector. It should never return. But before we can call the
; boot code we have to copy the kernel image size.

loaded:	cld
	push	ds
	les	si,oldvect
	xor	ax,ax			; restore old vector
	mov	ds,ax
	mov	word ptr ds:[DPTVECTOR + 0],si
	mov	word ptr ds:[DPTVECTOR + 2],es
	pop	ds
	xor	ax,ax
	xor	dx,dx			; reset disk system
	int	13h

	mov	bx,LOADADR
	mov	es,bx
	mov	ax,word ptr cs:romsiz	; set rom size
	mov	es:[DATAOFS],ax

	push	cs
ifdef IS186
	push	offset load_text:boot
	push	bx
	push	offset cgroup:start
else
	mov	ax,offset load_text:boot
	push	ax
	push	bx
	mov	ax,offset cgroup:start
	push	ax
endif
	retf				; call boot rom startup code

boot:	int	BOOTROM			; call the bootrom loader

reboot:	mov	si,offset load_text:botmsg
	call	prnstr
	xor	ah,ah			; wait for a key press
	int	16h
	mov	ax,0040h
	mov	ds,ax
	mov	ds:[0072h],1234h	; indicates warm boot
	jmp	doboot			; reboot the system


;
;**************************************************************************
;
; Routine to convert an absolute sector into a sector/track/head information.
; Input:  AX  -  absolute sector number
; Output: CL  -  sector number
;         CH  -  track number
;         DH  -  head number
;         DL  -  0 (number for drive 0)
; Registers changed: AX, CX, DX
;
cvtsec	proc	near

	xor	dx,dx
	div	sects			; compute sector number in a track
	inc	dl			; sector numbers always start with 1
	mov	cl,dl
	mov	dx,ax
	shr	ax,1			; divide by number of heads
	mov	ch,al			; save track number
	and	dx,1			; mask head number
	xchg	dl,dh
	ret

cvtsec	endp


;
;**************************************************************************
;
; Routine to print a string onto the console.
; Input:  DS:SI  -  address of NULL terminated string
; Output: None
; Registers changed: AX, SI
;
prnstr	proc	near

	push	bx
prnst1:	lodsb				; loop over all characters
	or	al,al
	jz	short prnst2
	mov	ah,0Eh			; BIOS command for character output
	mov	bx,0007h		; screen page and colors
	int	10h			; send character to screen
	jmp	short prnst1
prnst2:	pop	bx
	ret

prnstr	endp


;
;**************************************************************************
;
; Strings and messages:
;
sigmsg	db	'Disk loader for net boot'
	db	0Dh,0Ah,0Dh,0Ah,0

dskmsg	db	'Error reading diskette'
	db	0Dh,0Ah,0

botmsg	db	'Press a key to reboot...',0


;
;**************************************************************************
;
; Data space:
;
sects		dw	0	; number of sectors per track
oldvect		dd	0	; old dpt vector
newdpt		dptab	<>	; changed disk parameter table

		db	(offset load_text:[SECTSIZE-2] - $) dup (0)
		dw	0AA55h


;
;**************************************************************************
;
load_end			; end load segment

		end

