;;;
;;; tl-misc.el --- miscellaneous utility of tl.
;;;
;;; Copyright (C) 1995 Free Software Foundation, Inc.
;;; Copyright (C) 1995,1996 MORIOKA Tomohiko
;;;
;;; Author: MORIOKA Tomohiko <morioka@jaist.ac.jp>
;;; Version:
;;;	$Id: tl-misc.el,v 7.1 1996/02/12 07:34:30 morioka Exp $
;;; Keywords: load-path, module, structure
;;;
;;; This file is part of tl (Tiny Library).
;;;
;;; This program is free software; you can redistribute it and/or
;;; modify it under the terms of the GNU General Public License as
;;; published by the Free Software Foundation; either version 2, or
;;; (at your option) any later version.
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;;; General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with This program.  If not, write to the Free Software
;;; Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
;;;
;;; Code:

(require 'emu)
(require 'tl-str)


(defvar default-load-path load-path)

(defun add-path (path &rest options)
  "Add PATH to `load-path' if it exists under `default-load-path'
directories and it does not exist in `load-path'.

You can use following PATH styles:
	load-path relative: \"PATH/\"
			(it is searched from `defaul-load-path')
	home directory relative: \"~/PATH/\" \"~USER/PATH/\"
	absolute path: \"/HOO/BAR/BAZ/\"

You can specify following OPTIONS:
	'all-paths	search from `load-path'
			instead of `default-load-path'
	'append		add PATH to the last of `load-path'

\[tl-misc.el]"
  (let ((rest (if (memq 'all-paths options)
		  load-path
		default-load-path))
	p)
    (if (and (catch 'tag
	       (while rest
		 (setq p (expand-file-name path (car rest)))
		 (if (file-directory-p p)
		     (throw 'tag p)
		   )
		 (setq rest (cdr rest))
		 ))
	     (not (member p load-path))
	     )
	(setq load-path
	      (if (memq 'append options)
		  (append load-path (list p))
		(cons p load-path)
		))
      )))


(defun file-installed-p (file &optional paths)
  "Return t if FILE exists in PATHS.
If PATHS is omitted, `load-path' is used. [tl-misc.el]"
  (if (null paths)
      (setq paths load-path)
    )
  (catch 'tag
    (let (path)
      (while paths
	(setq path (expand-file-name file (car paths)))
	(if (file-exists-p path)
	    (throw 'tag path)
	  )
	(setq paths (cdr paths))
	))))


(defun call-after-loaded (module func &optional hook-name)
  "If MODULE is provided, then FUNC is called.
Otherwise func is set to MODULE-load-hook.
If optional argument HOOK-NAME is specified,
it is used as hook to set. [tl-misc.el]"
  (if (featurep module)
      (funcall func)
    (progn
      (if (null hook-name)
	  (setq hook-name (symbol-concat module "-load-hook"))
	)
      (add-hook hook-name func)
      )))


(defmacro define-structure (name &rest slots)
  (let ((pred (symbol-concat name '-p)))
    (cons 'progn
	  (nconc
	   (list
	    (` (defun (, pred) (obj)
		 (and (vectorp obj)
		      (eq (elt obj 0) '(, name))
		      ))
	       )
	    (` (defun (, (symbol-concat name '/create)) (, slots)
		 (, (cons 'vector (cons (list 'quote name) slots)))
		 )
	       ))
	   (let ((i 1))
	     (mapcar (function
		      (lambda (slot)
			(prog1
			    (` (defun (, (symbol-concat name '/ slot)) (obj)
				 (if ((, pred) obj)
				     (elt obj (, i))
				   ))
			       )
			  (setq i (+ i 1))
			  )
			)) slots)
	     )
	   (list (list 'quote name))
	   ))))


;;; @ end
;;;

(provide 'tl-misc)

;;; tl-misc.el ends here
