/*
 *  PGGraphvizActivityController.m
 *  graphviz
 *
 *  Created by Glen Low on Sat Jan 10 2004.
 *  Copyright (c) 2004, Pixelglow Software. All rights reserved.
 *  http://www.pixelglow.com/graphviz/
 *  graphviz@pixelglow.com
 *
 *  Redistribution and use in source and binary forms, with or without modification, are permitted
 *  provided that the following conditions are met:
 *  * Redistributions of source code must retain the above copyright notice, this list of conditions
 *    and the following disclaimer.
 *  * Redistributions in binary form must reproduce the above copyright notice, this list of conditions
 *    and the following disclaimer in the documentation and/or other materials provided with the distribution.
 *  * Neither the name of Pixelglow Software nor the names of its contributors may be used to endorse or
 *    promote products derived from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 *  INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *  ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 *  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 *  WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#import "PGGraphvizActivityController.h"
#import "PGSectionedString.h"
#import "PGTasker.h"

NSDictionary* commandAttributes_;
NSDictionary* errorAttributes_;

@implementation PGGraphvizActivityController

+ (void) initialize
	{
		commandAttributes_ = [[NSDictionary alloc] initWithObjectsAndKeys:
			[NSFont fontWithName: @"Courier-Bold" size: 11.0], NSFontAttributeName,
			[NSColor grayColor], NSForegroundColorAttributeName, nil];
		errorAttributes_ = [[NSDictionary alloc] initWithObjectsAndKeys:
			[NSFont fontWithName: @"Courier" size: 11.0], NSFontAttributeName,
			[NSColor grayColor], NSForegroundColorAttributeName, nil];
	}
	
- (void) awakeFromNib
	{
		objectSections_ = [[NSMutableDictionary alloc] init];
		
		[activity setAlphaValue: 0.9];
		
		NSNotificationCenter* defaultCenter = [NSNotificationCenter defaultCenter];
		
		[defaultCenter
			addObserver: self
			selector: @selector (taskerDidStart:)
			name: TaskerDidStart
			object: nil];

		[defaultCenter
			addObserver: self
			selector: @selector (taskerDidError:)
			name: TaskerDidError
			object: nil];
	}

- (IBAction) orderFrontActivityWindow: (id) sender
	{
		[activity orderFront: self];
	}
	
- (void) taskerDidStart: (NSNotification*) notification
	{
		PGTasker* tasker = [notification object];
		
		[self appendMessage: [tasker commandLine]
			attributes: commandAttributes_
			forObject: tasker];
	}

- (void) taskerDidError: (NSNotification*) notification
	{
		PGTasker* tasker = [notification object];
		
		NSData* errorData = [[notification userInfo] objectForKey: TaskerDataItem];
		if ([errorData length])
			{
				NSString* message =  [[NSString alloc] initWithData: errorData encoding: NSASCIIStringEncoding];
				[self appendMessage: message attributes: errorAttributes_ forObject: tasker];
				[message release];
			}
		else
			{
				// output a last line break
				[self appendMessage: @"\n" attributes: errorAttributes_ forObject: tasker];
				
				// remove gray color attribute from task output
				[[objectSections_ objectForKey: tasker] removeAttribute: NSForegroundColorAttributeName];
				[text setNeedsDisplay: YES];
					
				// forget about the task
				[objectSections_ removeObjectForKey: tasker];
			}

	}
	
- (void) appendMessage: (NSString*) message attributes: (NSDictionary*) attributes forObject: (NSObject*) object 
	{
		PGSectionedString* string = [objectSections_ objectForKey: object];
		
		if (!string)
			{
				string = [[PGSectionedString alloc] initWithMutableAttributedString: [text textStorage]];
				[objectSections_ setObject: string forKey: object];
				[string release];
			}
			
		NSAttributedString* attributedMessage = [[NSAttributedString alloc] initWithString: message attributes: attributes];
		[string appendAttributedString: attributedMessage];
		[attributedMessage release];
		
		// make sure addition is visible in text view
		NSRange stringRange = [string range];
		unsigned int messageLength = [message length];
		[text scrollRangeToVisible: NSMakeRange (stringRange.location + stringRange.length - messageLength, messageLength)];
	}
	
- (void) dealloc
	{
		[[NSNotificationCenter defaultCenter] removeObserver: self];

		[objectSections_ release];
		[super dealloc];
	}
	
@end
