/*
 *  apcevents.c  -- Output EVENTS information
 *
 *   Copyright (C) 1999 Kern Sibbald
 *	 19 November 1999
 *
 *  apcupsd.c -- Simple Daemon to catch power failure signals from a
 *		 BackUPS, BackUPS Pro, or SmartUPS (from APCC).
 *	      -- Now SmartMode support for SmartUPS and BackUPS Pro.
 *
 *  Copyright (C) 1996-99 Andre M. Hedrick <andre@suse.com>
 *  All rights reserved.
 *
 */

/*
 *		       GNU GENERAL PUBLIC LICENSE
 *			  Version 2, June 1991
 *
 *  Copyright (C) 1989, 1991 Free Software Foundation, Inc.
 *			     675 Mass Ave, Cambridge, MA 02139, USA
 *  Everyone is permitted to copy and distribute verbatim copies
 *  of this license document, but changing it is not allowed.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 *  IN NO EVENT SHALL ANY AND ALL PERSONS INVOLVED IN THE DEVELOPMENT OF THIS
 *  PACKAGE, NOW REFERRED TO AS "APCUPSD-Team" BE LIABLE TO ANY PARTY FOR 
 *  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING
 *  OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF ANY OR ALL
 *  OF THE "APCUPSD-Team" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  THE "APCUPSD-Team" SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING,
 *  BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 *  FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 *  ON AN "AS IS" BASIS, AND THE "APCUPSD-Team" HAS NO OBLIGATION TO PROVIDE
 *  MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 *  THE "APCUPSD-Team" HAS ABSOLUTELY NO CONNECTION WITH THE COMPANY
 *  AMERICAN POWER CONVERSION, "APCC".  THE "APCUPSD-Team" DID NOT AND
 *  HAS NOT SIGNED ANY NON-DISCLOSURE AGREEMENTS WITH "APCC".  ANY AND ALL
 *  OF THE LOOK-A-LIKE ( UPSlink(tm) Language ) WAS DERIVED FROM THE
 *  SOURCES LISTED BELOW.
 *
 */

#include "apc.h"

#define NLE   10		      /* number of events to send and keep */
#define MAXLE 50		      /* truncate file when this many events */

/*
 * If the EVENTS file exceeds MAXLE records, truncate it. 
 *
 * Returns:
 *
 *  0 if file not truncated
 *  1 if file truncated
 */
int truncate_events_file(UPSINFO *ups)
{
    char *le[NLE], *buf;
    int i, j;
    int nrec = 0;
    int tlen = 0;
    int trunc = FALSE;
    FILE *events_file;
    int stat = 0;
    
    if ((events_file = fopen(ups->eventfile, "r+")) == NULL)
	return 0;
    for (i=0; i<NLE; i++)
	le[i] = NULL;
    for (i=0; i<NLE; i++)
	if ((le[i] = malloc(MAXSTRING)) == NULL)
	    goto bailout;
    i = 0;
    while (fgets(le[i], MAXSTRING, events_file) != NULL) {
	nrec++;
	i++;
	if (i >= NLE)		       /* wrap */
	    i = 0;
    }
    if (nrec > MAXLE)
	trunc = TRUE;		       /* file too large, truncate it */
    if (nrec > NLE) {
	nrec = NLE;		       /* number of records to output */
	i -= (i/NLE)*NLE;	       /* first record to output */
    } else
	i = 0;
    /* get total length to output */
    for (j=0; j < nrec; j++)
	tlen += strlen(le[j]);
    if ((buf = malloc(tlen+1)) == NULL)
	goto bailout;
    *buf = 0;
    /* Put records in single buffer in correct order */
    for (j=0; j < nrec; j++) {
	strcat(buf, le[i++]); 
	if (i >= NLE)
	    i = 0;
    }
    if (trunc) {
	ftruncate(fileno(events_file), 0L);
	rewind(events_file);
	fwrite(buf, tlen, 1, events_file); /* write last NLE records to file */
	stat = 1;			   /* we truncated it */
    }

    free(buf);

bailout:

   fclose(events_file);
   for (i=0; i<NLE; i++)
       if (le[i] != NULL)
	   free(le[i]);
   return stat;
}


/*
 * Send the last events.
 * Returns:
 *	    -1 error or EOF
 *	     0 OK
 */
int output_events(int sockfd, FILE *events_file)
{
    char *le[NLE];
    int i, j;
    int nrec = 0;
    int stat = 0;

    for (i=0; i<NLE; i++)
	le[i] = NULL;
    for (i=0; i<NLE; i++)
	if ((le[i] = malloc(MAXSTRING)) == NULL)
	    goto bailout;
    i = 0;
    while (fgets(le[i], MAXSTRING, events_file) != NULL) {
	nrec++;
	i++;
	if (i >= NLE)		       /* wrap */
	    i = 0;
    }
    if (nrec > NLE) {
	nrec = NLE;		       /* number of records to output */
	i -= (i/NLE)*NLE;	       /* first record to output */
    } else
	i = 0;
    for (j=0; j < nrec; j++) {
	if (net_send(sockfd, le[i], strlen(le[i])) <= 0)  
	    goto bailout;
	if (++i >= NLE)
	    i = 0;
     }

    goto goodout;

bailout:
    stat = -1;

goodout:
    if (net_send(sockfd, NULL, 0) < 0) /* send eof */
	stat = -1;
    for (i=0; i<NLE; i++)
	if (le[i] != NULL)
	    free(le[i]);
    return stat;
}

#ifdef HAVE_CYGWIN

#include <windows.h>

extern UPSINFO myUPS;
extern int shm_OK;

/*  
 * Fill the Events list box with the last events
 * 
 */
void FillEventsBox(HWND hwnd, int id_list)
{
    char buf[1000];
    int len;
    FILE *events_file;
    
    if (!shm_OK || myUPS.eventfile[0] == 0 ||
        (events_file = fopen(myUPS.eventfile, "r")) == NULL) {
	SendDlgItemMessage(hwnd, id_list, LB_ADDSTRING, 0, 
           (LONG)"Events not available");
	return;
    }

    while (fgets(buf, sizeof(buf), events_file) != NULL) {
	len = strlen(buf);
	/* strip trailing cr/lfs */
        while (len > 0 && (buf[len-1] == '\n' || buf[len-1] == '\r'))
	    buf[--len] = 0;
	SendDlgItemMessage(hwnd, id_list, LB_ADDSTRING, 0, (LONG)buf);
    }
    return;
}

#endif /* HAVE_CYGWIN */
