/*****************************************************************************/
/*  config_file.c - config file routines                                     */
/*  Copyright (C) 1998-1999 Brian Masney <masneyb@seul.org>                  */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#include "ftp.h"

static void make_general_tab (GtkWidget *notebook);
static void make_ftp_tab (GtkWidget *notebook);
static void make_http_tab (GtkWidget *notebook);
static void make_proxy_hosts_tab (GtkWidget *notebook);
static void add_host_to_listbox (GList *templist);
static void add_proxy_host (GtkWidget *widget, gpointer data);
static void add_toggle (GtkWidget *widget, gpointer data);
static void add_ok (GtkWidget *widget, gpointer data);
static void delete_proxy_host (GtkWidget *widget, gpointer data);
static void proxy_toggle(GtkList *list, GtkWidget *child, gpointer data);
static void apply_changes (GtkWidget *widget, gpointer data);
static void clean_old_changes (GtkWidget *widget, gpointer data);
static char *get_proxy_config (void);
static void add_to_bookmark (struct conn_categories *newentry);
static void write_comment (FILE *fd, const char *comment);
static int copyfile (char *source, char *dest);
static gint hash_compare (gconstpointer path1, gconstpointer path2);
static guint hash_function (gconstpointer key);

struct config_vars {
   char *key, /* Our config file variable name */
        *description, /* How this field will show up in the options dialog */
        type; /* Type Character or Integer or AlphaNumeric */
   void *var; /* Pointer to our variable */
   int general_option : 1; /* Will this go into the general settings tab */
   char *comment; /* Comment to write out to the config file */
   GtkWidget *widget;
}; 

struct proxy_type_tag {
   char *key, 
        *desc;
};


static GtkWidget *proxy_text, *proxy_list, *new_proxy_domain, *network1,
	*network2, *network3, *network4, *netmask1, *netmask2, *netmask3,
	*netmask4, *domain_active;
static GList *new_proxy_hosts;
static char *custom_proxy;
static int proxy_num;

struct config_vars config_file_vars[] = {
/* You MUST keep the firewall stuff up here as the first entries and in the 
   same order. The options dialog relies on it being here */
   {"firewall_host", 		N_("Proxy hostname:"), 		'C', &firewall_host, 		0, N_("Firewall hostname"), NULL},
   {"firewall_port", 		N_("Proxy port:"), 		'A', &firewall_port,		0, N_("Port to connect to on the firewall"), NULL},
   {"firewall_username", 	N_("Proxy username:"), 		'C', &firewall_username,	0, N_("Your firewall username"), NULL},
   {"firewall_password", 	N_("Proxy password:"), 		'C', &firewall_password, 	0, N_("Your firewall password"), NULL},
   {"firewall_account", 	N_("Proxy account:"), 		'C', &firewall_account,		0, N_("Your firewall account (optional)"), NULL},
   {"proxy_config",		N_("Proxy config"),		'C', &proxy_config,		0, N_("This specifies how your proxy server expects us to log in"), NULL},
   {"http_proxy_host", 		N_("Proxy hostname:"), 		'C', &http_proxy_host, 		0, N_("Firewall hostname"), NULL},
   {"http_proxy_port", 		N_("Proxy port:"), 		'A', &http_proxy_port,		0, N_("Port to connect to on the firewall"), NULL},
   {"http_proxy_username", 	N_("Proxy username:"), 		'C', &http_proxy_username,	0, N_("Your firewall username"), NULL},
   {"http_proxy_password", 	N_("Proxy password:"), 		'C', &http_proxy_password, 	0, N_("Your firewall password"), NULL},
   {"email", 			N_("Email address:"), 		'C', &emailaddr, 		1, N_("Enter your email address here"), NULL},
   {"view_program", 		N_("View program:"), 		'C', &view_program, 		1, N_("The default program used to view files. If this is blank, the internal file viewer will be used"), NULL},
   {"edit_program", 		N_("Edit program:"), 		'C', &edit_program, 		1, N_("The default program used to edit files."), NULL},
   {"connect_timeout",		N_("Connect timeout:"),		'A', &timeout,			1, N_("The max timeout for the connection"), NULL},
   {"retries",			N_("Connect retries:"),		'A', &retries,			1, N_("The number of auto-retries to do. Set this to 0 to retry indefinately"), NULL},
   {"sleep_time",		N_("Retry sleep time:"),	'A', &sleep_time,		1, N_("The number of seconds to wait between retries"), NULL},
   {"maxkbs",			N_("Max KB/S:"),		'F', &maxkbs,			1, N_("The maximum KB/s a file transfer can get. (Set to 0 to disable)"), NULL},
   {"reconnect_diag",		N_("Bring up reconnect dialog"),'I', &reconnect_diag,		1, N_("Bring up the reconnect dialog after login failure"), NULL},
   {"confirm_delete",		N_("Confirm delete"),		'I', &confirm_delete,		1, N_("Confirm when deleting files"), NULL},
   {"one_transfer", 		N_("Do one transfer at a time"),'I', &do_one_transfer_at_a_time,1, N_("Do only one transfer at a time?"), NULL},
   {"passive_transfer",		N_("Passive file transfers"), 	'I', &passive_transfer,		1, N_("Send PASV command or PORT command for data transfers"), NULL},
   {"preserve_permissions",	N_("Preserve permissions"),	'I', &preserve_permissions,	1, N_("Save the permissions of the transfered files"), NULL},
   {"refresh_files",		N_("Refresh after each file transfer"), 'I', &refresh_files,	1, N_("Refresh the listbox after each file is transfered"), NULL},
   {"resolve_symlinks",		N_("Resolve Remote Symlinks"),	'I', &resolve_symlinks,		1, N_("If you disable this feature, then gFTP will only send LIST to the remote server instead of LIST -L"), NULL},
   {"save_geometry", 		N_("Save geometry"), 		'I', &save_geometry, 		1, N_("Save the size of each widget for next startup"), NULL},
   {"show_hidden_files",	N_("Show hidden files"),	'I', &show_hidden_files,	1, N_("Show hidden files in the listboxes"), NULL},
   {"sort_dirs_first", 		N_("Sort directories first"), 	'I', &sort_dirs_first, 		1, N_("Put the directories first then the files"), NULL},
   {"start_transfers",		N_("Start file transfers"), 	'I', &start_file_transfers, 	1, N_("Automatically start the file transfers when they get queued?"), NULL},
   {"usecache", 		N_("Use cache"),	 	'I', &use_cache, 		1, N_("Do you want to use the cache?"), NULL},
   {"use_default_dl_types",	"", 				'I', &use_default_dl_types, 	0, N_("(*) If this is set, and there is a ext= line below for the file extension, it will download the file as specified below"), NULL},
   {"listbox_local_width",	"", 				'I', &listbox_local_width, 	0, N_("The default width of the local files listbox"), NULL},
   {"listbox_remote_width", 	"", 				'I', &listbox_remote_width, 	0, N_("The default width of the remote files listbox"), NULL},
   {"listbox_file_height", 	"", 				'I', &listbox_file_height, 	0, N_("The default height of the local/remote files listboxes"), NULL},
   {"transfer_height", 		"", 				'I', &transfer_height, 		0, N_("The default height of the transfer listbox"), NULL},
   {"log_height", 		"", 				'I', &log_height, 		0, N_("The default height of the logging window"), NULL},
   {"file_trans_column",	"",				'I', &file_trans_column,	0, N_("The width of the filename column in the transfer window. Set this to 0 to have this column automagically resize."), NULL},
   {"local_file_width",		"",				'I', &local_columns[0],		0, N_("The width of the filename column in the file listboxes. Set this to 0 to have this column automagically resize. Set this to -1 to disable this column"), NULL},
   {"local_size_width", 	"", 				'I', &local_columns[1], 	0, N_("The width of the size column in the file listboxes. Set this to 0 to have this column automagically resize. Set this to -1 to disable this column"), NULL},
   {"local_user_width", 	"", 				'I', &local_columns[2], 	0, N_("The width of the user column in the file listboxes. Set this to 0 to have this column automagically resize. Set this to -1 to disable this column"), NULL},
   {"local_group_width", 	"", 				'I', &local_columns[3], 	0, N_("The width of the group column in the file listboxes. Set this to 0 to have this column automagically resize. Set this to -1 to disable this column"), NULL},
   {"local_date_width", 	"", 				'I', &local_columns[4], 	0, N_("The width of the date column in the file listboxes. Set this to 0 to have this column automagically resize. Set this to -1 to disable this column"), NULL},
   {"local_attribs_width", 	"", 				'I', &local_columns[5], 	0, N_("The width of the attribs column in the file listboxes. Set this to 0 to have this column automagically resize. Set this to -1 to disable this column"), NULL},
   {"remote_file_width",	"",				'I', &remote_columns[0],	0, N_("The width of the filename column in the file listboxes. Set this to 0 to have this column automagically resize. Set this to -1 to disable this column"), NULL},
   {"remote_size_width", 	"", 				'I', &remote_columns[1], 	0, N_("The width of the size column in the file listboxes. Set this to 0 to have this column automagically resize. Set this to -1 to disable this column"), NULL},
   {"remote_user_width", 	"", 				'I', &remote_columns[2], 	0, N_("The width of the user column in the file listboxes. Set this to 0 to have this column automagically resize. Set this to -1 to disable this column"), NULL},
   {"remote_group_width", 	"", 				'I', &remote_columns[3], 	0, N_("The width of the group column in the file listboxes. Set this to 0 to have this column automagically resize. Set this to -1 to disable this column"), NULL},
   {"remote_date_width", 	"", 				'I', &remote_columns[4], 	0, N_("The width of the date column in the file listboxes. Set this to 0 to have this column automagically resize. Set this to -1 to disable this column"), NULL},
   {"remote_attribs_width", 	"", 				'I', &remote_columns[5], 	0, N_("The width of the attribs column in the file listboxes. Set this to 0 to have this column automagically resize. Set this to -1 to disable this column"), NULL},
   {"", "", '\0', NULL, 0, "", NULL}};

struct proxy_type_tag proxy_type[] = {
   {N_("none"), 	""},
   {N_("SITE command"), "USER %pu\nPASS %pp\nSITE %hh\nUSER %hu\nPASS %hp\n"},
   {N_("user@host"), 	"USER %pu\nPASS %pp\nUSER %hu@%hh\nPASS %hp\n"},
   {N_("user@host:port"), "USER %hu@%hh:%ho\nPASS %hp\n"},
   {N_("AUTHENTICATE"),	"USER %hu@%hh\nPASS %hp\nSITE AUTHENTICATE %pu\nSITE RESPONSE %pp\n"},
   {N_("user@host port"), "USER %hu@%hh %ho\nPASS %hp\n"},
   {N_("user@host NOAUTH"), "USER %hu@%hh\nPASS %hp\n"},
   {N_("HTTP Proxy"),	"http"},
   {N_("Custom"),	""},
   {"", ""}};

#define CUSTOM_PROXY_NUM	8

void options_dialog (gpointer data) {
   GtkWidget *tempwid, *dialog, *notebook;

   dialog = gtk_dialog_new ();
   gtk_window_set_title (GTK_WINDOW (dialog), _("Options"));
   gtk_container_border_width (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), 10);
   gtk_container_border_width (GTK_CONTAINER (GTK_DIALOG (dialog)->action_area), 5);
   gtk_box_set_spacing (GTK_BOX (GTK_DIALOG (dialog)->vbox), 2);
   gtk_box_set_spacing (GTK_BOX (GTK_DIALOG (dialog)->action_area), 15);
   gtk_box_set_homogeneous (GTK_BOX (GTK_DIALOG (dialog)->action_area), TRUE);
   gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_MOUSE);
   gtk_signal_connect_object (GTK_OBJECT (dialog), "delete_event", GTK_SIGNAL_FUNC (gtk_widget_destroy), GTK_OBJECT (dialog));
   
   notebook = gtk_notebook_new ();
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), notebook, TRUE, TRUE, 0);
   gtk_widget_show (notebook);

   make_general_tab (notebook);
   make_ftp_tab (notebook);
   make_http_tab (notebook);
   make_proxy_hosts_tab (notebook);
   
   tempwid = gtk_button_new_with_label (_("OK"));
   GTK_WIDGET_SET_FLAGS (tempwid, GTK_CAN_DEFAULT);
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->action_area), tempwid, TRUE, TRUE, 0);
   gtk_signal_connect (GTK_OBJECT (tempwid), "clicked", GTK_SIGNAL_FUNC (apply_changes), NULL);
   gtk_signal_connect_object (GTK_OBJECT (tempwid), "clicked", GTK_SIGNAL_FUNC (gtk_widget_destroy), GTK_OBJECT (dialog));
   gtk_widget_show (tempwid);

   tempwid = gtk_button_new_with_label (_("  Cancel  "));
   GTK_WIDGET_SET_FLAGS (tempwid, GTK_CAN_DEFAULT);
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->action_area), tempwid, TRUE, TRUE, 0);
   gtk_signal_connect (GTK_OBJECT (tempwid), "clicked", GTK_SIGNAL_FUNC (clean_old_changes), NULL);
   gtk_signal_connect_object (GTK_OBJECT (tempwid), "clicked", GTK_SIGNAL_FUNC (gtk_widget_destroy), GTK_OBJECT (dialog));
   gtk_widget_show (tempwid);

   tempwid = gtk_button_new_with_label (_("Apply"));
   GTK_WIDGET_SET_FLAGS (tempwid, GTK_CAN_DEFAULT);
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->action_area), tempwid, TRUE, TRUE, 0);
   gtk_signal_connect (GTK_OBJECT (tempwid), "clicked", GTK_SIGNAL_FUNC (apply_changes), NULL);
   gtk_widget_grab_default (tempwid);
   gtk_widget_show (tempwid);

   gtk_widget_show (dialog);
}
/*****************************************************************************/
static void make_general_tab (GtkWidget *notebook) {
   GtkWidget *tempwid, *box, *inttbl, *chartbl;
   int num, tbl_len, tbl_num, tbl_col;
   char *tempstr;
   
   box = gtk_vbox_new (FALSE, 0);
   gtk_container_border_width (GTK_CONTAINER (box), 10);
   gtk_widget_show (box);

   tempwid = gtk_label_new (_("General"));
   gtk_widget_show (tempwid);
   gtk_notebook_append_page (GTK_NOTEBOOK (notebook), box, tempwid);   

   tbl_len = 1;
   chartbl = gtk_table_new (tbl_len, 2, FALSE);
   gtk_table_set_row_spacings (GTK_TABLE (chartbl), 5);
   gtk_table_set_col_spacings (GTK_TABLE (chartbl), 5);
   gtk_box_pack_start (GTK_BOX (box), chartbl, FALSE, FALSE, 0);
   gtk_widget_show (chartbl);
   
   tbl_num = tbl_col = 0;
   inttbl = gtk_table_new (1, 2, FALSE);
   gtk_table_set_row_spacings (GTK_TABLE (inttbl), 2);
   gtk_table_set_col_spacings (GTK_TABLE (inttbl), 5);
   gtk_box_pack_start (GTK_BOX (box), inttbl, FALSE, FALSE, 0);
   gtk_widget_show (inttbl);

   num = 0;
   while (config_file_vars[num].var != NULL) {
      if (config_file_vars[num].general_option) {
         if (config_file_vars[num].type == 'I') {
            tempwid = gtk_check_button_new_with_label (_(config_file_vars[num].description));
            gtk_table_attach_defaults (GTK_TABLE (inttbl), tempwid, tbl_col, tbl_col + 1, tbl_num, tbl_num + 1);
            gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (tempwid), *(int *) config_file_vars[num].var);
            gtk_widget_show (tempwid);
            config_file_vars[num].widget = tempwid;
            tbl_col++;
            if (tbl_col == 2) {
               tbl_col = 0;
               tbl_num++;
               gtk_table_resize (GTK_TABLE (inttbl), tbl_num + 1, 2);
            }
         }
         else {
            tbl_len++;
            gtk_table_resize (GTK_TABLE (chartbl), tbl_len, 2);
            
            tempwid = gtk_label_new (_(config_file_vars[num].description));
            gtk_misc_set_alignment (GTK_MISC (tempwid), 1, 0.5);
            gtk_table_attach_defaults (GTK_TABLE (chartbl), tempwid, 0, 1, tbl_len-1, tbl_len);
            gtk_widget_show (tempwid);

            tempwid = gtk_entry_new ();
            gtk_table_attach_defaults (GTK_TABLE (chartbl), tempwid, 1, 2, tbl_len-1, tbl_len);
            if (config_file_vars[num].type == 'A') {
               tempstr = g_strdup_printf ("%d", *(int *) config_file_vars[num].var);
               gtk_entry_set_text (GTK_ENTRY (tempwid), tempstr);
               g_free (tempstr);
            }
            else if (config_file_vars[num].type == 'F') {
               tempstr = g_strdup_printf ("%.2f", *(float *) config_file_vars[num].var);
               gtk_entry_set_text (GTK_ENTRY (tempwid), tempstr);
               g_free (tempstr);
            }
            else {
               gtk_entry_set_text (GTK_ENTRY (tempwid), *(char **) config_file_vars[num].var);
            }
            gtk_widget_show (tempwid);
            config_file_vars[num].widget = tempwid; 
         }
      }
      num++;
   }
}
/*****************************************************************************/
static void make_ftp_tab (GtkWidget *notebook) {
   GtkWidget *tempwid, *box, *table, *label, *proxy_combo;
   char *tempstr, *pos, *endpos, *oldstr;
   GList *proxy_list;
   int num;
   
   proxy_list = NULL;
   
   box = gtk_vbox_new (FALSE, 5);
   gtk_container_border_width (GTK_CONTAINER (box), 10);
   gtk_widget_show (box);

   tempwid = gtk_label_new (_("FTP Proxy"));
   gtk_widget_show (tempwid);
   gtk_notebook_append_page (GTK_NOTEBOOK (notebook), box, tempwid);   

   table = gtk_table_new (8, 2, FALSE);
   gtk_table_set_row_spacings (GTK_TABLE (table), 5);
   gtk_table_set_col_spacings (GTK_TABLE (table), 5);
   gtk_box_pack_start (GTK_BOX (box), table, FALSE, FALSE, 0);
   gtk_widget_show (table);

   tempwid = gtk_label_new (_(config_file_vars[0].description));
   gtk_misc_set_alignment (GTK_MISC (tempwid), 1, 0.5);
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 0, 1, 0, 1);
   gtk_widget_show (tempwid);

   tempwid = gtk_entry_new ();
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 1, 2, 0, 1);
   gtk_entry_set_text (GTK_ENTRY (tempwid), *(char **) config_file_vars[0].var);
   gtk_widget_show (tempwid);
   config_file_vars[0].widget = tempwid;

   tempwid = gtk_label_new (_(config_file_vars[1].description));
   gtk_misc_set_alignment (GTK_MISC (tempwid), 1, 0.5);
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 0, 1, 1, 2);
   gtk_widget_show (tempwid);

   tempwid = gtk_entry_new ();
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 1, 2, 1, 2);
   tempstr = g_strdup_printf ("%d", *(int *) config_file_vars[1].var);
   gtk_entry_set_text (GTK_ENTRY (tempwid), tempstr);
   g_free (tempstr);
   gtk_widget_show (tempwid);
   config_file_vars[1].widget = tempwid;

   tempwid = gtk_label_new (_(config_file_vars[2].description));
   gtk_misc_set_alignment (GTK_MISC (tempwid), 1, 0.5);
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 0, 1, 2, 3);
   gtk_widget_show (tempwid);

   tempwid = gtk_entry_new ();
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 1, 2, 2, 3);
   gtk_entry_set_text (GTK_ENTRY (tempwid), *(char **) config_file_vars[2].var);
   gtk_widget_show (tempwid);
   config_file_vars[2].widget = tempwid;

   tempwid = gtk_label_new (_(config_file_vars[3].description));
   gtk_misc_set_alignment (GTK_MISC (tempwid), 1, 0.5);
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 0, 1, 3, 4);
   gtk_widget_show (tempwid);

   tempwid = gtk_entry_new ();
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 1, 2, 3, 4);
   gtk_entry_set_text (GTK_ENTRY (tempwid), *(char **) config_file_vars[3].var);
   gtk_entry_set_visibility (GTK_ENTRY (tempwid), FALSE);
   gtk_widget_show (tempwid);
   config_file_vars[3].widget = tempwid;

   tempwid = gtk_label_new (_(config_file_vars[4].description));
   gtk_misc_set_alignment (GTK_MISC (tempwid), 1, 0.5);
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 0, 1, 4, 5);
   gtk_widget_show (tempwid);

   tempwid = gtk_entry_new ();
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 1, 2, 4, 5);
   gtk_entry_set_text (GTK_ENTRY (tempwid), *(char **) config_file_vars[4].var);
   gtk_entry_set_visibility (GTK_ENTRY (tempwid), FALSE);
   gtk_widget_show (tempwid);
   config_file_vars[4].widget = tempwid;

   tempwid = gtk_label_new (_("Proxy server type"));
   gtk_misc_set_alignment (GTK_MISC (tempwid), 1, 0.5);
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 0, 1, 5, 6);
   gtk_widget_show (tempwid);

   proxy_combo = gtk_combo_new ();
   gtk_table_attach_defaults (GTK_TABLE (table), proxy_combo, 1, 2, 5, 6);
   gtk_widget_show (proxy_combo);

   num = 0;
   while (*proxy_type[num].key != '\0') {
      label = gtk_list_item_new_with_label (_(proxy_type[num].key));
      gtk_widget_show (label);
      proxy_list = g_list_append (proxy_list, label);
      num++;
   }
   gtk_list_prepend_items (GTK_LIST (GTK_COMBO(proxy_combo)->list), proxy_list);

   custom_proxy = g_malloc (1);
   *custom_proxy = '\0';
   if (proxy_config == NULL || *proxy_config == '\0') {
      proxy_num = 0;
   }
   else {
      pos = proxy_config;
      while ((endpos = strstr (pos, "%n"))) {
         *endpos = '\0';
         oldstr = custom_proxy;
         custom_proxy = g_strconcat (custom_proxy, pos, "\n", NULL);
         g_free (oldstr);
         *endpos = '%';
         pos = endpos + 2;
      }
      if (strlen (pos) > 0) {
         oldstr = custom_proxy;
         custom_proxy = g_strconcat (custom_proxy, pos, NULL);
         g_free (oldstr);
      }

      for (proxy_num = 1; proxy_num < CUSTOM_PROXY_NUM; proxy_num++) {
         if (strcmp (proxy_type[proxy_num].desc, custom_proxy) == 0) {
            break;
         }
      }
   }

   proxy_text = gtk_text_new (NULL, NULL);
   gtk_widget_set_usize (proxy_text, -2, 125);
   gtk_text_set_editable (GTK_TEXT (proxy_text), TRUE);
   gtk_table_attach_defaults (GTK_TABLE (table), proxy_text, 0, 2, 6, 7);
   gtk_widget_show (proxy_text);

   tempwid = gtk_table_new (5, 2, TRUE);
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 0, 2, 7, 8);
   gtk_widget_show (tempwid);
   table = tempwid;
   
   tempwid = gtk_label_new (_("%pu = proxy user"));
   gtk_misc_set_alignment (GTK_MISC (tempwid), 0, 0.5);
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 0, 1, 0, 1);
   gtk_widget_show (tempwid);
   
   tempwid = gtk_label_new (_("%hu = host user"));
   gtk_misc_set_alignment (GTK_MISC (tempwid), 1, 0.5);
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 1, 2, 0, 1);
   gtk_widget_show (tempwid);

   tempwid = gtk_label_new (_("%pp = proxy pass"));
   gtk_misc_set_alignment (GTK_MISC (tempwid), 0, 0.5);
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 0, 1, 1, 2);
   gtk_widget_show (tempwid);

   tempwid = gtk_label_new (_("%hp = host pass"));
   gtk_misc_set_alignment (GTK_MISC (tempwid), 1, 0.5);
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 1, 2, 1, 2);
   gtk_widget_show (tempwid);

   tempwid = gtk_label_new (_("%ph = proxy host"));
   gtk_misc_set_alignment (GTK_MISC (tempwid), 0, 0.5);
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 0, 1, 2, 3);
   gtk_widget_show (tempwid);

   tempwid = gtk_label_new (_("%hh = host"));
   gtk_misc_set_alignment (GTK_MISC (tempwid), 1, 0.5);
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 1, 2, 2, 3);
   gtk_widget_show (tempwid);

   tempwid = gtk_label_new (_("%po = proxy port"));
   gtk_misc_set_alignment (GTK_MISC (tempwid), 0, 0.5);
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 0, 1, 3, 4);
   gtk_widget_show (tempwid);

   tempwid = gtk_label_new (_("%ho = host port"));
   gtk_misc_set_alignment (GTK_MISC (tempwid), 1, 0.5);
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 1, 2, 3, 4);
   gtk_widget_show (tempwid);

   tempwid = gtk_label_new (_("%pa = proxy account"));
   gtk_misc_set_alignment (GTK_MISC (tempwid), 0, 0.5);
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 0, 1, 4, 5);
   gtk_widget_show (tempwid);

   tempwid = gtk_label_new (_("%ha = host account"));
   gtk_misc_set_alignment (GTK_MISC (tempwid), 1, 0.5);
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 1, 2, 4, 5);
   gtk_widget_show (tempwid);

   gtk_signal_connect (GTK_OBJECT (GTK_COMBO (proxy_combo)->list), "select_child", GTK_SIGNAL_FUNC (proxy_toggle), NULL);
   gtk_list_select_item (GTK_LIST (GTK_COMBO (proxy_combo)->list), proxy_num);
}
/*****************************************************************************/
static void make_http_tab (GtkWidget *notebook) {
   GtkWidget *box, *tempwid, *table;
   char *tempstr;
   
   box = gtk_vbox_new (FALSE, 5);
   gtk_container_border_width (GTK_CONTAINER (box), 10);
   gtk_widget_show (box);

   tempwid = gtk_label_new (_("HTTP Proxy"));
   gtk_widget_show (tempwid);
   gtk_notebook_append_page (GTK_NOTEBOOK (notebook), box, tempwid);   

   table = gtk_table_new (4, 2, FALSE);
   gtk_table_set_row_spacings (GTK_TABLE (table), 5);
   gtk_table_set_col_spacings (GTK_TABLE (table), 5);
   gtk_box_pack_start (GTK_BOX (box), table, FALSE, FALSE, 0);
   gtk_widget_show (table);

   tempwid = gtk_label_new (_(config_file_vars[6].description));
   gtk_misc_set_alignment (GTK_MISC (tempwid), 1, 0.5);
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 0, 1, 0, 1);
   gtk_widget_show (tempwid);

   tempwid = gtk_entry_new ();
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 1, 2, 0, 1);
   gtk_entry_set_text (GTK_ENTRY (tempwid), *(char **) config_file_vars[6].var);
   gtk_widget_show (tempwid);
   config_file_vars[6].widget = tempwid;

   tempwid = gtk_label_new (_(config_file_vars[7].description));
   gtk_misc_set_alignment (GTK_MISC (tempwid), 1, 0.5);
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 0, 1, 1, 2);
   gtk_widget_show (tempwid);

   tempwid = gtk_entry_new ();
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 1, 2, 1, 2);
   tempstr = g_strdup_printf ("%d", *(int *) config_file_vars[7].var);
   gtk_entry_set_text (GTK_ENTRY (tempwid), tempstr);
   g_free (tempstr);
   gtk_widget_show (tempwid);
   config_file_vars[7].widget = tempwid;

   tempwid = gtk_label_new (_(config_file_vars[8].description));
   gtk_misc_set_alignment (GTK_MISC (tempwid), 1, 0.5);
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 0, 1, 2, 3);
   gtk_widget_show (tempwid);

   tempwid = gtk_entry_new ();
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 1, 2, 2, 3);
   gtk_entry_set_text (GTK_ENTRY (tempwid), *(char **) config_file_vars[8].var);
   gtk_widget_show (tempwid);
   config_file_vars[8].widget = tempwid;

   tempwid = gtk_label_new (_(config_file_vars[9].description));
   gtk_misc_set_alignment (GTK_MISC (tempwid), 1, 0.5);
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 0, 1, 3, 4);
   gtk_widget_show (tempwid);

   tempwid = gtk_entry_new ();
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 1, 2, 3, 4);
   gtk_entry_set_text (GTK_ENTRY (tempwid), *(char **) config_file_vars[9].var);
   gtk_entry_set_visibility (GTK_ENTRY (tempwid), FALSE);
   gtk_widget_show (tempwid);
   config_file_vars[9].widget = tempwid;
}
/*****************************************************************************/
static void make_proxy_hosts_tab (GtkWidget *notebook) {
   GtkWidget *tempwid, *box, *hbox, *scroll;
   gftp_proxy_hosts *hosts, *newhosts;
   char *add_data[2];
   GList *templist;

   add_data[0] = _("Network");
   add_data[1] = _("Netmask");

   box = gtk_vbox_new (FALSE, 5);
   gtk_container_border_width (GTK_CONTAINER (box), 10);
   gtk_widget_show (box);

   tempwid = gtk_label_new (_("Local Hosts"));
   gtk_widget_show (tempwid);
   gtk_notebook_append_page (GTK_NOTEBOOK (notebook), box, tempwid);
   
   scroll = gtk_scrolled_window_new (NULL, NULL);
   gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll), 
   	GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
   gtk_box_pack_start (GTK_BOX (box), scroll, TRUE, TRUE, 0);

   proxy_list = gtk_clist_new_with_titles (2, add_data);
   gtk_container_add (GTK_CONTAINER (scroll), proxy_list);
   gtk_clist_set_column_auto_resize (GTK_CLIST (proxy_list), 0, TRUE);
   gtk_clist_set_column_auto_resize (GTK_CLIST (proxy_list), 1, TRUE);
   gtk_widget_show (proxy_list);
   gtk_widget_show (scroll);

   hbox = gtk_hbox_new (TRUE, 15);
   gtk_box_pack_start (GTK_BOX (box), hbox, FALSE, FALSE, 0);
   gtk_widget_show (hbox);

   tempwid = gtk_button_new_with_label (_("Add"));
   GTK_WIDGET_SET_FLAGS (tempwid, GTK_CAN_DEFAULT);
   gtk_box_pack_start (GTK_BOX (hbox), tempwid, TRUE, TRUE, 0);
   gtk_signal_connect (GTK_OBJECT (tempwid), "clicked", GTK_SIGNAL_FUNC (add_proxy_host), NULL);
   gtk_widget_show (tempwid);

   tempwid = gtk_button_new_with_label (_("Edit"));
   GTK_WIDGET_SET_FLAGS (tempwid, GTK_CAN_DEFAULT);
   gtk_box_pack_start (GTK_BOX (hbox), tempwid, TRUE, TRUE, 0);
   gtk_signal_connect (GTK_OBJECT (tempwid), "clicked", GTK_SIGNAL_FUNC (add_proxy_host), (gpointer) 1);
   gtk_widget_show (tempwid);

   tempwid = gtk_button_new_with_label (_("Delete"));
   GTK_WIDGET_SET_FLAGS (tempwid, GTK_CAN_DEFAULT);
   gtk_box_pack_start (GTK_BOX (hbox), tempwid, TRUE, TRUE, 0);
   gtk_signal_connect (GTK_OBJECT (tempwid), "clicked", GTK_SIGNAL_FUNC (delete_proxy_host), NULL);
   gtk_widget_show (tempwid);
   
   new_proxy_hosts = NULL;
   for (templist = new_proxy_hosts; templist != NULL; templist = templist->next) {
      hosts = templist->data;
      newhosts = g_malloc (sizeof (gftp_proxy_hosts));
      memcpy (newhosts, hosts, sizeof (gftp_proxy_hosts));
      if (newhosts->domain) {
         newhosts->domain = g_malloc (strlen (hosts->domain) + 1);
         strcpy (newhosts->domain, hosts->domain);
      }
      new_proxy_hosts = g_list_prepend (new_proxy_hosts, newhosts);
      add_host_to_listbox (new_proxy_hosts);
   }
}
/*****************************************************************************/
static void add_host_to_listbox (GList *templist) {
   gftp_proxy_hosts *hosts;
   char *add_data[2];
   int num;
   
   hosts = templist->data;
   if (hosts->domain) {
      add_data[0] = hosts->domain;
      add_data[1] = NULL;
      num = gtk_clist_append (GTK_CLIST (proxy_list), add_data);
   }
   else {
      add_data[0] = g_strdup_printf ("%d.%d.%d.%d", 
      	hosts->network_address >> 24 & 0xff,
      	hosts->network_address >> 16 & 0xff,
      	hosts->network_address >> 8 & 0xff,
      	hosts->network_address & 0xff);
      add_data[1] = g_strdup_printf ("%d.%d.%d.%d",
      	hosts->netmask >> 24 & 0xff,
      	hosts->netmask >> 16 & 0xff,
      	hosts->netmask >> 8 & 0xff,
      	hosts->netmask & 0xff);
      num = gtk_clist_append (GTK_CLIST (proxy_list), add_data);
      g_free (add_data[0]);
      g_free (add_data[1]);
   }
   gtk_clist_set_row_data (GTK_CLIST (proxy_list), num, (gpointer) templist);
}
/*****************************************************************************/
static void add_proxy_host (GtkWidget *widget, gpointer data) {
   GtkWidget *tempwid, *dialog, *frame, *box, *table;
   gftp_proxy_hosts *hosts;
   GList *templist;
   char *tempstr;

   if (data) {
      if ((templist = GTK_CLIST (proxy_list)->selection) == NULL) return;
      templist = gtk_clist_get_row_data (GTK_CLIST (proxy_list), (int) templist->data);
      hosts = templist->data;
   }
   else {
      hosts = NULL;
      templist = NULL;
   }
     
   dialog = gtk_dialog_new ();
   gtk_window_set_title (GTK_WINDOW (dialog), hosts ? _("Edit Host") : _("Add Host"));
   gtk_container_border_width (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), 10);
   gtk_container_border_width (GTK_CONTAINER (GTK_DIALOG (dialog)->action_area), 5);
   gtk_box_set_spacing (GTK_BOX (GTK_DIALOG (dialog)->vbox), 2);
   gtk_box_set_spacing (GTK_BOX (GTK_DIALOG (dialog)->action_area), 15);
   gtk_box_set_homogeneous (GTK_BOX (GTK_DIALOG (dialog)->action_area), TRUE);
   gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_MOUSE);
   gtk_signal_connect_object (GTK_OBJECT (dialog), "delete_event", GTK_SIGNAL_FUNC (gtk_widget_destroy), GTK_OBJECT (dialog));
   gtk_grab_add (dialog);
   
   frame = gtk_frame_new (NULL);
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), frame, TRUE, TRUE, 0);
   gtk_widget_show (frame);
   
   box = gtk_hbox_new (FALSE, 5);
   gtk_container_border_width (GTK_CONTAINER (box), 5);
   gtk_container_add (GTK_CONTAINER (frame), box);
   gtk_widget_show (box);
   
   tempwid = gtk_label_new (_("Domain"));
   gtk_box_pack_start (GTK_BOX (box), tempwid, TRUE, TRUE, 0);
   gtk_widget_show (tempwid);
 
   new_proxy_domain = gtk_entry_new ();
   gtk_box_pack_start (GTK_BOX (box), new_proxy_domain, TRUE, TRUE, 0);
   gtk_widget_show (new_proxy_domain);

   frame = gtk_frame_new (NULL);
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), frame, TRUE, TRUE, 0);
   gtk_widget_show (frame);
   
   table = gtk_table_new (4, 2, FALSE);
   gtk_container_border_width (GTK_CONTAINER (table), 5);
   gtk_table_set_row_spacings (GTK_TABLE (table), 5);
   gtk_table_set_col_spacings (GTK_TABLE (table), 5);
   gtk_container_add (GTK_CONTAINER (frame), table);
   gtk_widget_show (table);
   
   tempwid = gtk_label_new (_("Network Address"));
   gtk_misc_set_alignment (GTK_MISC (tempwid), 1, 0.5);
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 0, 1, 0, 1);
   gtk_widget_show (tempwid);
 
   box = gtk_hbox_new (FALSE, 5);
   gtk_table_attach_defaults (GTK_TABLE (table), box, 1, 2, 0, 1);
   gtk_widget_show (box);

   network1 = gtk_entry_new ();
   gtk_widget_set_usize (network1, 28, -1);
   gtk_box_pack_start (GTK_BOX (box), network1, TRUE, TRUE, 0);
   gtk_widget_show (network1);

   network2 = gtk_entry_new ();
   gtk_widget_set_usize (network2, 28, -1);
   gtk_box_pack_start (GTK_BOX (box), network2, TRUE, TRUE, 0);
   gtk_widget_show (network2);

   network3 = gtk_entry_new ();
   gtk_widget_set_usize (network3, 28, -1);
   gtk_box_pack_start (GTK_BOX (box), network3, TRUE, TRUE, 0);
   gtk_widget_show (network3);

   network4 = gtk_entry_new ();
   gtk_widget_set_usize (network4, 28, -1);
   gtk_box_pack_start (GTK_BOX (box), network4, TRUE, TRUE, 0);
   gtk_widget_show (network4);

   tempwid = gtk_label_new (_("Netmask"));
   gtk_misc_set_alignment (GTK_MISC (tempwid), 1, 0.5);
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 0, 1, 1, 2);
   gtk_widget_show (tempwid);

   box = gtk_hbox_new (FALSE, 5);
   gtk_table_attach_defaults (GTK_TABLE (table), box, 1, 2, 1, 2);
   gtk_widget_show (box);

   netmask1 = gtk_entry_new ();
   gtk_widget_set_usize (netmask1, 28, -1);
   gtk_box_pack_start (GTK_BOX (box), netmask1, TRUE, TRUE, 0);
   gtk_widget_show (netmask1);

   netmask2 = gtk_entry_new ();
   gtk_widget_set_usize (netmask2, 28, -1);
   gtk_box_pack_start (GTK_BOX (box), netmask2, TRUE, TRUE, 0);
   gtk_widget_show (netmask2);

   netmask3 = gtk_entry_new ();
   gtk_widget_set_usize (netmask3, 28, -1);
   gtk_box_pack_start (GTK_BOX (box), netmask3, TRUE, TRUE, 0);
   gtk_widget_show (netmask3);

   netmask4 = gtk_entry_new ();
   gtk_widget_set_usize (netmask4, 28, -1);
   gtk_box_pack_start (GTK_BOX (box), netmask4, TRUE, TRUE, 0);
   gtk_widget_show (netmask4);

   box = gtk_hbox_new (FALSE, 5);
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), box, TRUE, TRUE, 0);
   gtk_widget_show (box);

   domain_active = gtk_radio_button_new_with_label (NULL, _("Domain"));
   gtk_signal_connect (GTK_OBJECT (domain_active), "toggled", GTK_SIGNAL_FUNC (add_toggle), (gpointer) 1);
   gtk_box_pack_start (GTK_BOX (box), domain_active, TRUE, TRUE, 0);
   gtk_widget_show (domain_active);
   
   tempwid = gtk_radio_button_new_with_label (
   	gtk_radio_button_group (GTK_RADIO_BUTTON (domain_active)), _("Network"));
   gtk_signal_connect (GTK_OBJECT (tempwid), "toggled", GTK_SIGNAL_FUNC (add_toggle), NULL);
   gtk_box_pack_start (GTK_BOX (box), tempwid, TRUE, TRUE, 0);
   gtk_widget_show (tempwid);

   if (!hosts || !hosts->domain) {
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (tempwid), TRUE);
      add_toggle (NULL, NULL);
   }
   else {
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (domain_active), TRUE);
      add_toggle (NULL, (gpointer) 1);
   }
   
   if (hosts) {
      if (hosts->domain) {
         gtk_entry_set_text (GTK_ENTRY (new_proxy_domain), hosts->domain);
      }
      else {
         tempstr = g_strdup_printf ("%d", hosts->network_address >> 24 & 0xff);
         gtk_entry_set_text (GTK_ENTRY (network1), tempstr);
         g_free (tempstr);

         tempstr = g_strdup_printf ("%d", hosts->network_address >> 16 & 0xff);
         gtk_entry_set_text (GTK_ENTRY (network2), tempstr);
         g_free (tempstr);

         tempstr = g_strdup_printf ("%d", hosts->network_address >> 8 & 0xff);
         gtk_entry_set_text (GTK_ENTRY (network3), tempstr);
         g_free (tempstr);

         tempstr = g_strdup_printf ("%d", hosts->network_address & 0xff);
         gtk_entry_set_text (GTK_ENTRY (network4), tempstr);
         g_free (tempstr);

         tempstr = g_strdup_printf ("%d", hosts->netmask >> 24 & 0xff);
         gtk_entry_set_text (GTK_ENTRY (netmask1), tempstr);
         g_free (tempstr);

         tempstr = g_strdup_printf ("%d", hosts->netmask >> 16 & 0xff);
         gtk_entry_set_text (GTK_ENTRY (netmask2), tempstr);
         g_free (tempstr);

         tempstr = g_strdup_printf ("%d", hosts->netmask >> 8 & 0xff);
         gtk_entry_set_text (GTK_ENTRY (netmask3), tempstr);
         g_free (tempstr);

         tempstr = g_strdup_printf ("%d", hosts->netmask & 0xff);
         gtk_entry_set_text (GTK_ENTRY (netmask4), tempstr);
         g_free (tempstr);
      }
   }
   
   tempwid = gtk_button_new_with_label (_("OK"));
   GTK_WIDGET_SET_FLAGS (tempwid, GTK_CAN_DEFAULT);
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->action_area), tempwid, TRUE, TRUE, 0);
   gtk_signal_connect (GTK_OBJECT (tempwid), "clicked", GTK_SIGNAL_FUNC (add_ok), (gpointer) templist);
   gtk_signal_connect_object (GTK_OBJECT (tempwid), "clicked", GTK_SIGNAL_FUNC (gtk_widget_destroy), GTK_OBJECT (dialog));
   gtk_widget_show (tempwid);

   tempwid = gtk_button_new_with_label (_("  Cancel  "));
   GTK_WIDGET_SET_FLAGS (tempwid, GTK_CAN_DEFAULT);
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->action_area), tempwid, TRUE, TRUE, 0);
   gtk_signal_connect_object (GTK_OBJECT (tempwid), "clicked", GTK_SIGNAL_FUNC (gtk_widget_destroy), GTK_OBJECT (dialog));
   gtk_widget_show (tempwid);

   gtk_widget_show (dialog);
}
/*****************************************************************************/
static void add_toggle (GtkWidget *widget, gpointer data) {
   gtk_widget_set_sensitive (new_proxy_domain, data != NULL);
   gtk_widget_set_sensitive (network1, data == NULL);
   gtk_widget_set_sensitive (network2, data == NULL);
   gtk_widget_set_sensitive (network3, data == NULL);
   gtk_widget_set_sensitive (network4, data == NULL);
   gtk_widget_set_sensitive (netmask1, data == NULL);
   gtk_widget_set_sensitive (netmask2, data == NULL);
   gtk_widget_set_sensitive (netmask3, data == NULL);
   gtk_widget_set_sensitive (netmask4, data == NULL);
}
/*****************************************************************************/
static void add_ok (GtkWidget *widget, gpointer data) {
   gftp_proxy_hosts *hosts;
   GList *templist;
   char *edttxt;
   int num;
   
   if ((templist = data) == NULL) {
      hosts = g_malloc0 (sizeof (gftp_proxy_hosts));
      new_proxy_hosts = g_list_append (new_proxy_hosts, hosts);
      for (templist = new_proxy_hosts; templist->next != NULL; templist = templist->next);
   }
   else {
      num = gtk_clist_find_row_from_data (GTK_CLIST (proxy_list), templist);
      if (num != -1) {
         gtk_clist_remove (GTK_CLIST (proxy_list), num);
      }
      hosts = templist->data;
   }
   
   if (hosts->domain) {
      g_free (hosts->domain);
      hosts->domain = NULL;
   }
   hosts->netmask = hosts->network_address = 0;
   
   if (GTK_TOGGLE_BUTTON (domain_active)->active) {
      edttxt = gtk_entry_get_text (GTK_ENTRY (new_proxy_domain));
      hosts->domain = g_malloc (strlen (edttxt) + 1);
      strcpy (hosts->domain, edttxt);
   }
   else {
      edttxt = gtk_entry_get_text (GTK_ENTRY (network1));
      hosts->network_address = (strtol (edttxt, NULL, 10) & 0xff) << 24;

      edttxt = gtk_entry_get_text (GTK_ENTRY (network2));
      hosts->network_address |= (strtol (edttxt, NULL, 10) & 0xff) << 16;

      edttxt = gtk_entry_get_text (GTK_ENTRY (network3));
      hosts->network_address |= (strtol (edttxt, NULL, 10) & 0xff) << 8;

      edttxt = gtk_entry_get_text (GTK_ENTRY (network4));
      hosts->network_address |= strtol (edttxt, NULL, 10) & 0xff;

      edttxt = gtk_entry_get_text (GTK_ENTRY (netmask1));
      hosts->netmask = (strtol (edttxt, NULL, 10) & 0xff) << 24;

      edttxt = gtk_entry_get_text (GTK_ENTRY (netmask2));
      hosts->netmask |= (strtol (edttxt, NULL, 10) & 0xff) << 16;

      edttxt = gtk_entry_get_text (GTK_ENTRY (netmask3));
      hosts->netmask |= (strtol (edttxt, NULL, 10) & 0xff) << 8;

      edttxt = gtk_entry_get_text (GTK_ENTRY (netmask4));
      hosts->netmask |= strtol (edttxt, NULL, 10) & 0xff;
   }
   add_host_to_listbox (templist);
}
/*****************************************************************************/
static void delete_proxy_host (GtkWidget *widget, gpointer data) {
   GList *templist;
   int num;

   if ((templist = GTK_CLIST (proxy_list)->selection) == NULL) return;
   num = (int) templist->data;
   templist = gtk_clist_get_row_data (GTK_CLIST (proxy_list), num);
   new_proxy_hosts = g_list_remove_link (new_proxy_hosts, templist);
   gtk_clist_remove (GTK_CLIST (proxy_list), num);
}
/*****************************************************************************/
static void proxy_toggle(GtkList *list, GtkWidget *child, gpointer data) {
   gtk_text_set_point (GTK_TEXT (proxy_text), 0);
   gtk_text_forward_delete (GTK_TEXT (proxy_text), gtk_text_get_length (GTK_TEXT (proxy_text)));

   proxy_num = gtk_list_child_position (list, child);
   if (proxy_num == CUSTOM_PROXY_NUM) {
      gtk_text_insert (GTK_TEXT (proxy_text), NULL, NULL, NULL,
      	custom_proxy, strlen (custom_proxy));
   }
   else {
      gtk_text_insert (GTK_TEXT (proxy_text), NULL, NULL, NULL, 
      	proxy_type[proxy_num].desc, strlen (proxy_type[proxy_num].desc));
   }
}
/*****************************************************************************/
static void apply_changes (GtkWidget *widget, gpointer data) {
   GList *templist;
   char *tempstr;
   int num;
   
   num = 0;
   while (config_file_vars[num].var != NULL) {
      if (config_file_vars[num].widget != NULL) {
         if(config_file_vars[num].type == 'I') {
            *(int *) config_file_vars[num].var = GTK_TOGGLE_BUTTON (config_file_vars[num].widget)->active;
         }
         else {
            tempstr = gtk_entry_get_text (GTK_ENTRY (config_file_vars[num].widget));
            if (config_file_vars[num].type == 'A') {
               *(int *) config_file_vars[num].var = strtol (tempstr, NULL, 10);
            }
            else if (config_file_vars[num].type == 'F') {
               *(float *) config_file_vars[num].var = strtod (tempstr, NULL);
            }
            else {
               g_free (*(char **) config_file_vars[num].var);
               *(char **) config_file_vars[num].var = g_malloc (strlen (tempstr) + 1);
               strcpy (*(char **) config_file_vars[num].var, tempstr);
            }
         }
      }
      num++;
   }
   
   templist = proxy_hosts;
   proxy_hosts = new_proxy_hosts;
   new_proxy_hosts = templist;
   clean_old_changes (NULL, NULL);
   proxy_config = get_proxy_config ();
   GFTP_SET_TRANSFER_TYPE (window2.hdata->ftpdata,
   	passive_transfer ? gftp_transfer_passive : gftp_transfer_active);
   window2.hdata->ftpdata->resolve_symlinks = resolve_symlinks;
   write_config_file ();
}
/*****************************************************************************/
static void clean_old_changes (GtkWidget *widget, gpointer data) {
   gftp_proxy_hosts *hosts;
   GList *templist;
   
   templist = new_proxy_hosts;
   while (templist != NULL) {
      hosts = templist->data;
      if (hosts->domain) g_free (hosts->domain);
      g_free (hosts);
   }
   g_list_free (new_proxy_hosts);
   new_proxy_hosts = NULL;
}
/*****************************************************************************/
static char *get_proxy_config (void) {
   char *newstr, *oldstr, *pos, *endpos, tempchar, tmp[128];
   unsigned int len;
   
   newstr = g_malloc (1);
   *newstr = '\0';
   
   /*
     GTK_TEXT uses wchar_t instead of char in environment of multibyte encoding
     locale (ex Japanese),  so we must convert from wide character 
     to multibyte charator....   Yasuyuki Furukawa (yasu@on.cs.keio.ac.jp)
   */
   if (GTK_TEXT (proxy_text)->use_wchar) {
      wcstombs (tmp, (wchar_t*)GTK_TEXT (proxy_text)->text.wc, sizeof (tmp));
      pos = tmp;
   } 
   else pos = GTK_TEXT (proxy_text)->text.ch;
   
   len = gtk_text_get_length (GTK_TEXT (proxy_text));
   tempchar = pos[len];
   pos[len] = '\0';
   do {
      if ((endpos = strchr (pos, '\n')) != NULL) {
         if (*(endpos - 1) == '\r') *(endpos - 1) = '\0';
         *endpos = '\0';
      }
      oldstr = newstr;
      if (endpos != NULL) {
         newstr = g_strconcat (newstr, pos, "%n", NULL);
      }
      else {
         newstr = g_strconcat (newstr, pos, NULL);
      }
      g_free (oldstr);
      if (endpos != NULL) {
         *endpos = '\n';
         pos = endpos + 1;
      }
   } while (endpos != NULL);
   
   if (GTK_TEXT (proxy_text)->use_wchar) {
     wchar_t* wpos = (wchar_t*) GTK_TEXT (proxy_text)->text.wc;
     wchar_t tempwchar;
     mbtowc (&tempwchar, &tempchar, sizeof(tempwchar));
     wpos[len] = tempwchar;
   }
   else {
     pos = GTK_TEXT (proxy_text)->text.ch;
     pos[len] = tempchar;
   }
   return (newstr);
}
/*****************************************************************************/
void read_config_file (void) {
   char *tempstr, *temp1str, *curpos, *str, buf[MAXSTR];
   struct conn_categories *newentry;
   struct pix_ext *tempext;
   gftp_proxy_hosts *host;
   unsigned int nums[4];
   FILE *conffile;
   int line, pos;
   
   line = 0;
   hostcat = g_malloc0 (sizeof (struct conn_categories));
   hostcat->isfolder = 1;
   hostcat->path = g_malloc (1);
   *hostcat->path = '\0';
   bookmarks_htable = g_hash_table_new (hash_function, hash_compare);

   if ((tempstr = expand_path (CONFIG_FILE)) == NULL) {
      printf (_("gFTP Error: Bad config file name %s\n"), CONFIG_FILE);
      exit (0);
   }
   if (access (tempstr, F_OK) == -1) {
      temp1str = expand_path (BASE_CONF_DIR);
      if (access (temp1str, F_OK) == -1) {
         if (mkdir (temp1str, 488) != 0) {
            printf (_("gFTP Error: Could not make directory %s: %s\n"), temp1str, g_strerror(errno));
            exit (-1);
         }
      }
      g_free (temp1str);
      
      temp1str = g_strdup_printf ("%s/gftprc", SHARE_DIR);
      if (access (temp1str, F_OK) == -1) {
         printf (_("gFTP Error: Cannot find master config file %s\n"), temp1str);
         printf (_("Did you do a make install?\n"));
         exit (-1);
      }
      copyfile (temp1str, tempstr);
      g_free (temp1str);
   }
   chmod (tempstr, S_IRUSR | S_IWUSR);
   if ((conffile = fopen (tempstr, "r")) == NULL) {
      printf (_("gFTP Error: Cannot open config file %s: %s\n"), CONFIG_FILE, g_strerror (errno));
      exit (0);
   }
   g_free (tempstr);
   
   while (fgets (buf, sizeof (buf), conffile)) {
      if (buf[strlen (buf) - 1] == '\n') buf[strlen (buf) - 1] = '\0';

      line++;
      if (strncmp (buf, "host=", 5) == 0) {
         curpos = buf + 5;
         while (*curpos == '/') curpos++;
         newentry = g_malloc0 (sizeof (struct conn_categories));
         newentry->isfolder = 0;
         parse_args (curpos, 8, line, &newentry->path, &newentry->hostname,
         	&temp1str, &newentry->remote_dir, &newentry->user,
         	&newentry->pass, &tempstr, &newentry->acct);
         g_free (tempstr);
         newentry->port = strtol (temp1str, NULL, 10);
         newentry->save_password = *newentry->pass != '\0';
         add_to_bookmark (newentry);
      }
      else if (strncmp (buf, "dont_use_proxy=", 15) == 0) {
         curpos = buf + 15;
         host = g_malloc0 (sizeof (gftp_proxy_hosts));
         if ((tempstr = strchr (curpos, '/')) == NULL) {
            host->domain = g_malloc (strlen (curpos) + 1);
            strcpy (host->domain, curpos);
         }
         else {
            *tempstr = '\0';
            sscanf (curpos, "%d.%d.%d.%d", &nums[0], &nums[1], &nums[2], &nums[3]);
            host->network_address = nums[0] << 24 | nums[1] << 16 | nums[2] << 8 | nums[3];

            if (strchr (tempstr + 1, '.') == NULL) {
               host->netmask = 0xffffffff << (32 - strtol (tempstr + 1, NULL, 10));
            }
            else {
               sscanf (tempstr + 1, "%d.%d.%d.%d", &nums[0], &nums[1], &nums[2], &nums[3]);
               host->netmask = nums[0] << 24 | nums[1] << 16 | nums[2] << 8 | nums[3];
            }
         }
         proxy_hosts = g_list_append (proxy_hosts, host);
      }
      else if (strncmp (buf, "ext=", 4) == 0) {
         curpos = buf + 4;
         tempext = g_malloc (sizeof (struct pix_ext));
         parse_args (curpos, 4, line, &tempext->ext, &tempext->filename,
         	&tempext->ascii_binary, &tempext->view_program);

         if ((tempstr = get_xpm_path (tempext->filename, 1)) != NULL) {
            g_free (tempstr);
         }

         tempext->stlen = strlen (tempext->ext);
         tempext->next = registered_exts;
         registered_exts = tempext;
      }         
      else if (strncmp (buf, "localhistory=", 13) == 0) {
         curpos = buf + 13;
         str = g_malloc (strlen (curpos) + 1);
         strcpy (str, curpos);
         window1.history = g_list_append (window1.history, str);
         window1.histlen++;
      }
      else if (strncmp (buf, "remotehistory=", 14) == 0) {
         curpos = buf + 14;
         str = g_malloc (strlen (curpos) + 1);
         strcpy (str, curpos);
         window2.history = g_list_append (window2.history, str);
         window2.histlen++;
      }
      else if (strncmp (buf, "hosthistory=", 12) == 0) {
         curpos = buf + 12;
         str = g_malloc (strlen (curpos) + 1);
         strcpy (str, curpos);
         host_history = g_list_append (host_history, str);
         host_len++;
      }
      else if (strncmp (buf, "porthistory=", 12) == 0) {
         curpos = buf + 12;
         str = g_malloc (strlen (curpos) + 1);
         strcpy (str, curpos);
         port_history = g_list_append (port_history, str);
         port_len++;
      }
      else if (strncmp (buf, "userhistory=", 12) == 0) {
         curpos = buf + 12;
         str = g_malloc (strlen (curpos) + 1);
         strcpy (str, curpos);
         user_history = g_list_append (user_history, str);
         user_len++;
      }
      else if (*buf != '#' && *buf != '\0') {
         pos = 0;
         while (config_file_vars[pos].var != NULL) {
            if (strncmp (config_file_vars[pos].key, buf, strlen (config_file_vars[pos].key)) == 0) {
               curpos = buf + strlen (config_file_vars[pos].key) + 1;
               if (config_file_vars[pos].type == 'C') {
                  *(char **) config_file_vars[pos].var = g_malloc (strlen (curpos) + 1);
                  strcpy (*(char **) config_file_vars[pos].var, curpos);
                  break;
               }
               else if (config_file_vars[pos].type == 'F') {
                  *(float *) config_file_vars[pos].var = strtod (curpos, NULL);
                  break;
               }
               else {
                  *(int *) config_file_vars[pos].var = strtol (curpos, NULL, 10);
                  break;
               }
            }
            pos++;
         }

         if (config_file_vars[pos].var == NULL) {
            printf (_("gFTP Warning: Skipping line %d in config file: %s\n"), line, buf);
         }
      }
   }
   make_nonnull (&proxy_config);
   make_nonnull (&firewall_host);
   make_nonnull (&firewall_username);
   make_nonnull (&firewall_password);
   make_nonnull (&firewall_account);
   make_nonnull (&http_proxy_host);
   make_nonnull (&http_proxy_username);
   make_nonnull (&http_proxy_password);
   make_nonnull (&view_program);
   make_nonnull (&edit_program);
   
   read_bookmarks ();
   return;
}
/*****************************************************************************/
void read_bookmarks (void) {
   char *tempstr, *temp1str, buf[MAXSTR], *curpos;
   struct conn_categories *newentry;
   FILE *bmfile;
   int line;
   
   if ((tempstr = expand_path (BOOKMARKS_FILE)) == NULL) {
      printf (_("gFTP Error: Bad bookmarks file name %s\n"), BOOKMARKS_FILE);
      exit (0);
   }
   if (access (tempstr, F_OK) == -1) {
      temp1str = expand_path (BASE_CONF_DIR);
      if (access (temp1str, F_OK) == -1) {
         if (mkdir (temp1str, 488) != 0) {
            printf (_("gFTP Error: Could not make directory %s: %s\n"), temp1str, g_strerror(errno));
            exit (-1);
         }
      }
      g_free (temp1str);
      
      temp1str = g_strdup_printf ("%s/bookmarks", SHARE_DIR);
      if (access (temp1str, F_OK) == -1) {
         printf (_("Warning: Cannot find master bookmark file %s\n"), temp1str);
         return;
      }
      copyfile (temp1str, tempstr);
      g_free (temp1str);
   }
   chmod (tempstr, S_IRUSR | S_IWUSR);
   if ((bmfile = fopen (tempstr, "r")) == NULL) {
      printf (_("gFTP Error: Cannot open config file %s: %s\n"), CONFIG_FILE, g_strerror (errno));
      exit (0);
   }
   g_free (tempstr);
   
   line = 0;
   newentry = NULL;
   while (fgets (buf, sizeof (buf), bmfile)) {
      line++;
      if (buf[strlen (buf) - 1] == '\n') buf[strlen (buf) - 1] = '\0';

      line++;
      if (*buf == '[') {
         newentry = g_malloc0 (sizeof (struct conn_categories));
         for (;buf[strlen (buf) - 1] == ' '; buf[strlen (buf) - 1] = '\0');
         buf[strlen (buf) - 1] = '\0';
         newentry->path = g_malloc (strlen (buf));
         strcpy (newentry->path, buf + 1);
         newentry->isfolder = 0;
         add_to_bookmark (newentry);
      }
      else if (strncmp (buf, "hostname", 8) == 0 && newentry) {
         curpos = buf + 9;
         if (newentry->hostname) g_free (newentry->hostname);
         newentry->hostname = g_malloc (strlen (curpos) + 1);
         strcpy (newentry->hostname, curpos);
      }
      else if (strncmp (buf, "port", 4) == 0 && newentry) {
         newentry->port = strtol (buf + 5, NULL, 10);
      }
      else if (strncmp (buf, "protocol", 8) == 0 && newentry) {
         curpos = buf + 9;
         if (newentry->protocol) g_free (newentry->protocol);
         newentry->protocol = g_malloc (strlen (curpos) + 1);
         strcpy (newentry->protocol, curpos);
      }
      else if (strncmp (buf, "remote directory", 16) == 0 && newentry) {
         curpos = buf + 17;
         if (newentry->remote_dir) g_free (newentry->remote_dir);
         newentry->remote_dir = g_malloc (strlen (curpos) + 1);
         strcpy (newentry->remote_dir, curpos);
      }
      else if (strncmp (buf, "local directory", 15) == 0 && newentry) {
         curpos = buf + 16;
         if (newentry->local_dir) g_free (newentry->local_dir);
         newentry->local_dir = g_malloc (strlen (curpos) + 1);
         strcpy (newentry->local_dir, curpos);
      }
      else if (strncmp (buf, "username", 8) == 0 && newentry) {
         curpos = buf + 9;
         if (newentry->user) g_free (newentry->user);
         newentry->user = g_malloc (strlen (curpos) + 1);
         strcpy (newentry->user, curpos);
      }
      else if (strncmp (buf, "password", 8) == 0 && newentry) {
         curpos = buf + 9;
         if (newentry->pass) g_free (newentry->pass);
         newentry->pass = g_malloc (strlen (curpos) + 1);
         strcpy (newentry->pass, curpos);
         newentry->save_password = *newentry->pass != '\0';
      }
      else if (strncmp (buf, "account", 7) == 0 && newentry) {
         curpos = buf + 8;
         if (newentry->acct) g_free (newentry->acct);
         newentry->acct = g_malloc (strlen (curpos) + 1);
         strcpy (newentry->acct, curpos);
      }
      else if (*buf != '#' && *buf != '\0') {
         printf (_("gFTP Warning: Skipping line %d in bookmarks file: %s\n"), line, buf);
      }
   }
}
/*****************************************************************************/
static void add_to_bookmark (struct conn_categories *newentry) {
   struct conn_categories *preventry, *folderentry, *endentry;
   char *curpos;

   if (!newentry->protocol) {
      newentry->protocol = g_malloc (4);
      strcpy (newentry->protocol, "FTP");
   }
   
   /* We have to create the folders. For example, if we have 
      Debian Sites/Debian, we have to create a Debian Sites entry */
   preventry = hostcat;
   if (preventry->children != NULL) { 
      endentry = preventry->children;
      while (endentry->next != NULL) endentry = endentry->next;
   }
   else endentry = NULL;
   curpos = newentry->path;
   while ((curpos = strchr (curpos, '/')) != NULL) {
      *curpos = '\0';
      /* See if we already made this folder */
      if ((folderentry = (struct conn_categories *) 
      	g_hash_table_lookup (bookmarks_htable, newentry->path)) == NULL) {

         /* Allocate the individual folder. We have to do this for the edit 
            bookmarks feature */
         folderentry = g_malloc0 (sizeof (struct conn_categories));
         folderentry->path = g_malloc (strlen (newentry->path) + 1);
         strcpy (folderentry->path, newentry->path);
         folderentry->prev = preventry;
         folderentry->isfolder = 1;
         g_hash_table_insert (bookmarks_htable, folderentry->path, folderentry);
         if (preventry->children == NULL) preventry->children = folderentry;
         else endentry->next = folderentry;
         preventry = folderentry;
         endentry = NULL;
      }
      else {
         preventry = folderentry;
         if (preventry->children != NULL) {
            endentry = preventry->children;
            while (endentry->next != NULL) endentry = endentry->next;
         }
         else endentry = NULL;
      }
      *curpos = '/';
      curpos++;
   }

   /* Get the parent node */
   if ((curpos = strrchr (newentry->path, '/')) == NULL) preventry = hostcat;
   else {
      *curpos = '\0';
      preventry = (struct conn_categories *) 
      	g_hash_table_lookup (bookmarks_htable, newentry->path);
      *curpos = '/';
   }

   if (preventry->children != NULL) {
      endentry = preventry->children;
      while (endentry->next != NULL) endentry = endentry->next;
      endentry->next = newentry;
   }
   else preventry->children = newentry;
   newentry->prev = preventry;
   g_hash_table_insert (bookmarks_htable, newentry->path, newentry);
}
/*****************************************************************************/
void write_config_file (void) {
   char *hdr, *proxyhdr, *exthdr, *histhdr, *bmhdr;
   struct conn_categories *tempentry;
   gftp_proxy_hosts *hosts;
   FILE *conffile, *bmfile;
   struct pix_ext *tempext;
   char *tempstr, *str;
   GList *tempnode;
   int pos;

   hdr = N_("# Config file for gFTP\n# Copyright (C) 1998-1999 Brian Masney <masneyb@seul.org>\n# Warning: Any comments that you add to this file WILL be overwritten\n# If a entry has a (*) in it's comment, you can't change it inside gFTP\n\n");
   bmhdr = N_("# Bookmarks file for gFTP\n# Copyright (C) 1998-1999 Brian Masney <masneyb@seul.org>\n# Warning: Any comments that you add to this file WILL be overwritten\n\n");
   proxyhdr = N_("This section specifies which hosts are on the local subnet and won't need to go out the proxy server (if available). Syntax: dont_use_proxy=.domain or dont_use_proxy=network number/netmask");
   exthdr = N_("ext=file extenstion:XPM file:Ascii or Binary (A or B):viewer program. Note: All arguments except the file extension are optional");
   histhdr = N_("This section contains the data that is in the history");
   if ((tempstr = expand_path (CONFIG_FILE)) == NULL) {
      printf (_("gFTP Error: Bad config file name %s\n"), CONFIG_FILE);
      exit (0);
   }
   if ((conffile = fopen (tempstr, "w+")) == NULL) {
      printf (_("gFTP Error: Cannot open config file %s: %s\n"), CONFIG_FILE, g_strerror (errno));
      exit (0);
   }
   g_free (tempstr);

   fwrite (_(hdr), 1, strlen (_(hdr)), conffile);

   if ((tempstr = expand_path (BOOKMARKS_FILE)) == NULL) {
      printf (_("gFTP Error: Bad bookmarks file name %s\n"), CONFIG_FILE);
      exit (0);
   }
   if ((bmfile = fopen (tempstr, "w+")) == NULL) {
      printf (_("gFTP Error: Cannot open bookmarks file %s: %s\n"), CONFIG_FILE, g_strerror (errno));
      exit (0);
   }
   g_free (tempstr);

   fwrite (_(bmhdr), 1, strlen (_(bmhdr)), bmfile);

   pos = 0;
   while (config_file_vars[pos].var != NULL) {
      if (*config_file_vars[pos].comment != '\0') {
         write_comment (conffile, _(config_file_vars[pos].comment));
      }
      
      if (config_file_vars[pos].type == 'C') {
         fprintf (conffile, "%s=%s\n\n", config_file_vars[pos].key, *(char **) config_file_vars[pos].var);
      }
      else if (config_file_vars[pos].type == 'F') {
         fprintf (conffile, "%s=%.2f\n\n", config_file_vars[pos].key, *(float *) config_file_vars[pos].var);
      }
      else {
         fprintf (conffile, "%s=%d\n\n", config_file_vars[pos].key, *(int *) config_file_vars[pos].var);
      }
      pos++;
   }

   tempentry = hostcat->children;
   while (tempentry != NULL) {
      if (tempentry->children != NULL) {
         tempentry = tempentry->children;
         continue;
      }
      str = tempentry->path;
      while (*str == '/') str++;
      fprintf (bmfile, "[%s]\nhostname=%s\nport=%d\nprotocol=%s\nremote directory=%s\nlocal directory=%s\nusername=%s\npassword=%s\naccount=%s\n\n",
      	str, tempentry->hostname == NULL ? "" : tempentry->hostname, 
      	tempentry->port, 
      	tempentry->protocol == NULL || *tempentry->protocol == '\0' ? gftp_protocols[0].name : tempentry->protocol,
      	tempentry->remote_dir == NULL ? "" : tempentry->remote_dir, 
      	tempentry->local_dir == NULL ? "" : tempentry->local_dir,
      	tempentry->user == NULL ? "" : tempentry->user,
      	!tempentry->save_password || tempentry->pass == NULL ? "" : tempentry->pass, 
      	tempentry->acct == NULL ? "" : tempentry->acct);
      
      if (tempentry->next == NULL) {
         tempentry = tempentry->prev;
         while (tempentry->next == NULL && tempentry->prev != NULL) tempentry = tempentry->prev;
         tempentry = tempentry->next;
      }
      else tempentry = tempentry->next;
   }

   write_comment (conffile, _(proxyhdr));
   tempnode = proxy_hosts;
   while (tempnode != NULL) {
      hosts = tempnode->data;
      if (hosts->domain) {
         fprintf (conffile, "dont_use_proxy=%s\n", hosts->domain);
      }
      else {
         fprintf (conffile, "dont_use_proxy=%d.%d.%d.%d/%d.%d.%d.%d\n", 
         	hosts->network_address >> 24 & 0xff,
         	hosts->network_address >> 16 & 0xff,
         	hosts->network_address >> 8 & 0xff,
         	hosts->network_address & 0xff,
         	hosts->netmask >> 24 & 0xff,
         	hosts->netmask >> 16 & 0xff,
         	hosts->netmask >> 8 & 0xff,
         	hosts->netmask & 0xff);
      }
      tempnode = tempnode->next;
   }
   
   fwrite ("\n", 1, 1, conffile);
   write_comment (conffile, _(exthdr));
   tempext = registered_exts;
   while (tempext != NULL) {
      fprintf (conffile, "ext=%s:%s:%c:%s\n", tempext->ext, tempext->filename, 
      	*tempext->ascii_binary == '\0' ? ' ' : *tempext->ascii_binary, 
      	tempext->view_program);
      tempext = tempext->next;
   }

   fwrite ("\n", 1, 1, conffile);
   write_comment (conffile, _(histhdr));
   tempnode = window1.history;
   while (tempnode != NULL) {
      fprintf (conffile, "localhistory=%s\n", (char *) tempnode->data);
      tempnode = tempnode->next;
   }

   tempnode = window2.history;
   while (tempnode != NULL) {
      fprintf (conffile, "remotehistory=%s\n", (char *) tempnode->data);
      tempnode = tempnode->next;
   }

   tempnode = host_history;
   while (tempnode != NULL) {
      fprintf (conffile, "hosthistory=%s\n", (char *) tempnode->data);
      tempnode = tempnode->next;
   }

   tempnode = port_history;
   while (tempnode != NULL) {
      fprintf (conffile, "porthistory=%s\n", (char *) tempnode->data);
      tempnode = tempnode->next;
   }

   tempnode = user_history;
   while (tempnode != NULL) {
      fprintf (conffile, "userhistory=%s\n", (char *) tempnode->data);
      tempnode = tempnode->next;
   }

   fclose (conffile);
   fclose (bmfile);
}
/*****************************************************************************/
static void write_comment (FILE *fd, const char *comment) {
   const char *pos, *endpos;
   
   fwrite ("# ", 1, 2, fd);
   pos = comment;
   while (strlen (pos) > 76) {
      for (endpos = pos + 76; *endpos != ' ' && endpos > pos; endpos--);
      if (endpos == pos) {
         for (endpos = pos + 76; *endpos != ' ' && *endpos != '\0'; endpos++);
      }
      fwrite (pos, 1, endpos - pos, fd);
      fwrite ("\n# ", 1, 3, fd);
      if (*endpos == '\0') {
         pos = endpos;
         break;
      }
      else pos = endpos + 1;
   }
   if (strlen (pos) > 1) {
      fwrite (pos, 1, strlen (pos), fd);
      fwrite ("\n", 1, 1, fd);
   }
}
/*****************************************************************************/
static int copyfile (char *source, char *dest) {
   FILE *srcfd, *destfd;
   char buf[8192];
   size_t n;
   
   if ((srcfd = fopen (source, "rb")) == NULL) {
      return (0); 
   }
   if ((destfd = fopen (dest, "wb")) == NULL) {
      fclose (srcfd);
      return (0);
   }

   while ((n = fread (buf, 1, sizeof (buf), srcfd)) > 0) {
      fwrite (buf, 1, n, destfd);
   }

   fclose (srcfd);
   fclose (destfd);
   return (1);
}
/*****************************************************************************/
int parse_args (char *str, int numargs, int lineno, char **first, ...) {
   char *curpos, *endpos, *pos, **dest, tempchar;
   int ret, has_colon;
   va_list argp;
   
   ret = 1;
   va_start (argp, first);
   curpos = str;
   dest = first;
   *dest = NULL;
   while (numargs > 0) {
      has_colon = 0;
      if (numargs > 1) {
         if ((endpos = strchr (curpos, ':')) == NULL) {
            printf (_("gFTP Warning: Line %d doesn't have enough arguments\n"), lineno);
            ret = 0;
            endpos = curpos + strlen (curpos);
         }
         else {
            /* Allow colons inside the fields. If you want a colon inside a field,
               just put 2 colons in there */
            while (endpos != NULL && *(endpos - 1) == '\\') {
               endpos = strchr (endpos + 1, ':');
               has_colon = 1;
            }
         }
      }
      else endpos = curpos + strlen (curpos);

      *dest = g_malloc (endpos - curpos + 1);
      tempchar = *endpos;
      *endpos = '\0';
      strcpy (*dest, curpos);
      *endpos = tempchar;
      if (has_colon) {
         pos = *dest;
         curpos = *dest;
         while (*pos != '\0') {
            if (*pos != '\\' && *(pos+1) != ':') *curpos++ = *pos++;
            else pos++;
         }
         *curpos = '\0';
      }
      if (*endpos == '\0') break;
      curpos = endpos + 1;
      if (numargs > 1) {
         dest = va_arg (argp, char **);
         *dest = NULL;
      }
      numargs--;
   }

   while (numargs > 1) {
      dest = va_arg (argp, char **);
      *dest = g_malloc (1);
      **dest = '\0';
      numargs--;
   }
   va_end (argp);
   return (1);
}
/*****************************************************************************/
GHashTable *build_hash_table (struct conn_categories *entry) {
   struct conn_categories *tempentry;
   GHashTable *htable;
   
   htable = g_hash_table_new (hash_function, hash_compare);
   tempentry = entry;
   while (tempentry != NULL) {
      g_hash_table_insert (htable, tempentry->path, tempentry);
      if (tempentry->children != NULL) {
         tempentry = tempentry->children;
         continue;
      }
      while (tempentry->next == NULL && tempentry->prev != NULL) tempentry = tempentry->prev;
      tempentry = tempentry->next;
   }
   return (htable);
}
/*****************************************************************************/
static gint hash_compare (gconstpointer path1, gconstpointer path2) {
   return (strcmp ((char *) path1, (char *) path2) == 0);
}
/*****************************************************************************/
static guint hash_function (gconstpointer key) {
   return (((char *) key)[0] + ((char *) key)[1]);
}
/*****************************************************************************/
