/*
   CIPE - encrypted IP over UDP tunneling

   ciped.c - the user mode driver

   Copyright 1996 Olaf Titz <olaf@bigred.inka.de>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version
   2 of the License, or (at your option) any later version.
*/
/* $Id: ciped.c,v 1.22.2.3 1999/06/07 19:08:24 olaf Exp $ */

#define CTLDIR "/etc/cipe"

#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <netdb.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#define __USE_GNU
#include <string.h>
#include <syslog.h>
#include <unistd.h>
#include <arpa/inet.h>
#include <net/if.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <values.h>
#include "ciped.h"
#include "ioctl.h"

#ifndef LOGFAC
#define LOGFAC LOG_DAEMON
#endif

static const char daemon_version[]=VERSION;

void usage(const char *p)
{
    fprintf(stderr, "usage: %s [-o file ] [options] device=name\n", p);
    exit(1);
}

/* FIXME: places marked *#* have byteorder conversion missing */
/* This is corrected in version 3 protocol */

/*** The global options structure ***/

struct options {
    const char * const oname;
    const char otyp;
    union {
	int ovint;
	char *ovstr;
	struct sockaddr_in *ovsaddr;
    } ov;
} opts[] = {
#define Opt(x,t) { #x, T##t, {0} },
#define Tbool	0
#define Tint	1
#define Tstr	2
#define Taddr	3
#define Tuaddr	4
#define Ttaddr	5

#define Okey		0  /* The interface secret key
			      - set via options file only */
Opt(key,str)
#define Odebug		1  /* Debug mode - don't detach, verbose messages */
Opt(debug,bool)
#define Oipaddr		2  /* This CIPE device's IP address */
Opt(ipaddr,addr)
#define Optpaddr	3  /* The peer CIPE device's IP address */
Opt(ptpaddr,addr)
#define Omtu		4  /* Device MTU */
Opt(mtu,int)
#define Ometric		5  /* Device metric */
Opt(metric,int)
#define Ome		6  /* My UDP address */
Opt(me,uaddr)
#define Opeer		7  /* The peer's UDP address,
			      i.e. where to send packets */
Opt(peer,uaddr)
#define Odevice		8  /* name of the CIPE device, e.g. cip3b0 */
Opt(device,str)
#define Onokey		9  /* Don't encrypt */
Opt(nokey,bool)
#define Osocks		10  /* SOCKS5 server (ip:port) */
Opt(socks,taddr)
#define Otokxc		11  /* Timeout for key exchange (secs) */
Opt(tokxc,int)
#define Otokey		12  /* Dynamic key lifetime (secs) */
Opt(tokey,int)
#define Oipup		13  /* Name of the ip-up script */
Opt(ipup,str)
#define Oipdown		14  /* Name of the ip-down script */
Opt(ipdown,str)
#define Oarg		15  /* Extra argument to ip-up script */
Opt(arg,str)
#define Octtl		16  /* TTL value for carrier */
Opt(cttl,int)
#define Omaxerr		17  /* Maximum ECONNREFUSED tolerance */
Opt(maxerr,int)
#define Otokxts         18  /* Timeout for key exchange timestamps */
Opt(tokxts,int)
#define Oping           19  /* Keepalive ping interval */
Opt(ping,int)
#define Otoping         20  /* Keepalive ping timeout */
Opt(toping,int)

#undef Opt
{ NULL, 0, {0} }};

#define OI(x)      (opts[O##x].ov.ovint)
#define OS(x)      (opts[O##x].ov.ovstr)
#define OA(x)      (opts[O##x].ov.ovsaddr)
#define OAaddr(x)  (OA(x)->sin_addr)
#define OAport(x)  (OA(x)->sin_port)

/* Statistics */
struct kstat {
    int rreq, req, ind, indb, ack, ackb, bogus;
} ks = { 0, 0, 0, 0, 0, 0, 0 };

char device[16] = "";
int devnum = -1;

/*** Option parser ***/

/* Resolve a host:port address */
int getaddr(char *c, struct sockaddr_in *sa, const char *prot)
{
    struct hostent *h;
    struct servent *s;
    char *q=strchr(c, ':');

    sa->sin_family=AF_INET;
    sa->sin_port=0;
    sa->sin_addr.s_addr=INADDR_ANY;
    if (q) {
	if (!prot) {
	    fprintf(stderr, "raw IP address has no port number\n");
	    return -1;
	}
	*q++='\0';
	if (!(sa->sin_port=htons(atoi(q)))) {
	    if ((s=getservbyname(q, prot)))
		sa->sin_port=s->s_port;
	    else {
		fprintf(stderr, "port '%s' invalid/not found\n", q);
		return -1;
	    }
	}
    }
    if ((sa->sin_addr.s_addr=inet_addr(c))==-1) {
	if ((h=gethostbyname(c)))
	    memcpy(&sa->sin_addr.s_addr, h->h_addr_list[0],
		   sizeof(sa->sin_addr.s_addr));
	else {
	    fprintf(stderr, "host '%s' invalid/not found\n", c);
	    return -1;
	}
    }
    return 0;
}

/* Set an option */
int setopt(const char *n, char *v)
{
    struct options *o=opts;
    static const char * const protos[]={NULL, "udp", "tcp"};

    while (o->oname) {
	if (!strcmp(o->oname, n)) {
	    if ((!v) && (o->otyp!=Tbool)) {
		fprintf(stderr, "missing value for %s\n", n); return -1;
	    }
	    switch(o->otyp) {
	    case Tbool:
		o->ov.ovint=1; break;
	    case Tint:
		o->ov.ovint=atoi(v); break;
	    case Tstr:
		if (!(o->ov.ovstr=strdup(v))) {
		    fprintf(stderr, "setopt: out of memory\n"); return -1;
		}
		break;
	    case Taddr:
	    case Tuaddr:
	    case Ttaddr:
		if (!(o->ov.ovsaddr=malloc(sizeof(*o->ov.ovsaddr)))) {
		    fprintf(stderr, "setopt: out of memory\n"); return -1;
		}
		if (getaddr(v, o->ov.ovsaddr, protos[o->otyp-Taddr])<0) {
		    return -1;
		}
		break;
	    default:
		fprintf(stderr, "internal error: %d\n", __LINE__); return -1;
	    }
	    return 0;
	}
	++o;
    }
    fprintf(stderr, "unknown option: %s\n", n);
    return -1;
}

/* Set options from command line */
void setopt_cmdline(int argc, char *argv[])
{
    int i;
    char *q;
    for (i=0; i<argc; ++i) {
	if ((q=strchr(argv[i], '=')))
	    *q++='\0';
	if (!strcmp(argv[i], "key")) {
	    fprintf(stderr, "Refused to set key from command line!\n");
	    continue;
	}
	(void) setopt(argv[i], q);
    }
}

/* Set options from file, report file not found if (v) */
void setopt_file(const char *n, int v)
{
    int d;
    FILE *f;
    char b[128];
    char *p, *q;
    struct stat s;

    if ((d=open(n, O_RDONLY))<0) {
	if (v)
	    perror("setopt_file: open");
	return;
    }
    if (fstat(d, &s)<0) {
	perror("setopt_file: stat");
	return;
    }
    if (!S_ISREG(s.st_mode) || (s.st_mode&(S_IRWXG|S_IRWXO)) ||
	(s.st_uid!=0)) {
	fprintf(stderr, "file %s has incorrect permissions\n", n);
	return;
    }

    if (!(f=fdopen(d, "r"))) {
	if (v)
	    perror("setopt_file: fdopen");
	return;
    }
    while (fgets(b, sizeof(b), f)) {
	p=b;
	if ((q=strchr(p, '\n')))
	    *q='\0';
	while ((*p) && strchr(" \t", *p))
	    ++p;
	if ((!*p) || strchr("#;:!", *p))
	    continue;
	if ((q=strpbrk(p, " \t="))) {
	    while ((*q) && strchr(" \t=", *q))
		*q++='\0';
	    if (!*q)
		q=NULL;
	}
	(void) setopt(p, q);
    }
    fclose(f);
}

/* Make a "name=value" string of option */
char *optstr(struct options *o)
{
    static char buf[1024];
    switch(o->otyp) {
    case Tbool:
        snprintf(buf, sizeof(buf), "%s=%s", o->oname,
                 o->ov.ovint?"yes":"no");
        break;
    case Tint:
        snprintf(buf, sizeof(buf), "%s=%d", o->oname, o->ov.ovint);
        break;
    case Tstr:
        snprintf(buf, sizeof(buf), "%s=%s", o->oname,
                 o->ov.ovstr?o->ov.ovstr:"(none)");
        break;
    case Taddr:
        if (o->ov.ovsaddr)
            snprintf(buf, sizeof(buf), "%s=%s", o->oname,
                     inet_ntoa(o->ov.ovsaddr->sin_addr));
        else
            snprintf(buf, sizeof(buf), "%s=", o->oname);
        break;
    case Tuaddr:
    case Ttaddr:
        if (o->ov.ovsaddr)
            snprintf(buf, sizeof(buf), "%s=%s:%d", o->oname,
                     inet_ntoa(o->ov.ovsaddr->sin_addr),
                     ntohs(o->ov.ovsaddr->sin_port));
        else
            snprintf(buf, sizeof(buf), "%s=", o->oname);
        break;
    default: fprintf(stderr, "internal error: %d\n", __LINE__);
    }
    return buf;
}

/* Print out all options */
void dumpopt(void)
{
    struct options *o;
    for (o=opts; o->oname; ++o)
        printf("%s\n", optstr(o));
}

/*** Logging ***/

void logd(int lev, const char *f, int d)
{
    if (OI(debug)) {
	fprintf(stderr, f, d);
        fputc('\n', stderr);
    } else
	syslog(lev, f, d);
}

void logs(int lev, const char *f, const char *s)
{
    if (OI(debug)) {
	fprintf(stderr, f, s);
        fputc('\n', stderr);
    } else
	syslog(lev, f, s);
}

/* perror replacement */
void logerr(int lev, const char *s)
{
    if (OI(debug))
	perror(s);
    else
	syslog(lev, "%s: %s", s, strerror(errno));
}

/* missing option */
void argmiss(const char *s)
{
    logs(LOG_ERR, "missing argument: %s", s);
}

/* print if debugging */
#define dprintf(f)    if(OI(debug)){printf f;}

void dumppacket(const char *hdr, int f, const unsigned char *buf, int len)
{
    int i=999;
    printf("%s %d", hdr, f);
    while (len-->0) {
        if (++i>15) {
            printf("\n    ");
            i=0;
        }
        printf("%02x ", *buf++);
    }
    printf("\n");
}

/*** Initialize, open and attach the CIPE device. ***/

int opendev(void)
{
    int f, i;
#ifndef NO_DYNDEV
    struct siocsifcipall d;
#endif
    struct siocsifcippar p;
    struct siocsifcipkey k;
    struct siocsifcipatt a;
    struct ifreq r;

#define amiss(s) { argmiss(s); goto error; }
#define err(s)   { logerr(LOG_ERR, s); goto error; }

    /* Initialize, bind and connect the socket */
    if ((f=socket(AF_INET, SOCK_DGRAM, 0))<0) {
	logerr(LOG_ERR, "opendev: socket");
	return -1;
    }
    i=65536; /* we need much to receive fast */
    if (setsockopt(f, SOL_SOCKET, SO_RCVBUF, &i, sizeof(i)))
	/* not fatal */
	logerr(LOG_NOTICE, "opendev: setsockopt");
    if (!OA(me)) {
	if (!(OA(me)=malloc(sizeof(*OA(me)))))
	    err("opendev: malloc");
	OA(me)->sin_family=AF_INET;
	OAaddr(me).s_addr=INADDR_ANY;
	OAport(me)=0;
    }
    if (bind(f, (struct sockaddr *)OA(me), sizeof(*OA(me)))<0)
	err("opendev: bind");
    if (OA(peer)) {
	if (connect(f, (struct sockaddr *)OA(peer), sizeof(*OA(peer)))<0)
	    err("opendev: bind");
    } else
	amiss("peer");
    i=sizeof(*OA(me));
    if (getsockname(f, (struct sockaddr *)OA(me), &i)<0)
	/* not fatal */
	logerr(LOG_NOTICE, "opendev: getsockname");

#ifdef NO_DYNDEV
    if (!OS(device))
        amiss("device");
#else
    /* Allocate the device name */
    if (OS(device)) {
        char *x=OS(device)+strlen(OS(device))-1;
        while (x>=OS(device) && isdigit(*x))
            --x;
        d.num=atoi(++x);
    } else {
        d.num=-1;
    }
    if (ioctl_alloc(f, DEVNAME "0", &d)<0)
        err("opendev: alloc");
    strcpy(device, d.name);
    devnum=d.num;
    dprintf(("Using %s index %d\n", device, devnum));
#endif

    /* Set the CIPE operation parameters */
    memset(&p, 0, sizeof(p));
    if (OA(socks)) {
	p.sockshost=OAaddr(socks).s_addr;
	p.socksport=OAport(socks);
    }
    p.tmo_keyxchg=OI(tokxc);
    p.tmo_keylife=OI(tokey);
    p.mayclear=OI(nokey);
    p.cttl=OI(cttl);
    if (ioctl_setpar(f, device, &p)<0)
	err("opendev: setpar");

    /* Set the key */
    if (OS(key)) {
	unsigned char *c=OS(key);
	unsigned char *x=(unsigned char *)&k.thekey;
	i=0;
	k.which=KEY_STATIC;
	/* primitive hex decoder */
	while(*c) {
	    x[i]=((*c)<<4)&0xF0;
	    *c++='\0'; /* clear after use */
	    if (!*c) break;
	    x[i++]|=(*c)&0x0F;
	    *c++='\0';
	    if (i>=userKeySize) break;
	}
	if (ioctl_setkey(f, device, &k)<0)
	    err("opendev: setkey");
	memset(&k, 0, sizeof(k));
    } else {
	if (!OI(nokey))
	    amiss("key");
    }

    /* Perform attach */
    a.fd=f;
    if (ioctl_attach(f, device, &a)<0)
	err("opendev: attach");

    /* Perform ifconfig */
    strcpy(r.ifr_name, device);
    if (OA(ipaddr)) {
	memcpy(&r.ifr_addr, OA(ipaddr), sizeof(*OA(ipaddr)));
	if (ioctl(f, SIOCSIFADDR, &r)<0)
	    err("opendev: SIOCSIFADDR");
    } else
	amiss("ipaddr");
    if (OA(ptpaddr)) {
	memcpy(&r.ifr_dstaddr, OA(ptpaddr), sizeof(*OA(ptpaddr)));
	if (ioctl(f, SIOCSIFDSTADDR, &r)<0)
	    err("opendev: SIOCSIFDSTADDR");
    }
    if (OI(mtu)) {
	r.ifr_mtu=OI(mtu);
	if (ioctl(f, SIOCSIFMTU, &r)<0)
	    err("opendev: SIOCSIFMTU");
    }
    if (OI(metric)) {
	r.ifr_metric=OI(metric);
	if (ioctl(f, SIOCSIFMETRIC, &r)<0)
	    err("opendev: SIOCSIFMETRIC");
    }
    r.ifr_flags=IFF_UP|IFF_RUNNING|IFF_NOARP|IFF_NOTRAILERS|IFF_POINTOPOINT;
    if (ioctl(f, SIOCSIFFLAGS, &r)<0)
	err("opendev: SIOCSIFFLAGS");

    return f;

 error:
    close(f);
    return -1;

#undef amiss
#undef err
}

/* Close the device. */

void closedev(int f)
{
    struct ifreq r;
    strcpy(r.ifr_name, device);
    if (ioctl(f, SIOCGIFFLAGS, &r)<0)
	logerr(LOG_ERR, "closedev: SIOCGIFFLAGS");
    r.ifr_flags&=~(IFF_UP|IFF_RUNNING);
    if (ioctl(f, SIOCSIFFLAGS, &r)<0)
	logerr(LOG_ERR, "closedev: SIOCSIFFLAGS");
    close(f);
#ifndef NO_DYNDEV
    if (devnum>0) {
        struct siocsifcipall x;
        if ((f=socket(AF_INET, SOCK_DGRAM, 0))<0) {
            logerr(LOG_ERR, "closedev: socket");
            return;
        }
        x.num=devnum;
        if (ioctl_unalloc(f, device, &x)<0)
            logerr(LOG_ERR, "closedev: alloc");
        close(f);
    }
#endif
}

/* Fetch the device statistics. */

char *getstats(void)
{
    static char b[256];
    char *p, *q;
    FILE *f=fopen("/proc/net/dev/", "r");
    if (!f)
        return NULL;
    while (fgets(b, sizeof(b), f)) {
        for (p=b; *p==' '; ++p);
        for (q=p; *q && *q!=':'; ++q);
        if (!*q)
            continue;
        *q++='\0';
        if (strcmp(p, device))
            continue;
        fclose(f);
        return(q);
    }
    fclose(f);
    return NULL;
}


/*** Open a SOCKS5 connection ***/

int opensocks(void) {
    struct sockaddr_in a;
    int fd=socks5_open(OA(socks));
    if (fd>=0) {
	/* 3rd par of socks5_cmd is in: my address, out: relayer */
	a=*OA(me);
	fd=socks5_cmd(fd, S5_UDP, &a);
	/* Swap peer and socks addresses.
	   Use the SOCKS relayer as "peer" and the real peer as "socks"
	   parameter. See output.c why. */
	*OA(socks)=*OA(peer);
	*OA(peer)=a;
    }
    return fd;
}

/* Check whether the SOCKS5 server has closed the connection */

const char *eofsocks_str[] = {"no error", "exception", "read error", "EOF"};

int eofsocks(int fd) {
    fd_set ifd, efd;
    struct timeval t={0,0};
    FD_ZERO(&ifd);
    FD_ZERO(&efd);
    FD_SET(fd, &ifd);
    FD_SET(fd, &efd);
    if (select(fd+1, &ifd, NULL, &efd, &t)>0) {
	if (FD_ISSET(fd, &efd)) {
	    return 1;
	}
	if (FD_ISSET(fd, &ifd)) {
	    int e;
	    char c;
	    if ((e=read(fd, &c, 1))<=0) {
		return e<0 ? 2 : 3;
	    }
	}
    }
    return 0;
}

/*** Signal handler ***/

sig_atomic_t gotsig=0;

void sighand(int s)
{
    gotsig=s;
}

void sigignore(int s)
{
    /* Dummy */
}

void setsig(int sig, void (*fun)(int))
{
  struct sigaction sa;
  sa.sa_handler = fun;
  sa.sa_flags = 0;
  sigemptyset(&sa.sa_mask);
  sigaction(sig, &sa, NULL);
}

time_t pingtime=MAXINT, pongtime=MAXINT;

/*** Control message handling ***/

void dprintct(const char *h, char c)
{
    switch(c) {
    case CT_DUMMY: dprintf(("%s CT_DUMMY\n", h));  break;
    case CT_DEBUG: dprintf(("%s CT_DEBUG\n", h));  break;
    case CT_PING:  dprintf(("%s CT_PING\n", h));   break;
    case CT_PONG:  dprintf(("%s CT_PONG\n", h));   break;
    case CT_KILL:  dprintf(("%s CT_KILL\n",h));    break;
    default:       dprintf(("%s %d\n", h, c));
    }
}

/* send a control */
int sendctl(int f, const char *msg, int len)
{
    int e;
    dprintct("sending", msg[0]);
    if (OI(nokey))
        return -1; /* unencrypted cant handle controls */
    if ((e=send(f, msg, len, 0))<0)
        logerr(LOG_ERR, "sendctl");
    return e;
}

/* handle control message */
void ctrl(int f, char *msg, int len)
{
    if (OI(nokey))
        return; /* unencrypted cant handle controls */
    dprintct("received", msg[0]);
    switch (msg[0]) {
    case CT_DUMMY:
        /* ignore it */
        break;
    case CT_DEBUG:
        /* Log it */
        logs(LOG_NOTICE, "peer: %s", msg+1);
        break;
    case CT_PING:
        msg[0]=CT_PONG;
        sendctl(f, msg, len);
        break;
    case CT_PONG:
        pongtime=MAXINT;
        break;
    case CT_KILL:
        logs(LOG_NOTICE, "Shutting down at peer request %s", msg+1);
        gotsig=SIGHUP;
        break;
    default:
        logd(LOG_NOTICE, "unknown control %d", msg[0]);
    }
}

/*** The key exchange procedure ***/

#ifdef VER_CRC32
typedef unsigned long crc;
#define CRCFORM "%08lX"
#else
typedef unsigned short crc;
#define CRCFORM "%04X"
#endif

#define REM 	256	/* How many keys to remember */

/* Cache last valid key CRCs. */

static int dupcheck(crc c)
{
    static crc ca[REM];
    static int caf=0, cal=0;
    int i;

    for (i=cal; i<caf; ++i)
	if (ca[i%REM]==c)
	    return 1;
    if (caf-cal>=REM)
	if (++cal>REM) {
	    caf-=REM;
	    cal-=REM;
	}
    ca[caf%REM]=c;
    ++caf;
    return 0;
}

int errcnt;

int kxchg(int f, int r)
{
    int x=0, e=0;
    unsigned char kx[KEYXCHGBLKMIN+2];
    #define kx_typ       (kx[0])
    #define kx_nkind_key (kx+1)
    #define kx_ts        (*(time_t *)(kx+KEYXCHGTSPOS))
#ifdef VER_CRC32
    #define kx_nkind_crc (*(unsigned long *)(kx+1+userKeySize))
    #define kx_nkack_crc (*(unsigned long *)(kx+1))
#else
    #define kx_nkind_crc (*(unsigned short *)(kx+1+userKeySize))
    #define kx_nkack_crc (*(unsigned short *)(kx+1))
#endif
    static crc scrc=0;
    static unsigned char skey[userKeySize];

/*
    struct msghdr hdr;
    struct iovec iov;

    iov.iov_base=&kx;
    iov.iov_len=sizeof(kx)-1;
    hdr.msg_name=hdr.msg_control=NULL;
    hdr.msg_iov=&iov;
    hdr.msg_iovlen=1;
    hdr.msg_flags=0;
*/
    if ((x=recv(f, kx, KEYXCHGBLKMIN, 0))<=0) {
	if (errno==EAGAIN || errno==EINTR)
	     /* This is normal, would be even after select */
	    return 0;
	logerr(LOG_INFO, "kxchg: recv");
	if (errno==ENXIO)
	    /* Device was closed */
	    return -1;
	/* allow for some amount of ECONNREFUSED etc. */
	return (OI(maxerr)<0 ? 0 : --errcnt);
    }
    errcnt=OI(maxerr);
    /* dumppacket("recvmsg", hdr.msg_flags, kx, x); */
    if (kx[0]>=CT_DUMMY) {
        kx[55]='\0'; /* max length of message */
        ctrl(f, kx, x);
        return 0;
    }
    if (OI(tokxts) && (abs(time(0)-kx_ts)>OI(tokxts))) {
        dprintf(("KX: timestamp error\n"));
        return 0;
    }
    switch(kx_typ) {
    case NK_RREQ:
	kx_typ=NK_REQ;
	dprintf(("KX: [NK_RREQ] sending NK_REQ\n"));
	x=send(f, &kx, sizeof(kx), 0); e=errno;
	++ks.rreq;
	break;
    case NK_REQ:
	kx_typ=NK_IND;
	if (read(r, &skey, userKeySize)!=userKeySize) {
	    logerr(LOG_ERR, "kxchg: read(r)");
	    return -1;
	}
	memcpy(kx_nkind_key, skey, userKeySize);
#ifdef VER_CRC32
	kx_nkind_crc=(scrc=htonl(crc32(skey, userKeySize)));
#else
	kx_nkind_crc=(scrc=block_crc(skey, userKeySize)); /*#*/
#endif
	dprintf(("KX: [NK_REQ] sending NK_IND " CRCFORM "\n", scrc));
	x=send(f, &kx, sizeof(kx), 0); e=errno;
	{
	    struct siocsifcipkey sk;
	    sk.which=KEY_INVAL+KEY_SEND;
	    if (ioctl_setkey(f, device, &sk)<0)
		logerr(LOG_ERR, "setkey");
	    memset(&sk, 0, sizeof(sk));
	}
	++ks.req;
	break;
    case NK_IND:
	if (
#ifdef VER_CRC32
	    crc32(kx_nkind_key, userKeySize)==ntohl(kx_nkind_crc)
#else
	    block_crc(kx_nkind_key, userKeySize)==(kx_nkind_crc) /*#*/
#endif
	    ) {
	    struct siocsifcipkey sk;
	    if (dupcheck(kx_nkind_crc)) {
		dprintf(("KX: [NK_IND] duplicate\n"));
		++ks.indb;
	    } else {
		sk.which=KEY_RECV;
		memcpy(&sk.thekey, kx_nkind_key, userKeySize);
		if (ioctl_setkey(f, device, &sk)<0)
		    perror("setkey");
		else {
		    kx_typ=NK_ACK;
		    kx_nkack_crc=kx_nkind_crc; /* XX */
		    dprintf(("KX: [NK_IND] sending NK_ACK " CRCFORM "\n",
			     (kx_nkack_crc))); /*#*/
		    x=send(f, &kx, sizeof(kx), 0); e=errno;
		}
		++ks.ind;
	    }
	    memset(&sk, 0, sizeof(sk));
	} else {
	    dprintf(("KX: [NK_IND] invalid\n"));
	    ++ks.indb;
	}
	break;
    case NK_ACK:
	if (scrc==(kx_nkack_crc)) { /*#*/
	    struct siocsifcipkey sk;
	    sk.which=KEY_SEND;
	    memcpy(&sk.thekey, &skey, userKeySize);
	    dprintf(("KX: [NK_ACK] got " CRCFORM "\n", scrc));
	    if (ioctl_setkey(f, device, &sk)<0)
		perror("setkey");
	    ++ks.ack;
	    memset(&sk, 0, sizeof(sk));
	    memset(&skey, 0, sizeof(skey));
	} else {
	    dprintf(("KX: [NK_ACK] unknown " CRCFORM "\n", scrc));
	    ++ks.ackb;
	}
	break;
    default:
	dprintf(("KX: invalid type %02X\n", kx_typ));
	++ks.bogus;
    }
    memset(kx, 0, sizeof(kx));
    if (x<0) {
	errno=e;
	logerr(LOG_NOTICE, "kxchg: send");
	if (e==ENXIO)
	    return -1;
	x=(OI(maxerr)<0 ? 0 : --errcnt);
    }
    return x;
}

/*** Call the ip-up, ip-down scripts ***/

void pspawn(char *s)
{
    int p, c;
    p=getpid();
    if ((c=fork())<0) {
	logerr(LOG_ERR, "pspawn: fork");
	return;
    }
    if (c) {
	(void) wait(NULL); /* exactly one child is active */
    } else {
	/* call <interface> <my-addr> <daemon-pid> <local> <remote> <arg> */
	char bufa[32], bufb[32], bufc[32], bufd[32];
	char *na[]={s, device, bufa, bufb, bufc, bufd, OS(arg), NULL};
        struct options *o = opts;
	sprintf(bufa, "%s:%d", inet_ntoa(OAaddr(me)), ntohs(OAport(me)));
	sprintf(bufb, "%d", p);
	sprintf(bufc, "%s", inet_ntoa(OAaddr(ipaddr)));
	sprintf(bufd, "%s", inet_ntoa(OAaddr(ptpaddr)));
        /* put all options but the first (key) into environment */
        for (o=opts+1; o->oname; ++o)
            putenv(strdup(optstr(o)));
	close(0); close(1); close(2);
	(void) open("/dev/null", O_RDWR);
	(void) dup(0); (void) dup(0);
	execv(s, na);
	logerr(LOG_WARNING, "pspawn: exec");
	exit(1);
    }
}

/*** Main program ***/

int main(int argc, char *argv[])
{
    int i, ur, fd, sd=-1, pi[2];
    time_t t;
    char *st;
    char *pn=strrchr(argv[0], '/');
    if (pn)
	++pn;
    else
	pn=argv[0];

    if (getuid() || geteuid()) {
	fprintf(stderr, "This program must be run by root.\n");
	exit(1);
	/* DO NOT remove this check */
    }
    /* Snarf options from: standard options file, user-supplied file and
       finally, command line */
    setopt_file(CTLDIR "/options", 0);
    if ((argc>1) && (!strcmp(argv[1], "-o"))) {
	if (argc>2) {
	    setopt_file(argv[2], 1);
	    argc-=2; argv+=2;
	} else {
	    usage(pn);
	}
    }
    setopt_cmdline(--argc, ++argv);
    if (!OI(maxerr))
	OI(maxerr)=8; /* compatibility */

    for (i=getdtablesize()-1; i>2; --i)
	(void) close(i);
    if (!OI(debug)) {
	/* daemonize */
        if (pipe(pi)<0) {
            perror("pipe"); exit(1);
        }
	if ((i=fork())<0) {
	    perror("fork"); exit(1);
	}
	if (i) {
            /* wait until the child is ready */
            close(pi[1]);
            (void) read(pi[0], &i, 1);
	    exit(0);
        }
        close(pi[0]);
	openlog(pn, LOG_PID, LOGFAC);
	/* don't let the syslog socket get in the way of the standard fds */
	close(0); close(1); close(2);
	setsid();
	syslog(LOG_INFO, "CIPE daemon vers %s (c) Olaf Titz 1996-1998",
	       daemon_version);
    } else {
	printf("CIPE daemon vers %s (c) Olaf Titz 1996-1998\n",
	       daemon_version);
	dumpopt();
    }

    setsig(SIGHUP, sighand);
    setsig(SIGINT, sighand);
    setsig(SIGTERM, sighand);
    errcnt=OI(maxerr);
    if ((ur=open("/dev/urandom", O_RDONLY))<0) {
	logerr(LOG_ERR, "open(/dev/urandom)"); exit(1);
    }
    i=MAXINT;
    if (OA(socks)) {
        i=30; /* set timeout for control socket check */
        if ((sd=opensocks())<0)
	    exit(1);
    }
    if ((fd=opendev())<0)
	exit(1);
    pspawn(OS(ipup) ? OS(ipup) : CTLDIR "/ip-up");

    /* Set timer for periodic checks: ping, ping-timeout */
    if (OI(ping)) {
        pingtime=time(0);
        if (OI(ping)<i)
            i=OI(ping);
    }
    if (OI(toping)) {
        if (OI(toping)<i)
            i=OI(toping);
    }
    if (i<MAXINT) {
	struct itimerval t;
	setsig(SIGALRM, sigignore);
	t.it_interval.tv_sec = i;
	t.it_interval.tv_usec = 0;
	t.it_value.tv_sec = i+10;
	t.it_value.tv_usec = 0;
	setitimer(ITIMER_REAL, &t, 0);
    }

    if (!OI(debug))
        close(pi[1]);
    /* sendctl(fd, "\x71starting up", 13); */

    /* Main loop */
    while (kxchg(fd, ur)>=0) {
        if (gotsig) {
            logs(LOG_NOTICE, "got %s", strsignal(gotsig));
            break;
        }
	if ((sd>=0) && ((i=eofsocks(sd)))) {
	    logs(LOG_NOTICE, "%s on socks connection", eofsocks_str[i]);
	    break;
	}
        t=time(0);
        if (t>=pongtime) {
            logs(LOG_NOTICE, "keepalive timeout", NULL);
            if (OI(maxerr)>=0 && (--errcnt<0))
                break;
        }
        if (t>=pingtime) {
            char buf=CT_PING;
            sendctl(fd, &buf, 1);
            pingtime=t+OI(ping);
            pongtime=t+OI(toping);
        }
    }
    sendctl(fd, "\x71going down", 12);
    if ((st=getstats()))
        logs(LOG_INFO, "Interface stats %s\n", st);
    closedev(fd);
    if (OI(debug))
	printf("KX stats: rreq=%d, req=%d, ind=%d, indb=%d, ack=%d,"
	       " ackb=%d, unknown=%d\n", ks.rreq, ks.req, ks.ind, ks.indb,
	       ks.ack, ks.ackb, ks.bogus);
    else
	syslog(LOG_INFO, "KX stats: rreq=%d, req=%d, ind=%d, indb=%d, ack=%d,"
	       " ackb=%d, unknown=%d\n", ks.rreq, ks.req, ks.ind, ks.indb,
	       ks.ack, ks.ackb, ks.bogus);
    pspawn(OS(ipdown) ? OS(ipdown) : CTLDIR "/ip-down");
    logs(LOG_INFO, "%s: daemon exiting", device);
    return 0;
}
