------------------------------------------------------------------------------
--                              G N A T L I B                               --
--                                                                          --
--                     Copyright (C) 2001-2015, AdaCore                     --
--                                                                          --
-- This library is free software;  you can redistribute it and/or modify it --
-- under terms of the  GNU General Public License  as published by the Free --
-- Software  Foundation;  either version 3,  or (at your  option) any later --
-- version. This library is distributed in the hope that it will be useful, --
-- but WITHOUT ANY WARRANTY;  without even the implied warranty of MERCHAN- --
-- TABILITY or FITNESS FOR A PARTICULAR PURPOSE.                            --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
-- You should have received a copy of the GNU General Public License and    --
-- a copy of the GCC Runtime Library Exception along with this program;     --
-- see the files COPYING3 and COPYING.RUNTIME respectively.  If not, see    --
-- <http://www.gnu.org/licenses/>.                                          --
--                                                                          --
------------------------------------------------------------------------------

with GNATCOLL.VFS;       use GNATCOLL.VFS;
with GNATCOLL.VFS_Types; use GNATCOLL.VFS_Types;
with Ada.Text_IO;        use Ada.Text_IO;
with GNAT.Strings;       use GNAT.Strings;

procedure Test_VFS is

   procedure Test_F (Dir : Virtual_File);
   --  Perform tests on a specific file

   procedure Test_F (Dir : Virtual_File) is
      F       : Virtual_File;
      W       : Writable_File;
      Str     : String_Access;
      Success : Boolean;
      Files   : File_Array_Access;
      Count   : Natural := 0;
   begin
      --  A file that does not exist yet
      F := Create_From_Dir
        (Dir => Dir, Base_Name => "foo.txt");

      Put_Line ("base name=" & (+Base_Name (F)));
      Put_Line ("full name=" & (+Full_Name (F)));
      Put_Line ("dir name="  & (+Dir_Name (F)));
      Put_Line ("extension=" & (+F.File_Extension));
      Put_Line ("is absolute=" & Is_Absolute_Path (F)'Img);
      Put_Line ("is regular file=" & Is_Regular_File (F)'Img);

      --  Now create the file

      W := Write_File (F);
      Write (W, "first word ");
      Write (W, "second word");
      Close (W);

      --  Check whether the file exists

      Put_Line ("=== after creation ===");
      Put_Line ("is regular file=" & Is_Regular_File (F)'Img);
      Put_Line ("is writable=" & Is_Writable (F)'Img);

      --  Make the file unreadable

      Set_Readable (F, False);
      Put_Line ("=== after making it unreadable ===");
      Put_Line ("is regular file=" & Is_Regular_File (F)'Img);

      --  Try and read the file

      Str := Read_File (F);
      if Str /= null then
         Put_Line ("Could read file, surprising!");
      end if;
      Free (Str);

      --  Make it readable again, and read again

      Set_Readable (F, True);
      Str := Read_File (F);
      Put_Line ("contents when readable=" & Str.all);
      Free (Str);

      --  Check contents of directory

      Put_Line ("is directory=" & Is_Directory (Dir)'Img);

      Files := Read_Dir (Create_From_Dir (Dir, "sub"));
      Put ("Files in directory=");
      for L in Files'Range loop
         if Base_Name (Files (L)) /= ".svn" then
            Count := Count + 1;
            --  Put (" " & Base_Name (Files (L)));
         end if;
      end loop;
      Put (Count'Img);
      Unchecked_Free (Files);
      New_Line;

      --  Delete the file

      Delete (F, Success);
      Put_Line ("Could delete: " & Success'Img);
      Delete (F, Success);
      Put_Line ("Could delete again: " & Success'Img);
   end Test_F;

begin
   Test_F (Get_Current_Dir);

   --  Try manipulating the No_File

   Put_Line ("=== manipulate no_file ===");
   Put_Line ("base name=" & (+Base_Name (No_File)));
   Put_Line ("full name=" & (+Full_Name (No_File)));
   Put_Line ("dir name="  & (+Dir_Name (No_File)));
   Put_Line ("extension=" & (+File_Extension (No_File)));
   Put_Line ("is absolute=" & Is_Absolute_Path (No_File)'Img);
   Put_Line ("is regular file=" & Is_Regular_File (No_File)'Img);

   --  Various tests

   Put_Line ("=== misc tests ===");
   Put_Line ("base name of dir=" & (+Base_Name (Get_Current_Dir)));

   --  Test for I820-010
   declare
      Default_Pref_Py : constant Virtual_File := Create_From_Dir
        (Dir => Get_Current_Dir, Base_Name => "default_pref.py");
      Default_Pref_Pyc : constant Virtual_File := Create_From_Dir
        (Dir => Get_Current_Dir, Base_Name => "default_pref.pyc");
   begin
      Put_Line
        ("default_pref.py < default_pref.pyc = " &
         Boolean (Default_Pref_Py < Default_Pref_Pyc)'Img);
      Put_Line
        ("default_pref.pyc < default_pref.py = " &
         Boolean (Default_Pref_Pyc < Default_Pref_Py)'Img);
   end;
end Test_VFS;
