/* -*- mode: c; c-file-style: "gnu" -*-
 * nqueue_poll.c - poll()-based event system
 * Copyright (C) 2003, 2004 Gergely Nagy <algernon@bonehunter.rulez.org>
 *
 * This file is part of Thy.
 *
 * Thy is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 dated June, 1991.
 *
 * Thy is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "system.h"

#if HAVE_SYS_POLL_H
#include <sys/types.h>
#include <inttypes.h>

#include <errno.h>
#include <fcntl.h>
#include <string.h>
#if HAVE_SYS_POLL_H
#include <sys/poll.h>
#endif
#include <sys/time.h>
#include <unistd.h>

#include "compat/compat.h"

#include "nqueue.h"
#include "nqueue_poll.h"
#include "types.h"
#include "thy.h"

static struct pollfd *pfds; /**< FDs to watch, for poll(). */

void
thy_nq_fd_control_poll (int fd, thy_nq_event_t event, int set)
{
  if (fd < 0)
    return;

  if (fdstates[fd] && thy_nq_act_fds)
    thy_nq_act_fds--;

  if (trigger[fd])
    trigger_count--;

  if (fd > thy_nq_max_fd)
    thy_nq_max_fd = fd;

  fdstates[fd] = 0;
  trigger[fd] = 0;
  pfds[fd].fd = -1;

  if (!set)
    return;

  switch (event)
    {
    case THY_NQ_EVENT_INPUT:
      pfds[fd].events = POLLIN | POLLHUP | POLLERR;
      break;
    case THY_NQ_EVENT_OUTPUT:
      pfds[fd].events = POLLOUT | POLLHUP | POLLERR;
      break;
    case THY_NQ_EVENT_TRIGGER:
      trigger[fd] = 1;
      trigger_count++;
      break;
    default:
      break;
    }

  thy_nq_act_fds++;
  fdstates[fd] = 1;
  pfds[fd].fd = fd;
}

int
thy_nq_init_poll (void)
{
  pfds = (struct pollfd *)bhc_calloc (_THY_MAXCONN + 1,
				      sizeof (struct pollfd));
  return 0;
}

int
thy_nq_fd_check_poll (int fd, thy_nq_event_t event)
{
  int rval = 0;

  if (fd < 0)
    return 0;

  switch (event)
    {
    case THY_NQ_EVENT_INPUT:
      if (pfds[fd].revents & (POLLIN | POLLHUP | POLLERR))
	rval = 1;
      break;
    case THY_NQ_EVENT_OUTPUT:
      if (pfds[fd].revents & (POLLOUT | POLLHUP | POLLERR))
	rval = 1;
      break;
    case THY_NQ_EVENT_TRIGGER:
      if (trigger[fd])
	rval = 1;
    default:
      return 0;
    }

  if (rval && fd > thy_nq_max_fd)
    thy_nq_max_fd = fd;

  return rval;
}

int
thy_nq_wait_poll (long int timeout)
{
  return poll (pfds, thy_nq_max_fd + 1, timeout * 1000);
}
#endif

/* arch-tag: e41b997b-c0d9-4603-9474-7fade5fc0f8d */
