/* -*- mode: c; c-file-style: "gnu" -*-
 * auth.c -- http authentication routines
 * Copyright (C) 2002, 2003, 2004 Gergely Nagy <algernon@bonehunter.rulez.org>
 *
 * This file is part of Thy.
 *
 * Thy is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 dated June, 1991.
 *
 * Thy is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/** @file auth.c
 * HTTP authentication support routines.
 *
 * The routines herein deal with HTTP authentication: whether it is
 * needed or not, whether the requesting client has permission to
 * access its destination, and so on. Of course, with helper
 * functions.
 *
 * The authentication itself is handed out to an external
 * authoriser, so the code here is a glue between that and Thy.
 */

#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>

#include "compat/compat.h"

#include "auth.h"
#include "config.h"
#include "fabs.h"
#include "misc.h"
#include "options.h"
#include "thy.h"
#include "types.h"

/** @internal Socket connected to the authoriser process.
 */
static int _thy_auth_socks[2] = {-1, -1};
/** @internal PID of the Authoriser process.
 */
static pid_t _thy_auth_child = 0;

/** @internal Table used for base64 decoding
 */
static char index_64[128] =
  {
    -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1,
    -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1,
    -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,62, -1,-1,-1,63,
    52,53,54,55, 56,57,58,59, 60,61,-1,-1, -1,-1,-1,-1,
    -1, 0, 1, 2,  3, 4, 5, 6,  7, 8, 9,10, 11,12,13,14,
    15,16,17,18, 19,20,21,22, 23,24,25,-1, -1,-1,-1,-1,
    -1,26,27,28, 29,30,31,32, 33,34,35,36, 37,38,39,40,
    41,42,43,44, 45,46,47,48, 49,50,51,-1, -1,-1,-1,-1
  };
/** @internal Decode a base64 encoded char.
 */
#define char64(c) (((c) < 0 || (c) > 127) ? -1 : index_64[(c)])

/** @internal Decode a base64 encoded string.
 * @param inbuf is the string to decode.
 *
 * @returns The decoded string, in a newly allocated area, which must
 * be freed later on.
 */
static char *
_auth_from64 (const char *inbuf)
{
  int c1, c2, c3, c4;
  size_t inbufpos = 0, outpos = 0, inbuflen = strlen (inbuf);
  char *outbuf = (char *)bhc_malloc (inbuflen);

  while ((c1 = inbuf[inbufpos++]) && inbufpos < inbuflen)
    {
      do c2 = inbuf[inbufpos++];
      while (inbufpos <= inbuflen && isspace (c2));
      do c3 = inbuf[inbufpos++];
      while (inbufpos <= inbuflen && isspace (c3));
      do c4 = inbuf[inbufpos++];
      while (inbufpos <= inbuflen && isspace (c4));

      if (inbufpos > inbuflen)
	{
	  free (outbuf);
	  return NULL;
	}
      if (c1 == '=' || c2 == '=')
	{
	  outbuf[outpos++] = 0;
	  return outbuf;
	}

      c1 = char64 (c1);
      c2 = char64 (c2);

      outbuf[outpos++] = ((c1<<2) | ((c2&0x30)>>4));
      if (c3 == '=')
	{
	  outbuf[outpos++]=0;
	  return outbuf;
	}
      else
	{
	  c3 = char64 (c3);
	  outbuf[outpos++] = (((c2&0XF) << 4) | ((c3&0x3C) >> 2));
	  if (c4 == '=')
	    {
	      outbuf[outpos++]=0;
	      return outbuf;
	    }
	  else
	    {
	      c4 = char64 (c4);
	      outbuf[outpos++] = (((c3&0x03) <<6) | c4);
	    }
	}
    }
  outbuf[outpos++]=0;
  return outbuf;
}

/** @internal Parse a base64 encoded string.
 * Given a base64 encoded TOKEN of the form USER:PASSWORD, this
 * function decodes it, and stores the values in USER and PASSWORD.
 *
 * @note Since both user and password are pointers to a string, and
 * are allocated by this function, they must be freed by the caller.
 */
static void
_auth_token_parse (const char *token, char **user, char **password)
{
  char *decoded;
  char *tmp;

  decoded = _auth_from64 (token);

  if ((tmp = strchr (decoded, ':')) == NULL)
    {
      free (decoded);
      *user = NULL;
      *password = NULL;
      return;
    }

  *user = bhc_strndup (decoded, (size_t)(tmp - decoded));
  *password = bhc_strdup (&decoded[(int)(tmp - decoded) + 1]);

  free (decoded);
}

/** Check if any authorisation is needed.
 * Starts from the end of the path the request in SESSION resolved
 * to, and dir-by-dir, walks backwards, checking each one for a
 * .realm (or whatever is in config->auth.file) file. If found, ask
 * the Authoriser proces whether we need authorisation or not.
 *
 * @returns Zero if no authorisation is needed, -1 otherwise.
 */
int
auth_need (session_t *session)
{
  thy_mappable_config_t *map_config =
    config_get_mapped (session->absuri, session->request->resolved);
  const config_t *config = config_get ();
  char *rest;
  size_t i;
  char *tmp;

  if (map_config->options.auth != THY_BOOL_TRUE)
    {
      free (map_config);
      return 0;
    }

  free (map_config);

  rest = fabs_urlmap (session->request->url, session->request->host,
		      session->absuri);
  i = strlen (rest);
  tmp = NULL;
  do
    {
      asprintf (&tmp, "%s/%s", rest, config->auth.file);
      if (!fabs_access (tmp, F_OK))
	{
	  char *msg = NULL, c[7];
	  int len;
	  int error = 0;

	  free (rest);

	  session->request->auth_file = bhc_strdup (tmp);

	  if (_thy_auth_socks[0] == -1 || _thy_auth_socks[1] == -1)
	    {
	      free (tmp);
	      bhc_error ("%s", "Authoriser not connected.");
	      return -1;
	    }

	  len = asprintf (&msg, "CHECK\n1\n%s\n", tmp);
	  free (tmp);

	  if (write (_thy_auth_socks[0], msg, len) != len)
	    {
	      bhc_error ("Error communicating with the Authoriser: %s",
			 strerror (errno));
	      error =1;
	    }
	  free (msg);
	  if (error)
	    return -1;

	  /* Check if authorisation is even needed */
	  if (read (_thy_auth_socks[1], c, 2) != 2)
	    {
	      bhc_error ("Error communicating with the Authoriser: %s",
			 strerror (errno));
	      return -1;
	    }

	  if (c[0] == '0')
	    return 0;

	  /* Determine the length of the realm. We support at most
	     65536 chars, which is 5+1 bytes, zero padded and LF
	     terminated. */
	  if (read (_thy_auth_socks[1], c, 6) != 6)
	    {
	      bhc_error ("Error communicating with the Authoriser: %s",
			 strerror (errno));
	      return -1;
	    }
	  c[5] = '\0';

	  /* Read the realm. */
	  len = atoi (c);
	  msg = (char *)bhc_malloc (len + 2);
	  memset (msg, 0, len + 1);
	  if (read (_thy_auth_socks[1], msg, len + 1) != len + 1)
	    {
	      bhc_error ("Error coummunicating with the Authoriser: %s",
			 strerror (errno));
	      free (msg);
	      return -1;
	    }
	  free (session->request->auth_realm);
	  session->request->auth_realm = bhc_strndup (msg, len);
	  free (msg);
	  return -1;
	}

      free (tmp);
      tmp = strrchr (rest, '/');
      if (tmp)
	{
	  i -= strlen (tmp);
	  if (i > 0)
	    rest[i] = '\0';
	}
    } while (i > 0);

  free (rest);
  return 0;
}

/** Do the authentication itself.
 * @returns Zero on sucess, -1 if authentication failed.
 */
int
auth_authenticate (const session_t *session)
{
  char *user, *pw, *msg = NULL, result[2];
  int error = 0, len;

  if (_thy_auth_socks[0] == -1 || _thy_auth_socks[1] == -1)
    {
      bhc_error ("%s", "Authoriser not connected.");
      return -1;
    }

  if (!session->request->auth_token)
    {
      bhc_error ("No authorisation token supplied by %s",
		 session->origin);
      return -1;
    }

  _auth_token_parse (session->request->auth_token,
		     &user, &pw);

  len = asprintf (&msg, "AUTH\n5\n%s\n%s\n%s\n%s\n%s\n", user, pw,
		  session->request->auth_realm,
		  session->request->resolved,
		  session->request->auth_file);
  if (write (_thy_auth_socks[0], msg, len) != len)
    error = 1;

  free (user);
  free (pw);
  free (msg);

  if (error)
    {
      bhc_error ("Error communicating with the Authoriser: %s",
		 strerror (errno));
      return -1;
    }

  if (read (_thy_auth_socks[1], result, 2) != 2)
    {
      bhc_error ("Error communicating with the Authoriser: %s",
		 strerror (errno));
      return -1;
    }

  return (result[0] == '1' && result[1] == '\n') ? 0 : -1;
}

/** @internal Shake hands with the Authoriser.
 * This function does the handshaking with the Authoriser. For now,
 * this consists only of telling it our protocol version, and seeing
 * if it supports it.
 *
 * @returns Zero on success, -1 otherwise.
 */
static int
_thy_auth_handshake (void)
{
  char c[2];
  char *msg = NULL;
  int len;

  len = asprintf (&msg, "VERSION\n1\n%d\n", _THY_AUTH_PROTOCOL_VERSION);

  if (write (_thy_auth_socks[0], msg, len) != len)
    {
      bhc_error ("Error communicating with the Authoriser: %s",
		 strerror (errno));
      free (msg);
      return -1;
    }
  free (msg);
  if (read (_thy_auth_socks[1], c, 2) != 2)
    {
      bhc_error ("Error communicating with the Authoriser: %s",
		 strerror (errno));
      return -1;
    }

  return (c[0] == '1' && c[1] == '\n') ? 0 : -1;
}

/** Initialise the authorisation engine.
 * This one spawns off the authorisation process, and sets up the
 * communication pipes.
 */
int
auth_init (void)
{
  int socks[2][2], i, j, e = 0;
  const config_t *config = config_get ();

  _thy_auth_socks[0] = -1;
  _thy_auth_socks[1] = -1;

  if (pipe (socks[0]) == -1)
    {
      bhc_error ("pipe() failed: %s", strerror (errno));
      return -1;
    }
  if (pipe (socks[1]) == -1)
    {
      bhc_error ("pipe() failed: %s", strerror (errno));
      return -1;
    }

  for (i = 0; i <= 1; i++)
    for (j = 0; j <= 1; j++)
      {
	if (fcntl (socks[i][j], F_SETFD, FD_CLOEXEC) == -1)
	  {
	    bhc_error ("fcntl: %s", strerror (errno));
	    close (socks[i][j]);
	    e = 1;
	  }
      }
  if (e)
    return -1;

  _thy_auth_child = fork ();
  switch (_thy_auth_child)
    {
    case 0:
      /* Child */
      closelog ();
      openlog ("thy/auth", LOG_PID, LOG_DAEMON);
      thy_priv_drop (config->auth.uid);
      dup2 (socks[0][0], STDIN_FILENO);
      dup2 (socks[1][1], STDOUT_FILENO);
      execv (config->auth.path, config->auth.args.argv);
      bhc_exit (1);
      break;
    case -1:
      /* Error */
      bhc_error ("fork(): %s", strerror (errno));
      return -1;
    default:
      /* Parent */
      bhc_log ("Authoriser (%s) started as PID %d", config->auth.path,
	       _thy_auth_child);
      close (socks[0][0]);
      close (socks[1][1]);
      _thy_auth_socks[0] = socks[0][1];
      _thy_auth_socks[1] = socks[1][0];

      if (_thy_auth_handshake () == -1)
	{
	  bhc_error ("%s", "Authoriser handshake failed.");
	  kill (_thy_auth_child, SIGTERM);
	  _thy_auth_socks[0] = -1;
	  _thy_auth_socks[1] = -1;
	}
      return 0;
    }
  return -1;
}

/** Terminate the Authoriser process.
 * Sends a QUIT command to the Authoriser. We cannot simply kill it,
 * as it might run with higher privileges than Thy herself.
 */
void
auth_done (void)
{
  write (_thy_auth_socks[0], "QUIT\n0\n", sizeof ("QUIT\n0\n"));
}
