/* -*- mode: c; c-file-style: "gnu" -*-
 * list.c -- list handling routines
 * Copyright (C) 2003, 2004 Gergely Nagy <algernon@bonehunter.rulez.org>
 *
 * This file is part of BoneHunter Libs.
 *
 * BoneHunter Libs is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 dated June,
 * 1991.
 *
 * BoneHunter libs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/** @file list.c
 * Append-optimised list implementation,
 *
 * Since most of the list-type variables in current bonehunter
 * programs are created once, then never manipulated, but searched and
 * destroyed, the implementation of _bhl_list_t is optimised towards
 * this. The notable exception is the TLS session cache in Thy's
 * src/tls.c.
 *
 * The entire list is in reality a dynamically allocated array of
 * bhl_list_entry_t elements. It is not really a list, to be
 * honest. Deleting and inserting elements before the last element is
 * very costy, while appending is very cheap - since the address of
 * the last element is easy to compute, it is a matter of allocating
 * the necessary memory, and copying the data over.
 *
 * Insertion and deletion would mean that all later elements must be
 * copied over, which is just too much overkill. Therefore, a clever
 * hack was introduced: bhl_list_delete() does not really delete the
 * element, just frees it up, and sets it to NULL. If one wants to
 * insert elements, instead of appennding, he must use
 * bhl_list_insert(), which will go and search for such empty
 * elements.
 *
 * For destroying struct-like list elements, the #bhl_list_free_f hook
 * is provided.
 */

#include "compat/compat.h"
#include "list.h"

#include <stdlib.h>
#include <string.h>

/** @internal Type of a list element.
 * This structure holds both the list element data, and some
 * meta-information (notably, the size of the data). This is used to
 * store entries in a _bhl_list_t.
 */
typedef struct
{
  void *data; /**< The entry data itself. */
  size_t size; /**< Size of the data. */
} bhl_list_entry_t;

/** @internal Internal list representation.
 * The real representation of a list - well, a growable array, but
 * anyway... Holds meta-data, such as the size and the length of the
 * list, and of course the entry-array itself.
 */
struct _bhl_list_t
{
  /** Allocated entries.
   * This is the maximum number of elements in the list. If it would
   * grow bigger, it must be reallocated.
   */
  size_t allocated;
  size_t length; /**< Used entries.\  Usually less than allocated. */
  bhl_list_free_f efree; /**< Function used to free the elements. */

  /** The list-element array.
   * A growable array, holding list elements and their meta-data.
   */
  bhl_list_entry_t **entries;
};

/** Initialize a new list.
 * Initialize a new list with the specified parameters.
 *
 * @param initsize specifies the initial size of the list.
 * @param efree is either a function to free one list element, or
 * NULL, if the standard free() call should be used instead.
 *
 * @returns A pointer to the newly created list.
 */
bhl_list_t *
bhl_list_init (size_t initsize, bhl_list_free_f efree)
{
  bhl_list_t *l = (bhl_list_t *)bhc_malloc (sizeof (bhl_list_t));

  l->allocated = initsize;
  l->length = 0;
  l->efree = efree;
  l->entries = (bhl_list_entry_t **)
    bhc_calloc (l->allocated, sizeof (bhl_list_entry_t *));
  return l;
}

/** Free all elements of a list.
 * Iterates through a list and frees all elements, by calling
 * bhl_list_delete() on them.
 *
 * @param list is the list to be freed up.
 */
void
bhl_list_free (bhl_list_t *list)
{
  size_t i;

  if (!list)
    return;

  for (i = 0; i < list->length; i++)
    bhl_list_delete (list, i);

  free (list->entries);
  free (list);
}

/** Append new data to a list.
 * Append custom data to an existing list, growing it too, if
 * necessary.
 *
 * @param list is the list to append to.
 * @param data is the element to append.
 * @param size is its size.
 *
 * @returns The index of the appended data.
 */
size_t
bhl_list_append (bhl_list_t *list, const void *data, size_t size)
{
  size_t pos = list->length;

  if (list->length >= list->allocated)
    {
      list->allocated = (list->allocated + 1) * 2;
      XSREALLOC (list->entries, bhl_list_entry_t *,
		 list->allocated);
    }
  list->entries[pos] =
    (bhl_list_entry_t *)bhc_malloc (sizeof (bhl_list_entry_t));
  list->entries[pos]->size = size;
  list->entries[pos]->data = bhc_malloc (size);
  memcpy (list->entries[pos]->data, data, size);

  return list->length++;
}

/** A convenience wrapper around bhl_list_append().
 * Calculates the length of the string automatically, then transfers
 * control to bhl_list_append().
 */
size_t
bhl_list_append_string (bhl_list_t *list, const char *str)
{
  return bhl_list_append (list, str, strlen (str) + 1);
}

/** Retrieve a list element.
 * Retrieves the specified element of a list. A copy of the requested
 * element is put into the third paramater (data). It must be freed by
 * the caller.
 *
 * @param list is the list to operate on.
 * @param idx is the index of the sought element.
 * @param data is where the result will go.
 *
 * @returns The size of the data, or 0 on error.
 */
size_t
bhl_list_get (const bhl_list_t *list, size_t idx, void **data)
{
  if (list->length <= idx)
    return 0;
  if (!list->entries[idx])
    return 0;

  *data = bhc_malloc (list->entries[idx]->size);
  memcpy (*data, list->entries[idx]->data, list->entries[idx]->size);

  return list->entries[idx]->size;
}

/** Figure out the length of a list.
 * @returns The length of the list.
 */
size_t
bhl_list_size (const bhl_list_t *list)
{
  return list->length;
}

/** Insert new data to a list.
 * Insert custom data to an existing list, growing it too, if
 * necessary. Unlike bhl_list_append(), this searches for free slots
 * in the list, and does not always append to the end.
 *
 * @param list is the list to append to.
 * @param data is the element to append.
 * @param size is its size.
 *
 * @returns The index of the inserted data.
 */
size_t
bhl_list_insert (bhl_list_t *list, const void *data, size_t size)
{
  size_t pos = 0;

  while (pos < bhl_list_size (list) && list->entries[pos])
    pos++;

  if (pos < bhl_list_size (list))
    {
      list->entries[pos] =
	(bhl_list_entry_t *)bhc_malloc (sizeof (bhl_list_entry_t));
      list->entries[pos]->size = size;
      list->entries[pos]->data = bhc_malloc (size);
      memcpy (list->entries[pos]->data, data, size);
      return pos;
    }
  else
    return bhl_list_append (list, data, size);
}

/** Delete an entry from a list.
 * Purges an element from the list, without touching anything else (it
 * does not copy the later elements over, so thy_list_append() will
 * still append to the list and waste some memory.)
 *
 * @param list is the list to operate on.
 * @param idx is the index of the entry to destroy.
 *
 * @returns Zero on success, -1 otherwise.
 *
 * @see bhl_list_insert().
 */
int
bhl_list_delete (bhl_list_t *list, size_t idx)
{
  if (list->length <= idx)
    return -1;
  if (!list->entries[idx])
    return -1;

  if (list->efree)
    (*(list->efree)) (list->entries[idx]->data);
  free (list->entries[idx]->data);
  free (list->entries[idx]);
  list->entries[idx] = NULL;

  return 0;
}

/* arch-tag: 2a968050-4e7c-4ae6-9b15-880cc8f64a8d */
