#ifndef TMATRIX_H
#define TMATRIX_H

// Copyright (c) 2000, 2001, 2002, 2003 by David Scherer and others.
// See the file license.txt for complete license terms.
// See the file authors.txt for a complete list of contributors.

// tmatrix - double-precision 3D transformation matrices
#include "vector.h"
#include "vertex.h"

namespace visual {
	
	
class tmatrix 
{
public:
	typedef double scalar;

	scalar M[4][4];

	tmatrix() throw() {}
    
	tmatrix( const tmatrix& t ) throw()
	{ memcpy(M, t.M, sizeof(M)); }
    
	tmatrix( const tmatrix& A, const tmatrix& B ) throw()
	{ concat( A, B); }

	static tmatrix 
	identity() throw()
    { tmatrix t; t.ident(); return t; }

	inline void ident( void) throw() 
	{
		x_column();
		y_column();
		z_column();
		w_column();
		w_row();
	}
	

	inline void x_column( const vector& v) throw()
	{
		M[0][0] = v.x;
		M[1][0] = v.y;
		M[2][0] = v.z;
	}
    
	inline void y_column( const vector& v) throw()
	{
		M[0][1] = v.x;
		M[1][1] = v.y;
		M[2][1] = v.z;
	}
    
	inline void z_column( const vector& v) throw()
	{
		M[0][2] = v.x;
		M[1][2] = v.y;
		M[2][2] = v.z;
	}
    
	inline void w_column( const vector& v) throw()
	{
		M[0][3] = v.x;
		M[1][3] = v.y;
		M[2][3] = v.z;
	}

	inline void x_column( double x=1, double y=0, double z=0) throw()
	{
		M[0][0] = x;
		M[1][0] = y;
		M[2][0] = z;
	}
    
	inline void y_column( double x=0, double y=1, double z=0) throw()
	{
		M[0][1] = x;
		M[1][1] = y;
		M[2][1] = z;
	}
    
	inline void z_column( double x=0, double y=0, double z=1) throw()
	{
		M[0][2] = x;
		M[1][2] = y;
		M[2][2] = z;
    }
    
	inline void w_column(double x=0, double y=0, double z=0) throw()
	{
		M[0][3] = x;
		M[1][3] = y;
		M[2][3] = z;
	}
    
	inline void w_row(double x=0, double y=0, double z=0, double w=1) throw()
	{
		M[3][0]=x;
		M[3][1]=y;
		M[3][2]=z;
		M[3][3]=w;
	}

	// Projects v to o using the current tmatrix values.
	// May be called with a float[3] or double[3] for v (implicitly converted).
	// We take a full copy just to be safe here.
	void project(const vector v, vertex& o) const throw();

	void concat(const tmatrix& A, const tmatrix& B) throw();

	void invert_ortho(const tmatrix& A) throw();

	inline const double*
	operator[]( int n) const throw() { return M[n]; }
	
	inline double*
	operator[]( int n) throw() { return M[n]; }

	// M^-1 * [x y z w]
	vector times_inv( const vector& v, double w = 1.0) const throw();

	// multiplication by a vector [x y z 0]
	vector times_v( const vector& v) const throw();

	// multiplication by a point [x y z 1]
	vector operator*( const vector& v) const throw();
    
	inline double 
	x( const vector& v) const throw()
	{
		return M[0][0]*v.x + M[0][1]*v.y + M[0][2]*v.z + M[0][3];
	}
    
	inline double 
	y( const vector& v) const throw()
	{
		return M[1][0]*v.x + M[1][1]*v.y + M[1][2]*v.z + M[1][3];
	}
    
	inline double 
	z( const vector& v) const throw()
	{
		return M[2][0]*v.x + M[2][1]*v.y + M[2][2]*v.z + M[2][3];
	}
    
	inline double 
	w(const vector& v) const throw()
	{
		return M[3][0]*v.x + M[3][1]*v.y + M[3][2]*v.z + M[3][3];
	}
	
	// Overwrites the currently active matrix in OpenGL.
	void
	gl_load(void);
};

void frustum( tmatrix& T, tmatrix& I, double l, double r, double b, double t, double n, double f ) throw();
void rotation( tmatrix& T, double angle, const vector& a) throw();
double norm_dot( const vector& a, const vector& b) throw();

void tmatrix_init_type();


// Set up a tmatrix to rotate a vector or object as specified by angle, axis and origin
void py_rotation( tmatrix& R, double angle, vector axis, vector origin) throw();

} // !namespace visual

#endif // !TMATRIX_H
