/*
** Copyright (C) 2000 Idan Shoham <idan@m-tech.ab.ca>
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
** 
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
** 
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software 
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

using namespace std;

#include "messenger.h"
#include "project.h"
#include "reporter.h"

#define MIN(A,B) ((A)<(B) ? (A) : (B))
#define MAX(A,B) ((A)>(B) ? (A) : (B))

int Reporter::Bound(int x, int x1, int x2)
{
    if ( x < x1 )
	return x1;
    if ( x > x2 )
	return x2;
    return x;
}


int Reporter::Offset(int x, int x1, int x2 )
{
    return Bound( x, x1, x2 ) - x1;
}

int Reporter::MapX( Project *project, int dayNo, int start, int finish)
{
    if ( dayNo < start )
	dayNo = start;
    if ( dayNo > finish )
	dayNo = finish;

    return tg_left + tg_width *
	Offset( project->nDays(dayNo), 
		project->nDays(start),
		project->nDays(finish) );
}

FILE *Reporter::OpenTaskGraph(const char *filename, int x1, int y1, int x2, int y2)
{ 
    FILE *f;

    f = fopen(filename,"w");
    if ( f == NULL )
	Error("Cannot open file [%s]",filename);

    fprintf(f,"%%!PS-Adobe-2.0 EPSF-2.0\n"); 
    fprintf(f,"%%%%BoundingBox: %d %d %d %d\n",x1,y1,x2,y2);

    fprintf(f,"/%s findfont\n", tg_fontname1); 
    fprintf(f,"dup length dict begin\n"); 
    fprintf(f," { 1 index /FID ne\n"); 
    fprintf(f," {def}\n"); 
    fprintf(f," {pop pop}\n"); 
    fprintf(f," ifelse\n"); 
    fprintf(f," } forall\n"); 
    fprintf(f," /Encoding ISOLatin1Encoding def\n"); 
    fprintf(f," currentdict\n"); 
    fprintf(f,"end\n"); 
    fprintf(f,"/%s-ISOLatin1 exch definefont pop\n", tg_fontname1); 

    if ( strcmp(tg_fontname1, tg_fontname2) != 0 )
    {
	fprintf(f,"/%s findfont\n", tg_fontname2); 
	fprintf(f,"dup length dict begin\n"); 
	fprintf(f," { 1 index /FID ne\n"); 
	fprintf(f," {def}\n"); 
	fprintf(f," {pop pop}\n"); 
	fprintf(f," ifelse\n"); 
	fprintf(f," } forall\n"); 
	fprintf(f," /Encoding ISOLatin	1Encoding def\n"); 
	fprintf(f," currentdict\n"); 
	fprintf(f,"end\n"); 
	fprintf(f,"/%s-ISOLatin1 exch definefont pop\n", tg_fontname2); 
    }

    if ( ( strcmp(tg_fontname1, tg_fontname3) != 0 ) 
	 && ( strcmp(tg_fontname2, tg_fontname3) != 0 ) )
    {
	fprintf(f,"/%s findfont\n", tg_fontname3); 
	fprintf(f,"dup length dict begin\n"); 
	fprintf(f," { 1 index /FID ne\n"); 
	fprintf(f," {def}\n"); 
	fprintf(f," {pop pop}\n"); 
	fprintf(f," ifelse\n"); 
	fprintf(f," } forall\n"); 
	fprintf(f," /Encoding ISOLatin	1Encoding def\n"); 
	fprintf(f," currentdict\n"); 
	fprintf(f,"end\n"); 
	fprintf(f,"/%s-ISOLatin1 exch definefont pop\n", tg_fontname3); 
    }

    fprintf(f,"/BOX\n");
    fprintf(f,"  {\n");
    fprintf(f,"  /y2 exch def /y1 exch def /x2 exch def /x1 exch def\n");
    fprintf(f,"  x1 y1 moveto\n");
    fprintf(f,"  x2 y1 lineto\n");
    fprintf(f,"  x2 y2 lineto\n");
    fprintf(f,"  x1 y2 lineto\n");
    fprintf(f,"  closepath\n");
    fprintf(f,"  %f setgray\n", tg_gray);
    fprintf(f,"  fill\n");
    fprintf(f,"  } def\n");

    fprintf(f,"/OUTLINE\n");
    fprintf(f,"  {\n");
    fprintf(f,"  /y2 exch def /y1 exch def /x2 exch def /x1 exch def\n");
    fprintf(f,"  x1 y1 moveto\n");
    fprintf(f,"  x2 y1 lineto\n");
    fprintf(f,"  x2 y2 lineto\n");
    fprintf(f,"  x1 y2 lineto\n");
    fprintf(f,"  closepath\n");
    fprintf(f,"  %f setgray\n", tg_lightgray);
    fprintf(f,"  fill\n");
    fprintf(f,"  } def\n");

    // In order to mark past days
    if (tg_showpast)
    {
      fprintf(f,"/PAST\n");
      fprintf(f,"  {\n");
      fprintf(f,"  /y2 exch def /y1 exch def /x2 exch def /x1 exch def\n");
      fprintf(f,"  /ymiddle y1 y2 add 2 div def\n");
      fprintf(f,"  x1 y1 ymiddle add 2 div moveto\n");
      fprintf(f,"  x2 y1 ymiddle add 2 div lineto\n");
      fprintf(f,"  x2 y2 ymiddle add 2 div lineto\n");
      fprintf(f,"  x1 y2 ymiddle add 2 div lineto\n");
      fprintf(f,"  closepath\n");
      fprintf(f,"  %f setgray\n", tg_pastgray);
      fprintf(f,"  fill\n");
      fprintf(f,"  } def\n");
    }

    // In order to mark vacations
    if (tg_showvacation)
    {
      fprintf(f,"/VACATION\n");
      fprintf(f,"  {\n");
      fprintf(f,"  /y2 exch def /y1 exch def /x2 exch def /x1 exch def\n");
      fprintf(f,"  x1 y1 moveto\n");
      fprintf(f,"  x2 y1 lineto\n");
      fprintf(f,"  x2 y2 lineto\n");
      fprintf(f,"  x1 y2 lineto\n");
      fprintf(f,"  closepath\n");
      fprintf(f,"  %f setgray\n", tg_vacationgray);
      fprintf(f,"  fill\n");
      fprintf(f,"  } def\n");

    }

    fprintf(f,"/FINISHED\n");
    fprintf(f,"  {\n");
    fprintf(f,"  /y2 exch def /y1 exch def /x2 exch def /x1 exch def\n");
    fprintf(f,"  gsave\n");
    fprintf(f,"  %f setgray\n", tg_white);
    fprintf(f,"  3 setlinewidth\n");
    fprintf(f,"  x1 y1 moveto\n");
    fprintf(f,"  x2 y2 lineto\n");
    fprintf(f,"  stroke\n");
    fprintf(f,"  grestore\n");
    fprintf(f,"  } def\n");

    fprintf(f,"/MBOX\n");
    fprintf(f,"  {\n");
    fprintf(f,"  /y2 exch def /y1 exch def /x2 exch def /x1 exch def\n");
    fprintf(f,"  /x3 x1 x2 add 2 div def\n");

    fprintf(f,"  /y3 y1 y2 add 2 div def\n");
    fprintf(f,"  x1 y1 moveto\n");
    fprintf(f,"  x2 y1 lineto\n");
    fprintf(f,"  x3 y3 lineto\n");
    fprintf(f,"  closepath\n");
    fprintf(f,"  %f setgray\n", tg_gray);
    fprintf(f,"  fill\n");
    fprintf(f,"  x1 y2 moveto\n");
    fprintf(f,"  x2 y2 lineto\n");
    fprintf(f,"  x3 y3 lineto\n");
    fprintf(f,"  closepath\n");
    fprintf(f,"  %f setgray\n", tg_gray);
    fprintf(f,"  fill\n");
    fprintf(f,"  } def\n");

    fprintf(f,"/%s-ISOLatin1 findfont %f scalefont setfont\n", tg_fontname1, tg_fontsize1);

    fprintf(f,"0 setgray\n");
    fprintf(f,"0.01 setlinewidth\n");

    fprintf(f,"%d %d moveto\n",x1,y1);
    fprintf(f,"%d %d lineto\n",x1,y2);
    fprintf(f,"%d %d lineto\n",x2,y2);
    fprintf(f,"%d %d lineto\n",x2,y1);
    fprintf(f,"stroke\n");

    return f;
}

/// Print the sorted list of tasks.  Used for debugging task sort order
void Project::printTaskList() 
{
    for ( int i = 0 ; i < nTasks() ; ++i ) 
    {
	TASK * t = mSortedTaskList[i];
	printf("Task id: %s, Assigned to: %s, Start date: %d\n",
	       t->id(), t->assigned()->id(), t->start());
    }
}


/// Display a TaskGraph
/// \param project Project to display
/// \param start first day of the graph
/// \param finish last day of the graph
/// \param filename Name of the file to be ceated/overwrited
void Reporter::TaskGraph(Project *project,
			 int start, int finish,
			 const char *filename)
{
    FILE *f;
    TASK *t;
    TASK *lastt = NULL;
    int i;
    int y;
    int x1=99999, y1=99999, x2=-99999, y2=-99999;
    int day;
    int daynum;
    int tb_x1, tb_y1, tb_x2, tb_y2;

    project->SortTasks( tg_sortbyresource );

    // printTaskList();  // Used for debugging     

    Debug("TaskGraph(%d/%s,%d/%s,%s)",
	  start,project->sDays(start),finish,project->sDays(finish),filename);

    Debug("start=%d finish=%d",start,finish);

    x1 = tg_tasklabel;
    x2 = MapX(project,finish+1,start,finish);
    // Affect 'y' coordinates to each tasks
    // First loop to compute the MIN/MAX 'y' coordinates
    y = tg_top;
    for( i=0; i<project->nTasks(); ++i )
    {
	t = project->sortedTask(i);
	if ( t->isVacation() )
	    continue;
	if( ! t->overlap(start, finish) )
	    continue;

	t->y2 = y;
	t->y1 = t->y2 - tg_height;
	y -= tg_space;
	y1 = MIN( y1, t->y1 );
	y2 = MAX( y2, t->y2 );
    }

    // Text at bottom occupies 2 tg_spaces, need to allow for decenders on text too 
    y1 -= static_cast<int>( tg_space + tg_fontsize3 / 4 );

    x1 -= tg_xborder;
    y1 -= tg_yborder;
    x2 += tg_xborder;
    y2 += tg_yborder;

    f = OpenTaskGraph(filename, x1, y1, x2, y2);

    // Second loop is used to actually draw the graph:
    y = tg_top;
    for( i=0; i<project->nTasks(); ++i )
    {
	t = project->sortedTask(i);
	if ( t->isVacation() )
	    continue;
	if( ! t->overlap(start, finish) )
	    continue;

	Debug("... (b) task %s: %d-%d", t->id(),t->start(),t->finish());

	t->x1 = MapX(project, t->start(), start, finish );
	t->x2 = MapX(project, t->finish(), start, finish ) + tg_width-1;
	t->y2 = y;
	t->y1 = t->y2 - tg_height;
	y -= tg_space;

        // markup vacations
        if (tg_showvacation)
        {
          for ( int ti = 0; ti < project->nTasks(); ++ti )
          {
            TASK* vt = project->sortedTask(ti);
            if (! vt->isVacation() )
                continue;
            if (! vt->overlap(start, finish) )
                continue;
            if ( strcmp(vt->assigned()->id(), t->assigned()->id() ) != 0 )
                continue;
 
            int v_x1 = MapX(project, vt->start(), start, finish );
            int v_x2 = MapX(project, vt->finish(), start, finish ) + tg_width;
            fprintf(f, "%d %d %d %d VACATION\n", v_x1, v_x2, t->y1, t->y2 );
          }
        }

	fprintf(f,"0 setgray\n");
	fprintf(f,"%d %d moveto\n", tg_tasklabel, t->y1 + tg_textup );
	if ( task_ids )
	    fprintf(f,"(%s %s: %s) show\n", t->id(), t->name(), t->assigned()->id());
	else
	    fprintf(f,"(%s: %s) show\n", t->name(), t->assigned()->id());
	fprintf(f,"%d %d moveto\n", x1, t->y1 );
	fprintf(f,"%d %d lineto stroke\n", x2, t->y1 );

	// Draw heavy line separating sections with different resources
	if (tg_sortbyresource) {
	    if ((lastt != NULL) &&
		(strcasecmp(t->assigned()->id(),lastt->assigned()->id()) != 0)) {

		fprintf(f,"3 setlinewidth\n");
		fprintf(f,"%d %d moveto\n", x1, t->y2+tg_height );
		fprintf(f,"%d %d lineto stroke\n", x2, t->y2+tg_height );
		fprintf(f,"0.01 setlinewidth\n");
	    }

	    lastt = t;
      
	}

	for (TaskTimeBlockIterator tb = t->begin_when(); tb != t->end_when(); tb++)
	{
	    int i;
	    Debug("... ... %s timeblock %s-%s type=%d",
		  t->id(), project->sDays(tb->start()), project->sDays(tb->finish()),
                  tb->type());
	    if( tb->finish() < start || tb->start() > finish )
		continue;

	    tb_x1 = MapX(project, tb->start(), start, finish );
	    tb_x2 = MapX(project, tb->finish(), start, finish ) + tg_width - 1;
	    tb_y2 = t->y2;
	    tb_y1 = t->y1;
	    fprintf(f,"%d %d %d %d OUTLINE\n", tb_x1, tb_x2, tb_y1, tb_y2);
	    for( i = tb->start(); i <= tb->finish(); ++i )
	    {
		tb_x1 = MapX(project, i, start, finish );
		fprintf(f,"%d %d %d %d BOX\n", tb_x1, tb_x1+tg_width, tb_y1, tb_y2);
		// Mark TimeBlock as PAST/DONE if
		// - type is WORK_DONE
		// - tg_showpast is required
		// - task is not completed (in which case it must be marked
		//   FINISHED
                if (tb->type() == TimeBlock::WORK_DONE &&
                    tg_showpast && t->percent_complete()<=1)
                {
		  fprintf(f,"%d %d %d %d PAST\n", tb_x1, tb_x1+tg_width, tb_y1, tb_y2);
                }
	    }
	}

	if( t->percent_complete()>1 )
	{
	    int dayNo, nDays=0;
	    int x1, x2;
	    for ( TaskTimeBlockIterator tb = t->begin_when(); 
		  tb != t->end_when(); tb++ )
	    {
		for ( dayNo = tb->start(); dayNo <= tb->finish() ; dayNo++ )
		{
		    if( t->DayBooked( dayNo ) )
		    {
			++nDays;
			if( (double)nDays/(double)t->fullduration() <=
			    0.01*t->percent_complete() )
			{ // still within the completion time
			    x1 = MapX(project, dayNo, start, finish );
			    x2 = x1 + tg_width-1;
			    fprintf(f,"%d %d %d %d FINISHED\n", x1+2, x2-2, t->y1+2, t->y2-2);
			}
			else
			{ // drawing past the complete time - quit early
			    break;
			}
		    } // only worry about a day if the task is booked for it
		} // scan every day in the time block
	    } // every time block in the task
	} // show hash marks for completed days.
    
	fprintf(f,"\n");
    }

    y -= tg_space;

    for( day=start; day<=finish; ++day )
    {

	daynum = Offset(project->nDays(day), project->nDays(start), project->nDays(finish));

	fprintf(f, "/%s-ISOLatin1 findfont %f scalefont setfont\n", tg_fontname2, tg_fontsize2);

	//print day labels unless requested not to do so.
	if (!tg_nodays) {

	    fprintf(f,"0 setgray %d %d moveto (%s) show\n",
		    tg_left + daynum * tg_width, y + tg_space,
		    project->dayNames[ Wday( project->tDays(day) ) ] );
	    fprintf(f,"%f setgray %d %d moveto 0 %d rlineto stroke\n",
		    tg_gray, tg_left + daynum * tg_width, y + tg_space, tg_space/2 );
		//if is set tg_daysofmonth the print days of month 
    		if (tg_daysofmonth)
		 fprintf(f,"0 setgray %d %d moveto (%d) show\n",
                 tg_left + daynum * tg_width, y + tg_space/2 + 3, Mday( project->tDays(day))); 
		// Draw a line all the way up to mark beginning of week
		if ( Wday( project->tDays(day)) == 1 ) {
			fprintf(f,"%f setgray %d %d moveto 0 %d rlineto stroke\n",
				tg_gray, tg_left + daynum * tg_width,
				y + tg_space, tg_top - (y + tg_space) );
		} else {
			fprintf(f,"%f setgray %d %d moveto 0 %d rlineto stroke\n",
				tg_gray, tg_left + daynum * tg_width,
				y + tg_space, tg_space/2 );
		}
	}
    
	// Display monthname if
	// - first day of the month
	//   + when first day is worked
	//   + when first day is not worked
	// - start day is 3 days (worked) before the end of the month
	//   (enougth place for writing the month name)
	if(Mday(project->tDays(day)) == 1 || 
	   (day > start &&
	    (Mday(project->tDays(day)) < Mday(project->tDays(day-1)))) ||
	   (day == start &&
	    Month(project->tDays(day)) == Month(project->tDays(day+3))))
	{
	  Debug("legend; day=%d yday=%d yday'=%d",
		day,project->tDays(day),project->tDays(day-1));
	  
	  fprintf(f, "/%s-ISOLatin1 findfont %f scalefont setfont\n",
		  tg_fontname3, tg_fontsize3);
	  daynum = Offset(project->nDays(day) - Mday( project->tDays(day) ) + 1,
			  project->nDays(start), project->nDays(finish));
	  fprintf(f, "%f setgray\n",tg_mlgray);
	  fprintf(f, "%d %d moveto %d %d lineto stroke\n",
		  tg_left + daynum * tg_width, y1,
		  tg_left + daynum * tg_width, y2);
	  fprintf(f, "0 setgray %d %d moveto (%s) show\n",
		  tg_left + daynum * tg_width, y,
		  monthNames[ Month( project->tDays(day) ) ] );
	}
    }

    fprintf(f,"showpage\n");

    fclose(f);
}

void Reporter::UtilGraph(Project *project, char *filename)
{
    int x1=99999, y1=99999, x2=-99999, y2=-99999;

    project->SortTasks(tg_sortbyresource);

    FILE * f = OpenOutputFile(filename);

    int resNum = 0;
    for ( Project::RPLCI rl = project->beginResourceList() ; 
	  rl != project->endResourceList() ; rl++ )
    {
	RESOURCE * r = *rl;
	int y = 72*1 + 72/6 * resNum;
	int x = 72;

	x1 = MIN(x1,x); x2=MAX(x2,x); y1 = MIN(y1,y); y2=MAX(y2,y);
	x += 72;
	x1 = MIN(x1,x); x2=MAX(x2,x);
	for ( int dayNum = 0 ; dayNum < MAX_TIME ; ++dayNum )
	{
	    x += 4;
	    if ( r->IsUsed( dayNum ) )
	    {
		x1 = MIN(x1,x); x2=MAX(x2,x+3);
	    }
	}
	resNum++;
    }

    fprintf(f,"%%!PS-Adobe-2.0 EPSF-2.0\n"); 
    fprintf(f,"%%%%BoundingBox: %d %d %d %d\n",x1,y1,x2,y2);

    fprintf(f,"/%s findfont\n", tg_fontname1); 
    fprintf(f,"dup length dict begin\n"); 
    fprintf(f," { 1 index /FID ne\n"); 
    fprintf(f," {def}\n"); 
    fprintf(f," {pop pop}\n"); 
    fprintf(f," ifelse\n"); 
    fprintf(f," } forall\n"); 
    fprintf(f," /Encoding ISOLatin1Encoding def\n"); 
    fprintf(f," currentdict\n"); 
    fprintf(f,"end\n"); 
    fprintf(f,"/%s-ISOLatin1 exch definefont pop\n", tg_fontname1); 

    if ( strcmp(tg_fontname1, tg_fontname2) != 0 )
    {
	fprintf(f,"/%s findfont\n", tg_fontname2); 
	fprintf(f,"dup length dict begin\n"); 
	fprintf(f," { 1 index /FID ne\n"); 
	fprintf(f," {def}\n"); 
	fprintf(f," {pop pop}\n"); 
	fprintf(f," ifelse\n"); 
	fprintf(f," } forall\n"); 
	fprintf(f," /Encoding ISOLatin	1Encoding def\n"); 
	fprintf(f," currentdict\n"); 
	fprintf(f,"end\n"); 
	fprintf(f,"/%s-ISOLatin1 exch definefont pop\n", tg_fontname2); 
    }

    if ( ( strcmp(tg_fontname1, tg_fontname3) != 0 ) 
	 && ( strcmp(tg_fontname2, tg_fontname3) != 0 ) )
    {
	fprintf(f,"/%s findfont\n", tg_fontname3); 
	fprintf(f,"dup length dict begin\n"); 
	fprintf(f," { 1 index /FID ne\n"); 
	fprintf(f," {def}\n"); 
	fprintf(f," {pop pop}\n"); 
	fprintf(f," ifelse\n"); 
	fprintf(f," } forall\n"); 
	fprintf(f," /Encoding ISOLatin	1Encoding def\n"); 
	fprintf(f," currentdict\n"); 
	fprintf(f,"end\n"); 
	fprintf(f,"/%s-ISOLatin1 exch definefont pop\n", tg_fontname3); 
    }

    fprintf(f, "/%s-ISOLatin1 findfont %f scalefont setfont\n", tg_fontname1, tg_fontsize1);

    fprintf(f,"/BOX { 3 0 rlineto 0 3 rlineto -3 0 rlineto closepath stroke } def\n");

    resNum = 0;
    for ( Project::RPLCI rl = project->beginResourceList() ; 
	      rl != project->endResourceList() ; rl++ )
    {
	RESOURCE * r = *rl;
	int y = 72*1 + 72/6 * resNum;
	int x = 72;
	fprintf(f,"0 setgray %d %d moveto (%s) show\n", x, y, r->id());
	x += 72;
	for ( int dayNum = 0 ; dayNum < MAX_TIME ; ++dayNum )
	{
	    x += 4;
	    if ( r->IsUsed( dayNum ) )
	    {
		fprintf(f,"%d %d moveto BOX\n", x, y);
	    }
	}
	resNum++;
    }

    fprintf(f,"showpage\n");

    fclose(f);
}
