/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file myx_gc_layer.cpp 
 * @brief Implementation of the GC layer class.
 * 
 */

#ifdef _WINDOWS
  #define  WIN32_LEAN_AND_MEAN
  #include <windows.h>
#endif // ifdef _WINDOWS

#include <GL/gl.h>
#include <GL/glu.h>
#include <math.h>

#include "myx_gc_figure.h"
#include "myx_gc_layer.h"
#include "myx_gc_gl_helper.h"

//----------------- CLayer ---------------------------------------------------------------------------------------------

CLayer::CLayer(CGenericCanvas* Owner)
{
  FUpdateCount = 0;
  FCanvas = Owner;

  // Initialize with useful values.
  FScaling[0] = 1;
  FScaling[1] = 1;
  FScaling[2] = 1;

  FTranslation[0] = 0;
  FTranslation[1] = 0;
  FTranslation[2] = 0;

  FDirty = true;
  FVisible = true;
  FEnabled = true;
}

//----------------------------------------------------------------------------------------------------------------------

CLayer::~CLayer(void)
{
  FUpdateCount++;

  if ((FCanvas != NULL) && (!FCanvas->IsUpdating()))
    FCanvas->RemoveLayer(this);
  Clear();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Applies the layer's transformations for rendering, feedback etc.
 */
void CLayer::ApplyTransformations()
{
  glTranslated(FTranslation[0], FTranslation[1], FTranslation[2]);
  glScaled(FScaling[0], FScaling[1], FScaling[2]);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Triggers the error checking of the canvas.
 */
void CLayer::CheckError(void)
{
  FCanvas->CheckError();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Marks the display list for this layer as invalid, hence it will be recreated next time Validate is called.
 * If a list already exists then it is freed.
 */
void CLayer::MakeDirty(void)
{
  if (!FDirty)
  {
    FDirty = true;
    if ((FCanvas != NULL) && (!FCanvas->IsUpdating()))
      FCanvas->Invalidate();
  };
}

//----------------------------------------------------------------------------------------------------------------------

/*
 * Helper method to determine the transformed vertices of the given figure instance. The layer applies its own
 * transformations and only renders the figure instance.
 *
 * @param Instance The figure instance for which feedback data is requested.
 */
void CLayer::RenderFeedback(CFigureInstance* Instance)
{
  ApplyTransformations();
  Instance->Render();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Renders layer content that is not determined by figure instances. This method might be overridden by descendants.
 */
void CLayer::RenderLayerContent(void)
{
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Creates the display list of this figure (and all child figures) if necessary.
 */
void CLayer::Validate(void)
{
  if (FDirty)
  {
    FDirty = false;

    ValidateLayerContent();

    // Give the instances the opportunity to trigger validation of their associated figures.
    for (CFigureInstanceList::iterator Iterator = FInstances.begin(); Iterator != FInstances.end(); ++Iterator)
      (*Iterator)->Validate();

    CheckError();
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Prepares layer content that is not determined by figure instances. This method might be overridden by descendants.
 */
void CLayer::ValidateLayerContent(void)
{
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Adds the given figure instance to the end of the instance list. If Instance belongs to another layer currently 
 * it is removed from the other's instance list first.
 *
 * @param Instance The figure instance to add.
 */
void CLayer::AddInstance(CFigureInstance* Instance)
{
  if (Instance->FLayer != NULL)
    Instance->FLayer->RemoveInstance(Instance);
  FInstances.insert(Instance);
  Instance->FLayer = this;
  MakeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * @see CGenericCanvas.BeginUpdate()
 */
void CLayer::BeginUpdate(void)
{
  FUpdateCount++;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Removes all figure instances from this layer.
 */
void CLayer::Clear(void)
{
  for (CFigureInstanceList::iterator Iterator = FInstances.begin(); Iterator != FInstances.end(); ++Iterator)
  {
    CFigureInstance* Instance = *Iterator;
    Instance->FLayer = NULL;
    if (Instance->Selected())
      FCanvas->RemoveFromSelection(Instance);
    delete Instance;
  };
  FInstances.clear();
  MakeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Creates a new instance for the given figure and adds it to this layer.
 *
 * @param Figure The figure for which the instance is to be created.
 * @return A new figure instance.
 */
CFigureInstance* CLayer::CreateInstance(CFigure* Figure)
{
  CFigureInstance* Instance = new CFigureInstance(this, Figure);

  return Instance;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * @see CGenericCanvas.EndUpdate()
 */
void CLayer::EndUpdate(void)
{
  if (FUpdateCount > 0)
    FUpdateCount++;
  if (FUpdateCount == 0)
    MakeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

CGenericCanvas* CLayer::GetCanvas(void)
{
  return FCanvas;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Returns the visibility state of the layer.
 *
 * @return true if the layer is visible in the canvas, otherwise false.
 */
bool CLayer::GetVisible(void)
{
  return FVisible;
}

//----------------------------------------------------------------------------------------------------------------------

bool CLayer::IsUpdating(void)
{
  return FUpdateCount != 0;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Release function for external access to free this layer. It will just delete itself.
 */
void CLayer::Release(void)
{
  delete this;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Removes the given figure instance from the instance list if it is currently there.
 * No error is raised if the instance does not belong to this layer.
 *
 * @param Instance The instance to be removed.
 */
void CLayer::RemoveInstance(CFigureInstance* Instance)
{
  for (CFigureInstanceList::iterator Iterator = FInstances.begin(); Iterator != FInstances.end(); ++Iterator)
    if (*Iterator == Instance)
    {
      FInstances.erase(Iterator);
      Instance->FLayer = NULL;
      MakeDirty();
      break;
    };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Checks the validity of the figure display list and executes it.
 */
void CLayer::Render(void)
{
  if (FVisible && (FUpdateCount == 0)) 
  {
    if (FDirty)
      Validate();

    ApplyTransformations();
    for (CFigureInstanceList::iterator Iterator = FInstances.begin(); Iterator != FInstances.end(); ++Iterator)
      (*Iterator)->Render();

    RenderLayerContent();
    CheckError();
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Scales the layer by the amount given in Factor. If Accumulative is true then the new scale factors are multiplied
 * with the existing values. This version of Scale uses single float values as parameters.
 *
 * @param Sx Scale factor for the x axis.
 * @param Sy Scale factor for the y axis.
 * @param Sz Scale factor for the z axis.
 * @param Accumulative If true then the given values are added to any existing values otherwiese they are used as given.
 */
void CLayer::Scale(double Sx, double Sy, double Sz, bool Accumulative)
{
  if (Accumulative)
  {
    FScaling[0] += Sx;
    FScaling[1] += Sy;
    FScaling[2] += Sz;
  }
  else
  {
    FScaling[0] = Sx;
    FScaling[1] = Sy;
    FScaling[2] = Sz;
  };
  MakeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/** 
 * Scales the layer by the amount given in Factor. If Accumulative is true then the new scale factors are multiplied
 * with the existing values. This version of Scale uses an array of values in the parameter list.
 *
 * @param Factor An array of 3 scale values, one for each axis.
 * @param Accumulative If true then the given values are added to any existing values otherwiese they are used as given.
 */
void CLayer::ScaleV(const double Factor[3], bool Accumulative)
{
  if (Accumulative)
  {
    FScaling[0] += Factor[0];
    FScaling[1] += Factor[1];
    FScaling[2] += Factor[2];
  }
  else
  {
    FScaling[0] = Factor[0];
    FScaling[1] = Factor[1];
    FScaling[2] = Factor[2];
  };
  MakeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Moves the layer by the amount given in Tx, Ty and Tz. If Accumulative is true then the new translation factors 
 * are multiplied with the existing values. This version of Translate uses an array for the values in the parameter list.
 *
 * @param Tx Scale factor for the x axis.
 * @param Ty Scale factor for the y axis.
 * @param Tz Scale factor for the z axis.
 * @param Accumulative If true then the given values are added to any existing values otherwiese they are used as given.
 */
void CLayer::Translate(double Tx, double Ty, double Tz, bool Accumulative)
{
  if (Accumulative)
  {
    FTranslation[0] += Tx;
    FTranslation[1] += Ty;
    FTranslation[2] += Tz;
  }
  else
  {
    FTranslation[0] = Tx;
    FTranslation[1] = Ty;
    FTranslation[2] = Tz;
  };
  MakeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Moves the layer by the amount given in Factor. If Accumulative is true then the new translation factors are multiplied
 * with the existing values. This version of Translate uses an array for the values in the parameter list.
 *
 * @param Factor An array of translation values, for each axis one.
 * @param Accumulative If true then the given values are added to any existing values otherwiese they are used as given.
 */
void CLayer::TranslateV(const double Factor[3], bool Accumulative)
{
  if (Accumulative)
  {
    FTranslation[0] += Factor[0];
    FTranslation[1] += Factor[1];
    FTranslation[2] += Factor[2];
  }
  else
  {
    FTranslation[0] = Factor[0];
    FTranslation[1] = Factor[1];
    FTranslation[2] = Factor[2];
  };
  MakeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets the layer's enabled state.
 *
 * @param IsEnabled Set it to true if you want the layer to be visible.
 */
void CLayer::SetEnabled(bool IsEnabled)
{
  if (FEnabled != IsEnabled)
  {
    FEnabled = IsEnabled;
    FCanvas->Invalidate();
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets the layer's visibility state.
 *
 * @param IsVisible Set it to true if you want the layer to be visible.
 */
void CLayer::SetVisible(bool IsVisible)
{
  if (FVisible != IsVisible)
  {
    FVisible = IsVisible;
    FCanvas->Invalidate();
  };
}

//----------------- CGridLayer -----------------------------------------------------------------------------------------

CGridLayer::CGridLayer(CGenericCanvas* Owner): CLayer(Owner)
{
}

//----------------------------------------------------------------------------------------------------------------------

void CGridLayer::RenderLayerContent(void)
{
  GLint ViewInfo[4];
  glGetIntegerv(GL_VIEWPORT, ViewInfo);
  double Left = ViewInfo[0];
  double Right = ViewInfo[0] + ViewInfo[2];
  double Top = ViewInfo[1];
  double Bottom = ViewInfo[1] + ViewInfo[3];

  // Disable antialiasing temporarily.
  glPushAttrib(GL_ENABLE_BIT);
  glDisable(GL_LINE_SMOOTH);
  glColor4d(0, 0, 0, 0.2);

  glBegin(GL_LINES);
  for (double Y = Top; Y < Bottom; Y += 30)
  {
    glVertex2d(Left, Y);
    glVertex2d(Right, Y);
  };

  for (double X = Left; X < Right; X += 30)
  {
    glVertex2d(X, Top);
    glVertex2d(X, Bottom);
  };

  glEnd();
  glPopAttrib();
}

/*
void CGLView::DrawNet2(double xsize, double zsize, double xstep, double
zstep)
{

 file://adjust xsize , zsize
 if (xstep<0.0000001 || zstep<0.0000001) return; file://avoid too small grid
 int countx= (int)xsize/xstep;
 int countz= (int)zsize/zstep;

 file://okay
 xsize= (double)countx * xstep;
 zsize= (double)countz * zstep;



 glPushMatrix();

 double x,z;

   glEnable (GL_LINE_STIPPLE);
   glLineStipple (1, 0x1c47);  //  dotted
   glLineWidth(1); //thin line

   minor line
 glBegin(GL_LINES);
 for (x=-xsize; x<xsize+xstep; x+=xstep) {
  glVertex3d(x, 0, -zsize);
  glVertex3d(x, 0,  zsize);

 }

 for (z=-zsize; z<zsize+zstep; z+=zstep) {
  glVertex3d(-xsize, 0, z);
  glVertex3d( xsize, 0, z);
 }

 glEnd();

 main line
 glDisable(GL_LINE_STIPPLE);
 int c=0;
 glBegin(GL_LINES);
 c=0;
 for (x=0; x<=xsize; x+=xstep) {
  if ( c++%5 == 0 ) {
  glVertex3d(x, 0, -zsize);
  glVertex3d(x, 0,  zsize);
  }
 }

 c=0;
 for (x=0; x>=-xsize; x-=xstep) {
  if ( c++%5 == 0 ) {
  glVertex3d(x, 0, -zsize);
  glVertex3d(x, 0,  zsize);
  }
 }

 c=0;
 for (z=0; z<=zsize; z+=zstep) {
  if (c++%5==0) {
   glVertex3d(-xsize, 0, z);
   glVertex3d( xsize, 0, z);
  }
 }

 c=0;
 for (z=0; z>=-zsize; z-=zstep) {
  if (c++%5==0) {
   glVertex3d(-xsize, 0, z);
   glVertex3d( xsize, 0, z);
  }
 }

 glEnd();



 major line
 glLineWidth(2);
 glBegin(GL_LINES);
 glVertex3d(0, 0, -zsize);
 glVertex3d(0, 0,  zsize);
 glVertex3d(-xsize, 0, 0);
 glVertex3d( xsize, 0, 0);
 glLineWidth(1);

 glEnd();


 glPopMatrix();

}
*/
//----------------- CFeedbackLayer ------------------------------------------------------------------------------------

CFeedbackLayer::CFeedbackLayer(CGenericCanvas* Owner): CLayer(Owner)
{
  FHandleSize = 5;
  FRubberband = 0;
  FStates = 0;
  FSelectionDecoration = 0;
}

//----------------------------------------------------------------------------------------------------------------------

CFeedbackLayer::~CFeedbackLayer(void)
{
  ClearSelection();
  if (FSelectionDecoration != 0)
    glDeleteLists(FSelectionDecoration, 1);
  if (FRubberband != 0)
    glDeleteLists(FRubberband, 1);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Creates the display list for the selection decoration, which is shared among all selection entries.
 */
void CFeedbackLayer::CreateSelectionDecoration(void)
{
  GLubyte Color[4];
  if (!ColorByName("Highlight", Color))
  {
    // If the system's highlight color could not be found then use a default one.
    Color[0] = 64;
    Color[1] = 64;
    Color[2] = 128;
  };

  FSelectionDecoration = glGenLists(1);
  glNewList(FSelectionDecoration, GL_COMPILE);

  // The interior.
  Color[3] = 30;
  glColor4ubv(Color);
  glRectd(0, 1, 1, 0);

  glPushAttrib(GL_LINE_BIT);
  glDisable(GL_LINE_SMOOTH);

  // The border.
  Color[3] = 200;
  glColor4ubv(Color);
  glBegin(GL_LINE_LOOP);
    glVertex2d(0, 1);
    glVertex2d(1, 1); 
    glVertex2d(1, 0);
    glVertex2d(0, 0); 
  glEnd();

  // The handles.
  Color[3] = 100;
  glColor4ubv(Color);
  glPointSize(FHandleSize);
  glBegin(GL_POINTS);
    glVertex2d(0, 1);
    glVertex2d(0.5, 1);
    glVertex2d(1, 1);

    glVertex2d(0, 0.5);
    glVertex2d(1, 0.5);

    glVertex2d(0, 0);
    glVertex2d(0.5, 0);
    glVertex2d(1, 0);
  glEnd();

  glPopAttrib();

  glEndList();

  CheckError();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Helper method to add a figure instance to the selection list. No change event is triggered.
 *
 * @param Instance The instance to add.
 * @return If the instance was added (because it wasn't already there) then true is returned, otherwise false.
 */
bool CFeedbackLayer::InternalAddToSelection(CFigureInstance* Instance)
{
  bool Result = !Instance->FSelected;
  if (Result)
  {
    Instance->FSelected = true;
    TSelectionEntry* Entry = new TSelectionEntry;
    Entry->Instance = Instance;
    Entry->Dirty = true;
    FSelection[Instance] = Entry;
  };

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Helper method to remove a figure instance from the selection list. No change event is triggered.
 *
 * @param Instance The instance to add.
 */
void CFeedbackLayer::InternalRemoveFromSelection(CFigureInstance* Instance)
{
  if (Instance->FSelected)
  {
    Instance->FSelected = false;
    CSelectionIterator Iterator = FSelection.find(Instance);
    if (Iterator != FSelection.end())
    {
      delete Iterator->second;
      FSelection.erase(Iterator);
    };
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Renders the decorations for all figure instances that are currently selected.
 */
void CFeedbackLayer::RenderLayerContent(void)
{
  for (CSelectionIterator Iterator = FSelection.begin(); Iterator != FSelection.end(); Iterator++)
  {
    TSelectionEntry* Entry = Iterator->second;
    glPushMatrix();
    glTranslated(Entry->Bounds.Left, Entry->Bounds.Bottom, 1);
    glScaled(Entry->Bounds.Right - Entry->Bounds.Left, Entry->Bounds.Top - Entry->Bounds.Bottom, 1);
    glCallList(FSelectionDecoration);
    glPopMatrix();
  };
  
  if ((FStates & GC_SLSTATE_RUBBERBAND) != 0)
  {
    // In order to avoid backface culling if the coordinates of the rubber band do not form a counter-clock-wise
    // face we simply disable face culling for the moment.
    glPushAttrib(GL_ENABLE_BIT);
    glDisable(GL_CULL_FACE);
    glTranslated(FRubberbandBounds.Left, FRubberbandBounds.Bottom, 0);
    glScaled(FRubberbandBounds.Right - FRubberbandBounds.Left, FRubberbandBounds.Top - FRubberbandBounds.Bottom, 1);
    glCallList(FRubberband);
    glPopAttrib();

  };

  CheckError();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Creates display lists for all invalid decorations.
 */
void CFeedbackLayer::ValidateLayerContent(void)
{
  if (FSelectionDecoration == 0)
    CreateSelectionDecoration();

  for (CSelectionIterator Iterator = FSelection.begin(); Iterator != FSelection.end(); Iterator++)
    if (Iterator->second->Dirty)
    {
      Iterator->second->Dirty = false;

      // Update the bounding box of the figure instance.
      TBounds Bounds;
      Iterator->second->Instance->GetBounds(&Bounds);

      // Give a small border around the figure instance.
      Bounds.Left -= 0.1f;
      Bounds.Top += 0.1f;
      Bounds.Right += 0.1f;
      Bounds.Bottom -= 0.1f;
      Iterator->second->Bounds = Bounds;
    };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Adds the given figure instance to the current selection.
 *
 * @param Instance The instance to be added to the selection. If it is already in the set it won't be added again.
 */
void CFeedbackLayer::AddToSelection(CFigureInstance* Instance)
{
  if (InternalAddToSelection(Instance))
  {
    GetCanvas()->Change(Instance, GC_CHANGE_SELECTION_ADD);
    if (!IsUpdating())
      MakeDirty();
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Removes all figure instances from the selection set, making it empty.
 */
void CFeedbackLayer::ClearSelection(void)
{
  if (FSelection.size() > 0)
  {
    for (CSelectionIterator Iterator = FSelection.begin(); Iterator != FSelection.end(); Iterator++)
    {
      Iterator->second->Instance->FSelected = false;
      delete Iterator->second;
    };

    FSelection.clear();
    GetCanvas()->Change(NULL, GC_CHANGE_SELECTION_CLEAR);
    if (!IsUpdating())
      MakeDirty();
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Retrieves the value of the given property, if it is a property of this class.
 *
 * @param Property The property to retrieve.
 * @param Value [out] The value of the property.
 * @return True if the asked property is known in this class, false otherwise (in this case Value is not changed).
 * @note Implicit datatype conversion is performed if datatypes of property and Value parameter do not match.
 *       This can result in lost precision or even conversion errors. So make sure to use the right datatype for the call.
 */
bool CFeedbackLayer::GetProperty(TProperty Property, double& Value)
{
  bool Result = true;
  switch (Property)
  {
    case GC_PROPERTY_HANDLE_SIZE:
      {
        Value = FHandleSize;
        break;
      };
    default:
      {
        Result = false;
        break;
      };
  };

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Retrieves the value of the given property, if it is a property of this class.
 *
 * @param Property The property to retrieve.
 * @param Value [out] The value of the property.
 * @return True if the asked property is known in this class, false otherwise (in this case Value is not changed).
 * @note Implicit datatype conversion is performed if datatypes of property and Value parameter do not match.
 *       This can result in lost precision or even conversion errors. So make sure to use the right datatype for the call.
 */
bool CFeedbackLayer::GetProperty(TProperty Property, int& Value)
{
  bool Result = true;
  switch (Property)
  {
    case GC_PROPERTY_HANDLE_SIZE:
      {
        Value = ROUND(FHandleSize);
        break;
      };
    default:
      {
        Result = false;
        break;
      };
  };

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Determines whether the given position corresponds to any of the parts (body, handles) of a selection decoration.
 * This test is quite fast and can be used for cursor feedback and such.
 * The method needs correctly set modelview and projection matrices on enter in order to convert mouse coordinates
 * correctly.
 *
 * @param X The horizontal mouse coordinate in layer coordinates.
 * @param Y The vertical mouse coordinate in layer coordinates.
 * @return One of the direction flags.
 */
TGCDirection CFeedbackLayer::GetSelectionInfo(double X, double Y)
{
  TGCDirection Result = GC_SI_NONE;

  // Convert window coordinates into object (scene) coordinates.
  ApplyTransformations();

  GLint Viewport[4];
  glGetIntegerv(GL_VIEWPORT, Viewport);

  GLdouble ModelviewMatrix[16];
  glGetDoublev(GL_MODELVIEW_MATRIX, ModelviewMatrix);

  GLdouble ProjectionMatrix[16];
  glGetDoublev(GL_PROJECTION_MATRIX, ProjectionMatrix);

  double LocalX, LocalY, LocalZ;
  gluUnProject(X, Viewport[3] - Y, 0, ModelviewMatrix, ProjectionMatrix, Viewport, &LocalX, &LocalY, &LocalZ);

  // Now try to find a decoration that is located at the given position.
  // We examine only those that are not dirty. Dirty decorations cannot be in view.
  // Iteration happens backwards to find decorations on top earlier than others.
  for (CSelectionIteratorReverse Iterator = FSelection.rbegin(); Iterator != FSelection.rend(); Iterator++)
  {
    if (!Iterator->second->Dirty)
    {
      TBounds Bounds = Iterator->second->Bounds;
      if ((LocalX >= Bounds.Left - 0.4) && (LocalX <= Bounds.Right + 0.4) &&
        (LocalY <= Bounds.Top + 0.4) && (LocalY >= Bounds.Bottom - 0.4))
      {
        // Found a decoration. Check if a handle is hit.
        Result = GC_SI_ON_OBJECT;

        bool LeftColumn = ::fabs(Bounds.Left - LocalX) <= 0.8;
        bool MiddleColumn = ::fabs((Bounds.Left + Bounds.Right) / 2 - LocalX) <= 0.8;
        bool RightColumn = ::fabs(Bounds.Right - LocalX) <= 0.8;
        bool TopRow = ::fabs(Bounds.Top - LocalY) <= 0.8;
        bool MiddleRow = ::fabs((Bounds.Top + Bounds.Bottom) / 2 - LocalY) <= 0.8;
        bool BottomRow = ::fabs(Bounds.Bottom - LocalY) <= 0.8;

        if (LeftColumn)
        {
          if (TopRow)
            Result = GC_SI_NORTH_WEST;
          else
            if (MiddleRow)
              Result = GC_SI_WEST;
            else
              if (BottomRow)
                Result = GC_SI_SOUTH_WEST;
        }
        else
          if (MiddleColumn)
          {
            if (TopRow)
              Result = GC_SI_NORTH;
            else
              if (BottomRow)
                Result = GC_SI_SOUTH;
          }
          else
            if (RightColumn)
            {
              if (TopRow)
                Result = GC_SI_NORTH_EAST;
              else
                if (MiddleRow)
                  Result = GC_SI_EAST;
                else
                  if (BottomRow)
                    Result = GC_SI_SOUTH_EAST;
            };

        break;
      };
    };
  };

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Invalidates the selection decoration of the given Instance (or all instances if Instance is NULL) so they are recomputed
 * next time the selection layer draws them.
 *
 * @param Instance The figure instance whose bounds need recomputation. If this parameter is NULL then all bounds are invalidated.
 */
void CFeedbackLayer::InvalidateBounds(CFigureInstance* Instance)
{
  if (Instance == NULL)
  {
    for (CSelectionIterator Iterator = FSelection.begin(); Iterator != FSelection.end(); Iterator++)
      Iterator->second->Dirty = true;
  }
  else
    if (Instance->FSelected)
    {
      CSelectionIterator Iterator = FSelection.find(Instance);
      if (Iterator != FSelection.end())
        Iterator->second->Dirty = true;
    };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Removes the given figure instance from the current selection.
 *
 * @param Instance The instance to be removed. If it isn't actually selected then nothing happens.
 */
void CFeedbackLayer::RemoveFromSelection(CFigureInstance* Instance)
{
  if (Instance->FSelected)
  {
    InternalRemoveFromSelection(Instance);
    GetCanvas()->Change(Instance, GC_CHANGE_SELECTION_REMOVE);
    if (!IsUpdating())
      MakeDirty();
  };
}

//----------------------------------------------------------------------------------------------------------------------

void CFeedbackLayer::ResizeFiguresStart(int X, int Y, TGCDirection Direction)
{
}

//----------------------------------------------------------------------------------------------------------------------

void CFeedbackLayer::ResizeFiguresStop(void)
{
}

//----------------------------------------------------------------------------------------------------------------------

void CFeedbackLayer::ResizeFiguresTo(int X, int Y)
{
}

//----------------------------------------------------------------------------------------------------------------------

/**
  * When in rubber band mode then this function extends the current rubber band rectangle from the start point to
  * the given coordinates and handles selection/deselection of figure instances.
  *
  * @param X The x coordinate of the new corner.
  * @param y The y coordinate of the new corner.
  * @param Action Determines if and how figure instance selection is to be handled. See TRBSelectionAction for
  *               a description of the various modes.
 */
void CFeedbackLayer::RubberbandResize(int X, int Y, TRBSelectionAction Action)
{
  if ((FStates & GC_SLSTATE_RUBBERBAND) != 0)
  {
    // Convert window coordinates into object (scene) coordinates.
    ApplyTransformations();

    GLint Viewport[4];
    glGetIntegerv(GL_VIEWPORT, Viewport);

    GLdouble ModelviewMatrix[16];
    glGetDoublev(GL_MODELVIEW_MATRIX, ModelviewMatrix);

    GLdouble ProjectionMatrix[16];
    glGetDoublev(GL_PROJECTION_MATRIX, ProjectionMatrix);

    double LocalX, LocalY, LocalZ;
    gluUnProject(X, Viewport[3] - Y, 0, ModelviewMatrix, ProjectionMatrix, Viewport, &LocalX, &LocalY, &LocalZ);

    TBounds OldBounds = SortBounds(FRubberbandBounds);
    FRubberbandBounds.Right = (float) LocalX;
    FRubberbandBounds.Bottom = (float) LocalY;
    TBounds NewBounds = SortBounds(FRubberbandBounds);

    bool HaveSelectionRemoved = false;
    bool HaveSelectionAdded = false;

    /*
    if (Action != GC_RBACTION_NONE)
    {
      CFigureInstanceEnumerator* Enumerator = GetCanvas()->GetFigureInstanceEnumerator();
      Enumerator->Reset();
      switch (Action)
      {
        case GC_RBACTION_SELECT:
          {
            // Select all figure instances if they intersect. Leave all others alone.
            while (Enumerator->HasNext())
            {
              CFigureInstance* Instance = Enumerator->Next();
              TBounds InstanceBounds;
              Instance->GetBounds(&InstanceBounds);
              if (!BoundsAreEmpty(InstanceBounds))
              {
                if (BoundsIntersect(NewBounds, InstanceBounds))
                {
                  InternalAddToSelection(Instance);
                  HaveSelectionAdded = true;
                };
              };
            };
            if (HaveSelectionAdded)
              GetCanvas()->Change(NULL, GC_CHANGE_SELECTION_ADD);

            break;
          };
        case GC_RBACTION_SELECT_REMOVE:
          {
            // Select figure instances, which intersect, unselect all others.
            while (Enumerator->HasNext())
            {
              CFigureInstance* Instance = Enumerator->Next();
              TBounds InstanceBounds;
              Instance->GetBounds(&InstanceBounds);
              if (!BoundsAreEmpty(InstanceBounds))
              {
                if (Instance->FSelected)
                {
                  // Instance is selected. See if it is still within the rubberband bounds.
                  // Remove it from the selection list if not.
                  if (!BoundsIntersect(NewBounds, InstanceBounds))
                  {
                    InternalRemoveFromSelection(Instance);
                    HaveSelectionRemoved = true;
                  };
                }
                else
                {
                  // Instance is not selected. Add it to the current selection if its bounds
                  // fall within the rubberband bounds.
                  if (BoundsIntersect(NewBounds, InstanceBounds))
                  {
                    InternalAddToSelection(Instance);
                    HaveSelectionAdded = true;
                  };
                };
              };
            };

            if (HaveSelectionRemoved)
              GetCanvas()->Change(NULL, GC_CHANGE_SELECTION_REMOVE);

            if (HaveSelectionAdded)
              GetCanvas()->Change(NULL, GC_CHANGE_SELECTION_ADD);

            break;
          };
        case GC_RBACTION_TOGGLE:
          {
            // Switch figure instance selection state if their bounds intersect either
            // the new rubber band bounds or the old, but not both.
            while (Enumerator->HasNext())
            {
              CFigureInstance* Instance = Enumerator->Next();
              TBounds InstanceBounds;
              Instance->GetBounds(&InstanceBounds);

              if (!BoundsAreEmpty(InstanceBounds))
              {
                if (BoundsIntersect(NewBounds, InstanceBounds) != BoundsIntersect(OldBounds, InstanceBounds))
                {
                  if (Instance->FSelected)
                  {
                    InternalRemoveFromSelection(Instance);
                    HaveSelectionRemoved = true;
                  }
                  else
                  {
                    InternalAddToSelection(Instance);
                    HaveSelectionAdded = true;
                  };
                };
              };
            };

            if (HaveSelectionRemoved)
              GetCanvas()->Change(NULL, GC_CHANGE_SELECTION_REMOVE);

            if (HaveSelectionAdded)
              GetCanvas()->Change(NULL, GC_CHANGE_SELECTION_ADD);

            break;
          };
      };
      delete Enumerator;
    };
    */
    if (HaveSelectionAdded || HaveSelectionRemoved)
      MakeDirty();
    else
      GetCanvas()->Invalidate();
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Starts rubber banding if none is active currently. Otherwise it does nothing.
 *
 * @param Style Determines the visible style of the rubber band.
 * @param X The x coordinate of the start point in window coordinates.
 * @param X The y coordinate of the start point in window coordinates.
 * @param RemoveSelection If true then the current selection will be cleared.
 */
void CFeedbackLayer::RubberbandStart(TRubberbandStyle Style, int X, int Y, bool RemoveSelection)
{
  if ((FStates & GC_SLSTATE_RUBBERBAND) == 0)
  {
    FStates |= GC_SLSTATE_RUBBERBAND;

    if (RemoveSelection)
      ClearSelection();

    // Convert window coordinates into object (scene) coordinates.
    ApplyTransformations();

    GLint Viewport[4];
    glGetIntegerv(GL_VIEWPORT, Viewport);

    GLdouble ModelviewMatrix[16];
    glGetDoublev(GL_MODELVIEW_MATRIX, ModelviewMatrix);

    GLdouble ProjectionMatrix[16];
    glGetDoublev(GL_PROJECTION_MATRIX, ProjectionMatrix);

    double LocalX, LocalY, LocalZ;
    gluUnProject(X, Viewport[3] - Y, 0, ModelviewMatrix, ProjectionMatrix, Viewport, &LocalX, &LocalY, &LocalZ);

    FRubberbandBounds.Left = (float) LocalX;
    FRubberbandBounds.Top = (float) LocalY;
    FRubberbandBounds.Right = (float) LocalX;
    FRubberbandBounds.Bottom = (float) LocalY;

    if (FRubberband == 0)
      FRubberband = glGenLists(1);
    glNewList(FRubberband, GL_COMPILE);
    glPushAttrib(GL_LINE_BIT);
    glDisable(GL_LINE_SMOOTH);

    switch (Style)
    {
      case GC_RBSTYLE_SOLID_THIN:   // A simple black rectangle with a one pixel wide border.
        {
          glColor3f(0, 0, 0);
          glLineWidth(1);
          glBegin(GL_LINE_LOOP);
            glVertex3d(0, 0, 1);
            glVertex3d(1, 0, 1);
            glVertex3d(1, 1, 1);
            glVertex3d(0, 1, 1);
          glEnd();
          break;
        };
      case GC_RBSTYLE_SOLID_THICK:  // A simple black rectangle with a 3 pixel wide border.
        {
          glColor3f(0, 0, 0);
          glLineWidth(3);
          glBegin(GL_LINE_LOOP);
            glVertex3d(0, 0, 1);
            glVertex3d(1, 0, 1);
            glVertex3d(1, 1, 1);
            glVertex3d(0, 1, 1);
          glEnd();
          break;
        };
      case GC_RBSTYLE_DOTTED_THIN:  // A simple black rectangle with a one pixel wide dotted border.
        {
          glColor3f(0, 0, 0);
          glLineWidth(1);
          glEnable(GL_LINE_STIPPLE);
          glLineStipple(1, 0xFF);
          glBegin(GL_LINE_LOOP);
            glVertex3d(0, 0, 1);
            glVertex3d(1, 0, 1);
            glVertex3d(1, 1, 1);
            glVertex3d(0, 1, 1);
          glEnd();
          break;
        };
      case GC_RBSTYLE_DOTTED_THICK: // A simple black rectangle with a 3 pixel wide dotted border.
        {
          glColor3f(0, 0, 0);
          glLineWidth(3);
          glEnable(GL_LINE_STIPPLE);
          glLineStipple(1, 0xFF);
          glBegin(GL_LINE_LOOP);
            glVertex3d(0, 0, 1);
            glVertex3d(1, 0, 1);
            glVertex3d(1, 1, 1);
            glVertex3d(0, 1, 1);
          glEnd();
          break;
        };
      case GC_RBSTYLE_BLENDED_CHECKERBOARD: // A filled rectangle with a one pixel border and a translucent interior.
        {
          GLubyte Color[4];
          if (!ColorByName("Highlight", Color))
          {
            // If the system's highlight color could not be found then use a default one.
            Color[0] = 64;
            Color[1] = 64;
            Color[2] = 128;
          };

          glLineWidth(1);
          glEnable(GL_LINE_SMOOTH);
          for (int Row = 0; Row < 10; Row++)
          {
            for (int Cell = 0; Cell < 10; Cell++)
            {
              if (((Row + Cell) & 1) == 0)
                Color[3] = 60;
              else
                Color[3] = 70;
              glColor4ubv(Color);
              glBegin(GL_POLYGON);
                glVertex3d(Cell * 0.1, Row * 0.1, 1);
                glVertex3d(Cell * 0.1, (Row + 1) * 0.1, 1);
                glVertex3d((Cell + 1) * 0.1, (Row + 1) * 0.1, 1);
                glVertex3d((Cell + 1) * 0.1, Row * 0.1, 1);
              glEnd();
            };
          };

          Color[3] = 200;
          glColor4ubv(Color);
          glBegin(GL_LINE_LOOP);
            glVertex3d(0, 0, 1);
            glVertex3d(1, 0, 1);
            glVertex3d(1, 1, 1);
            glVertex3d(0, 1, 1);
          glEnd();
          break;
        };
      case GC_RBSTYLE_BLENDED_DIAGONALS: // A filled rectangle with a one pixel border and a translucent interior.
        {
          GLubyte Color[4];
          if (!ColorByName("Highlight", Color))
          {
            // If the system's highlight color could not be found then use a default one.
            Color[0] = 64;
            Color[1] = 64;
            Color[2] = 128;
          };

          glLineWidth(1);

          // Interior.
          int Steps = 50;
          double X1 = 0;
          double Y1 = 0;
          double X2 = 0;
          double Y2 = 0;
          double dStep = 1.0 / Steps;

          // Left lower half.
          for (int X = 0; X < Steps; X++)
          {
            if (X % 2 == 0)
              Color[3] = 60;
            else
              Color[3] = 100;

            glColor4ubv(Color);
            glBegin(GL_POLYGON);
              glVertex3d(X * dStep, 0, 1);
              glVertex3d(0, X * dStep, 1);
              glVertex3d(0, (X + 1) * dStep, 1);
              glVertex3d((X + 1) * dStep, 0, 1);
            glEnd();
          };

          // Right upper half.
          for (int Y = 0; Y < Steps; Y++)
          {
            if (Y % 2 == 0)
              Color[3] = 60;
            else
              Color[3] = 100;

            glColor4ubv(Color);
            glBegin(GL_POLYGON);
              glVertex3d(1, Y * dStep, 1);
              glVertex3d(Y * dStep, 1, 1);
              glVertex3d((Y + 1) * dStep, 1, 1);
              glVertex3d(1, (Y + 1) * dStep, 1);
            glEnd();
          };

          // Border.
          Color[3] = 200;
          glColor4ubv(Color);
          glBegin(GL_LINE_LOOP);
            glVertex3d(0, 0, 1);
            glVertex3d(1, 0, 1);
            glVertex3d(1, 1, 1);
            glVertex3d(0, 1, 1);
          glEnd();
          break;
        };
    };
    glPopAttrib();
    glEndList();
    MakeDirty();
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Stops rubber banding if it is active currently. Does nothing if not.
 */
void CFeedbackLayer::RubberbandStop(void)
{
  if ((FStates & GC_SLSTATE_RUBBERBAND) != 0)
  {
    FStates &= ~GC_SLSTATE_RUBBERBAND;
    MakeDirty();
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets the the value of the given property, if it is a property of this class.
 *
 * @param Property The property to set.
 * @param Value The new value of the property.
 * @return True if the asked property is known in this class, false otherwise (in this case the property is not changed).
 * @note Implicit datatype conversion is performed if datatypes of property and Value parameter do not match.
 *       This can result in lost precision or even conversion errors. So make sure to use the right datatype for the call.
 */
bool CFeedbackLayer::SetProperty(TProperty Property, double Value)
{
  bool Result = true;
  switch (Property)
  {
    case GC_PROPERTY_HANDLE_SIZE:
      {
        // If point antialising is enabled then only certain point sizes are allowed. So we simply set the new value
        // and read back what OpenGL will use instead.
        GLfloat LastPointSize;
        glGetFloatv(GL_POINT_SIZE, &LastPointSize);
        glPointSize((float) Value);
        glGetFloatv(GL_POINT_SIZE, &FHandleSize);
        glPointSize(LastPointSize);

        break;
      };
    default:
      {
        Result = false;
        break;
      };
  };

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets the the value of the given property, if it is a property of this class.
 *
 * @param Property The property to set.
 * @param Value The new value of the property.
 * @return True if the asked property is known in this class, false otherwise (in this case the property is not changed).
 * @note Implicit datatype conversion is performed if datatypes of property and Value parameter do not match.
 *       This can result in lost precision or even conversion errors. So make sure to use the right datatype for the call.
 */
bool CFeedbackLayer::SetProperty(TProperty Property, int Value)
{
  bool Result = true;
  switch (Property)
  {
    case GC_PROPERTY_HANDLE_SIZE:
      {
        // If point antialising is enabled then only certain point sizes are allowed. So we simply set the new value
        // and read back what OpenGL will use instead.
        GLfloat LastPointSize;
        glGetFloatv(GL_POINT_SIZE, &LastPointSize);
        glPointSize((float) Value);
        glGetFloatv(GL_POINT_SIZE, &FHandleSize);
        glPointSize(LastPointSize);

        break;
      };
    default:
      {
        Result = false;
        break;
      };
  };

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

