/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file myx_gc_datatypes.h 
 * @brief Some commonly used data types.
 * 
 */

#ifndef __GC_GL_DATATYPES_H__
#define __GC_GL_DATATYPES_H__

#include <string>
#include <map>
#include <assert.h>
#include <set>
#include <vector>
#include <glib.h>

#include "myx_gc.h"

using namespace std;

#ifdef __GNUC__
  #include <ext/hash_map>

  #define stdext __gnu_cxx

  #ifndef __myhash
    #define __myhash
    namespace __gnu_cxx {
      template<class A> struct hash<A*>
      { size_t operator()(A *__x) const { return (unsigned long)__x; } };
    };
  #endif
  using namespace __gnu_cxx;
#else
  #include <hash_map>
  using namespace stdext;
#endif

//----------------- Macros ---------------------------------------------------------------------------------------------

// Assertion macro with custom message.
#ifdef _WINDOWS
  #define ASSERT(exp, message) (void)( (exp) || (_assert(message, __FILE__, __LINE__), 0))
#endif // #fidef _WINDOWS

// There is no ANSI rounding function for float numbers, so define our own.
#define ROUND(X) (int)((X) < 0 ? (X) - 0.5 : (X) + 0.5)

#define XML_IS(Node, Name) (xmlStrcmp(Node->name, (const xmlChar *) Name) == 0)

//----------------- Data types -----------------------------------------------------------------------------------------

#ifdef _WINDOWS
  // An opaque handle to a rendering context. Must be provided by the viewer.
  typedef HDC GCContext;
  typedef COLORREF GCColor;
#elif defined(__APPLE__)
  typedef CGLContextObj GCContext;
  typedef struct { GLushort red, green, blue; } GCColor;
#else
  typedef GLXContext GCContext;
  typedef XColor GCColor;
#endif // ifdef _WINDOWS

// A floating point rectangle used for area sizes (e.g. of the canvas).
typedef struct tagBounds
{
  float Left, Top, Right, Bottom;
} TBounds;

typedef struct tagViewport
{
  int Left, Top, Width, Height;
} TGCViewport;

// Some geometric data types.
typedef struct tagVertex
{
  float X;
  float Y;
} TVertex;

typedef vector<TVertex> CVertexVector;

typedef struct tagBoundingBox
{
  float UpperLeftX;
  float UpperLeftY;
  float UpperLeftZ;
  float LowerRightX;
  float LowerRightY;
  float LowerRightZ;

  tagBoundingBox()
  {
    UpperLeftX = 0;
    UpperLeftY = 0;
    UpperLeftZ = 0;
    LowerRightX = 0;
    LowerRightY = 0;
    LowerRightZ = 0;
  };
} TBoundingBox;

typedef struct tagConstraints
{
  float MaxHeight;
  float MaxWidth;
  float MinHeight;
  float MinWidth;

  tagConstraints()
  {
    MaxHeight = -1;
    MaxWidth = -1;
    MinHeight = -1;
    MinWidth = -1;
  };
} TConstraints;

typedef enum tagGCError
{
  GC_NO_ERROR = 0, 
  GC_CANT_OPEN_FILE, 
  GC_XML_PARSE_ERROR,
  GC_XML_INVALID_DOCUMENT, 
  GC_XML_EMPTY_DOCUMENT, 
  GC_OBJECT_NOT_FOUND, 
  GC_CANT_READ_FROM_FILE, 
  GC_CHARSET_CONVERSION_ERROR,
  GC_CHARSET_WRONG_CHARSET_SPECIFIED
} TGCError;

typedef enum tagChangeReason
{
  GC_CHANGE_SELECTION_ADD,
  GC_CHANGE_SELECTION_CLEAR,
  GC_CHANGE_SELECTION_REMOVE
} TGCChangeReason;

typedef enum tagGCDirection
{
  GC_SI_NONE,
  GC_SI_ON_OBJECT,
  GC_SI_NORTH,
  GC_SI_NORTH_EAST,
  GC_SI_EAST,
  GC_SI_SOUTH_EAST,
  GC_SI_SOUTH,
  GC_SI_SOUTH_WEST,
  GC_SI_WEST,
  GC_SI_NORTH_WEST
} TGCDirection;

typedef enum tagLayerType
{
  GC_LAYER_NORMAL,
  GC_LAYER_GRID
} TGCLayerType;

/**
 * TProperty is a list of all properties that can be set and queried in the canvas, figures etc. 
 */
typedef enum tagProperty
{
  GC_PROPERTY_HANDLE_SIZE         // Size of the selection handles in the selection layer.
} TProperty;

/**
 * TRubberbandStyle describes the look of the rubberband in the selection layer.
 */
typedef enum tagRubberbandStyle
{
  GC_RBSTYLE_SOLID_THIN,               // A simple black rectangle with a one pixel wide border.
  GC_RBSTYLE_SOLID_THICK,              // A simple black rectangle with a 3 pixel wide border.
  GC_RBSTYLE_DOTTED_THIN,              // A simple black rectangle with a one pixel wide dotted border.
  GC_RBSTYLE_DOTTED_THICK,             // A simple black rectangle with a 3 pixel wide dotted border.
  GC_RBSTYLE_BLENDED_CHECKERBOARD,     // A filled rectangle with a one pixel border and a translucent interior.
                                       // The system's selection color is used. The interior is a checker board.
  GC_RBSTYLE_BLENDED_DIAGONALS         // A filled rectangle with a one pixel border and a translucent interior.
                                       // The system's selection color is used. The interior consists of diagonal bands.
} TRubberbandStyle;

/**
 * TRBSelectionAction (rubber band selection action) determines how to manipulate the selection state of figure
 * instances with regard to their bounding box intersecting with the rubber band.
 */
typedef enum tagRBSelectionAction
{
  GC_RBACTION_NONE,               // Don't touch the selection state of any figure instance.
                                  // Usually used for non-selecting rubber bands (e.g. for figure creation).
  GC_RBACTION_SELECT,             // Always select figure instances if their bounding box intersects. Keep selected 
                                  // instances as their are if the do not intersect anymore.
                                  // Usually used for rubber bands with pressed shift key modifier.
  GC_RBACTION_SELECT_REMOVE,      // Select figure instances if they intersect, unselect those, which do not intersect.
                                  // Most common rubber band selection mode.
  GC_RBACTION_TOGGLE              // Revert the selection state of figure instances, which intersect. Don't touch the others.
                                  // Usually used for rubber bands with pressed control key modifier.
} TRBSelectionAction;

/** Layout variants for a figure element. */
typedef enum tagFigureElementLayout
{
  GC_LAYOUT_ROW,     
  GC_LAYOUT_COLUMN
} TFigureElementLayout;

/** Bidirectional mode */
typedef enum tagBidiMode
{
  GC_BIDI_LEFT_TO_RIGHT,          // Standard directionality (most languages).
  GC_BIDI_RIGHT_TO_LEFT           // Used for arabic and hebrew text.
} TBidiMode;

typedef map<string, unsigned char*> TColorMap;
typedef map<string, unsigned char*>::const_iterator TColorMapIterator;
typedef pair<string, unsigned char*> TColorMapPair;

/** Lists of styles, figure templates, figures and figure instances. */
class CGCStyle;
typedef hash_map<wstring, CGCStyle*> CStyleList;

class CFigureElementTemplate;
typedef vector<CFigureElementTemplate*> CElementTemplateList;

class CFigureElement;
typedef vector<CFigureElement*> CElementList;
typedef set<CFigureElement*> CElementSet;

class CFigureTemplate;
typedef hash_map<wstring, CFigureTemplate*> CLayoutList;

class CFigure;
typedef set<CFigure*> CFigureList;

class CFigureInstance;
typedef set<CFigureInstance*> CFigureInstanceList;

// Lists of listeners etc.

// A list of layers.
class CLayer;
typedef vector<CLayer*> CLayers;

//----------------------------------------------------------------------------------------------------------------------

#endif // #ifndef __GC_GL_DATATYPES_H__