/* This file is part of the KDE project
   Copyright (C) 2001 Wilco Greven <greven@kde.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

#ifndef CENTERINGSCROLLVIEW_H
#define CENTERINGSCROLLVIEW_H

#include <qpixmap.h>
#include <qscrollview.h>
#include <qptrvector.h>


/**
 * CenteringScrollview is a customized QScrollView, which can hold one
 * page. This page will be centered on the viewport.
 */
class CenteringScrollview : public QScrollView
{
    Q_OBJECT

public:
    CenteringScrollview( QWidget* parent = 0, const char* name = 0 );
    ~CenteringScrollview() { ; }
    
    void addChild( QPtrVector<QWidget> *wdgList );
    void addChild( QWidget *singleWdg );

    /** Sets the number of columns into which the widgets are
	aligned. Currently only '1' and '2' are permissible values. If
	nothing is set, '1' is the default. */
    void setNrColumns( Q_UINT8 cols );

    /** Returns the number of columns into which the widgets are
	aligned. Currently, '1' and '2' are the only possible return
	values, but to avoid problems with future versions, all
	possible return values should be handeled.  */
    Q_UINT8 getNrColumns() const {return nrCols;};


    /** Return true if the top resp. bottom of the page is visible. */
    bool atTop()    const;
    bool atBottom() const;
    
    /** Turn the scrollbars on/off. */
    void enableScrollBars( bool);
    
    /** Reimplementd from QScrollView to provide automatic
	centering */
    void resizeContents(int, int);
    
    /** Returns the number of columns in the display. That number is
	always >= 1 */
    Q_UINT8 numCols() {return nrCols;};

    /** Distance between pages in pixels (this is independent of
	the zoom level). */
    static const int distanceBetweenWidgets=10;

public slots:
    bool readUp();
    bool readDown();
    void scrollUp();
    void scrollDown();
    void scrollRight();
    void scrollLeft();
    void scrollBottom();
    void scrollTop();

    /** If the viewport is larger than the page, center the page on
	the viewport. */
    void centerContents();
   
signals:
    void viewSizeChanged( QSize size );
    void pageSizeChanged( QSize size );

    /** This signal is emitted when the scrollView receives a mouse
	wheel even. */
    void wheelEventReceived( QWheelEvent * );
    
protected:
    /** This method draws shadows about the child widgets that the
	scrollview holds. It is automatically called by QT whenever
	necessary. */
    void drawContents ( QPainter * p, int clipx, int clipy, int clipw, int cliph );

    void keyPressEvent( QKeyEvent* );
    void mousePressEvent( QMouseEvent* e );
    
    /** Reimplemented from QScrollView to make sure that the page is
	centered when it fits in the viewport. */
    void viewportResizeEvent( QResizeEvent* );

    /** Reimplemented from QScrollView, because the kviepart needs to
	handle the wheel events itself. The wheel event is passed on
	by emitting the singal "wheelEventReceived". Nothing else is
	done. */
    void contentsWheelEvent ( QWheelEvent * );
private:
    QPoint   _dragGrabPos;
    QPtrVector<QWidget> *widgetList;
    
    /** Flag to avoid infinite recursion in the re-implementation of
	resizeContents(). See the noted in the centerContents()
	method. */
    bool     centeringContents;

    /** Used internally by the centerContents()-method. Currently,
	only 1 and 2 are allowed as values. Set with the
	setNrColumns() method */
    Q_UINT8  nrCols;
    
    /** Color used internally in the shadow drawing routines in the
	drawContents method. */
    QColor   viewportBackgroundColorForCorners;

    /** Internal storages used in the shadow drawing routines in the
	drawContents method.*/
    QPixmap  URShadow, BRShadow, BLShadow;

    QPtrVector<QWidget> widgetListIfOldAddChildIsUsed;
};

#endif
