/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2004 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */
package org.gnu.gnomevfs;

import org.gnu.glib.GObject;

/**
 * Object for handing directory related tasks.
 */
public class VFSDirectory extends GObject {
	
	public VFSDirectory(String uri, VFSFileInfoOptions options) throws VFSException {
		int[] handle = new int[1];
		VFSResult result = VFSResult.intern(gnome_vfs_directory_open(handle, uri, options.getValue()));
		if (result == VFSResult.OK)
			setHandle(handle[0]);
		else
			throw new VFSException(result);
	}
	
	public VFSDirectory(VFSURI uri, VFSFileInfoOptions options) throws VFSException {
		int[] handle = new int[1];
		VFSResult result = VFSResult.intern(gnome_vfs_directory_open_from_uri(handle, uri.getHandle(), options.getValue()));
		if (result == VFSResult.OK)
			setHandle(handle[0]);
		else
			throw new VFSException(result);
	}
	
	public void close() throws VFSException {
		VFSResult result = VFSResult.intern(gnome_vfs_directory_close(getHandle()));
		if (result != VFSResult.OK)
			throw new VFSException(result);
	}
	
	public VFSFileInfo readNext() throws VFSException {
		int[] info = new int[1];
		VFSResult result = VFSResult.intern(gnome_vfs_directory_read_next(getHandle(), info));
		if (result == VFSResult.OK)
			return new VFSFileInfo(info[0]);
		else
			throw new VFSException(result);
	}
	
	/**
	 * Create a new directory
	 * 
	 * @param uri URI of the directory to be created.
	 * @param perm Unix-style permissions for the newly created directory.
	 */
	static public void makeDirectory(String uri, int perm) throws VFSException{
		VFSResult result =  VFSResult.intern(gnome_vfs_make_directory(uri, perm));
		if (result != VFSResult.OK)
			throw new VFSException(result);
	}
	
	/**
	 * Create a directory at uri.  Only succeeds if a file or directory does
	 * not already exist at uri.
	 * 
	 * @param uri
	 * @param perm
	 */
	static public void makeDirectory(VFSURI uri, int perm) throws VFSException {
		VFSResult result =  VFSResult.intern(gnome_vfs_make_directory_for_uri(uri.getHandle(), perm));
		if (result != VFSResult.OK)
			throw new VFSException(result);
	}
	
	/**
	 * Remove the directory.  The directory must be empty.
	 * 
	 * @param uri
	 */
	static public void removeDirectory(String uri) throws VFSException {
		VFSResult result = VFSResult.intern(gnome_vfs_remove_directory(uri));
		if (result != VFSResult.OK)
			throw new VFSException(result);
	}
	
	/**
	 * Remove the directory.  The directory must be empty.
	 * 
	 * @param uri
	 */
	static public void removeDirectory(VFSURI uri) throws VFSException {
		VFSResult result = VFSResult.intern(gnome_vfs_remove_directory_from_uri(uri.getHandle()));
		if (result != VFSResult.OK)
			throw new VFSException(result);
	}

    /****************************************
     * BEGINNING OF JNI CODE
     ****************************************/
	native static final protected int gnome_vfs_directory_open(int[] handle, String uri, int options);
	native static final protected int gnome_vfs_directory_open_from_uri(int[] handle, int uri, int options);
	native static final protected int gnome_vfs_directory_read_next(int handle, int[] fileInfo);
	native static final protected int gnome_vfs_directory_close(int handle);
	// static public API
    native static final protected int gnome_vfs_make_directory(String uri, int perm);
    native static final protected int gnome_vfs_make_directory_for_uri(int uri, int perm);
    native static final protected int gnome_vfs_remove_directory(String uri);
    native static final protected int gnome_vfs_remove_directory_from_uri(int uri);
    /****************************************
     * END OF JNI CODE
     ****************************************/
}
