/*
 *
 *   (C) Copyright IBM Corp. 2004
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: links.c
 */


#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>

#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>

#include <frontend.h>

#include "support.h"
#include "weblinks.h"
//#include "support.h"
#include "logging.h"

typedef struct link {
	gchar * label;
	gchar * url;
} link_t;


link_t links[] = {
	{"EVMS User Guide",   "http://evms.sourceforge.net/users_guide/"},
	{"EVMS FAQ",          "http://evms.sourceforge.net/faq.html"},
	{"EVMS Home Page",    "http://evms.sourceforge.net/"},
	{"EVMS Project Page", "http://sourceforge.net/projects/evms/"},
	{NULL, NULL}
};


/*
 *
 *   gchar *get_browser_path (void)
 *
 *   Description:
 *      This routine searches a set of paths to determine
 *      the location of a valid executable browser whether
 *      it be mozilla or netscape navigator.
 *
 *   Entry:
 *      url - string containing the URL to display
 *
 *   Exit:
 *      Returns the pathname of the executable or NULL if
 *      none found.
 *
 */
gchar *get_browser_path(void)
{
	gchar *path;

	if (access("/usr/bin/mozilla", X_OK) == 0)
		path = "/usr/bin/mozilla";
	else if (access("/usr/bin/netscape", X_OK) == 0)
		path = "/usr/bin/netscape";
	else if (access("/usr/local/bin/mozilla", X_OK) == 0)
		path = "/usr/local/bin/mozilla";
	else if (access("/usr/local/bin/netscape", X_OK) == 0)
		path = "/usr/local/bin/netscape";
	else if (access("/usr/X11R6/bin/mozilla", X_OK) == 0)
		path = "/usr/X11R6/bin/mozilla";
	else if (access("/usr/bin/X11/mozilla", X_OK) == 0)
		path = "/usr/bin/X11/mozilla";
	else if (access("/usr/X11/bin/mozilla", X_OK) == 0)
		path = "/usr/X11/bin/mozilla";
	else if (access("/usr/X11R6/bin/netscape", X_OK) == 0)
		path = "/usr/X11R6/bin/netscape";
	else if (access("/usr/bin/X11/netscape", X_OK) == 0)
		path = "/usr/bin/X11/netscape";
	else if (access("/usr/X11/bin/netscape", X_OK) == 0)
		path = "/usr/X11/bin/netscape";
	else
		path = NULL;

	return path;
}

/*
 *
 *   void display_url_in_browser (gchar *)
 *
 *   Description:
 *      This routine currently invokes either the mozilla or
 *      netscape browser to display the given URL.
 *
 *   Entry:
 *      url - string containing the URL to display
 *
 *   Exit:
 *      We try to fork a child that will exec either the mozilla or
 *      netscape browser to display the given URL.
 *
 */
void display_url_in_browser(gchar * url)
{
	gchar *path;

	path = get_browser_path();

	if (path) {
		pid_t pid = 0;

		pid = fork();

		if (pid == 0) {
			execl(path, path, url, NULL);

			/*
			 * Unable to exec the browser. Use the _exit() rather than
			 * exit() to avoid Xlib errors and the GUI going unresponsive.
			 */
			_exit(-1);
		} else if (pid == -1) {
			display_popup_window(_("Fork error"),
					     _
					     ("A problem was encountered attempting to fork a new process."));
			log_error("%s: fork() call to display URL failed.\n", __FUNCTION__);
		}
	} else {
		display_popup_window(_("Browser not found"),
				     _("A web browser was not found to display the page."));
		log_warning("%s: Could not find path to browser.\n", __FUNCTION__);
	}
}

static void on_go_button_clicked(GtkButton * button, gchar * url)
{
	display_url_in_browser(url);
}

GtkWidget *create_web_links_window(void)
{
	GtkWidget *web_links_window;
	GtkWidget *top_vbox;
	GtkWidget *table;
	GtkWidget *label;
	GtkWidget *entry;
	GtkWidget *button;
	gint row;
	GtkWidget *hseparator;
	GtkWidget *hbuttonbox;
	GtkWidget *hbox1;
	GtkWidget *ok_button_pixmap;
	guint ok_button_label_key;
	GtkWidget *ok_button_label;
	GtkAccelGroup *accel_group;

	accel_group = gtk_accel_group_new();

	web_links_window = gtk_window_new(GTK_WINDOW_DIALOG);
	gtk_widget_set_name(web_links_window, "web_links_window");
	gtk_object_set_data(GTK_OBJECT(web_links_window), "web_links_window", web_links_window);
	gtk_window_set_title(GTK_WINDOW(web_links_window), _("Web Links"));
	gtk_window_set_position(GTK_WINDOW(web_links_window), GTK_WIN_POS_CENTER);
	gtk_window_set_modal(GTK_WINDOW(web_links_window), TRUE);

	top_vbox = gtk_vbox_new(FALSE, 0);
	gtk_widget_set_name(top_vbox, "top_vbox");
	gtk_widget_ref(top_vbox);
	gtk_object_set_data_full(GTK_OBJECT(web_links_window), "top_vbox", top_vbox,
				 (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(top_vbox);
	gtk_container_add(GTK_CONTAINER(web_links_window), top_vbox);

	table = gtk_table_new(0, 3, FALSE);
	gtk_widget_show(table);
	gtk_table_set_row_spacings(GTK_TABLE(table), 10);
	gtk_table_set_col_spacings(GTK_TABLE(table), 10);
	gtk_container_add(GTK_CONTAINER(top_vbox), table);
	gtk_container_set_border_width(GTK_CONTAINER(table), 15);

	row = 0;
	do {
		gtk_table_resize(GTK_TABLE(table), row + 1, 3);

		label = gtk_label_new(links[row].label);
		gtk_widget_show(label);
		gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);
		gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row + 1,
				 (GtkAttachOptions) (GTK_FILL), (GtkAttachOptions) (0), 0, 0);

		entry = gtk_entry_new();
		gtk_widget_show(entry);
		gtk_entry_set_text(GTK_ENTRY(entry), links[row].url);
		gtk_entry_set_position(GTK_ENTRY(entry), 0);
		gtk_widget_set_usize(entry, 250, 20);
		gtk_editable_set_editable(GTK_EDITABLE(entry), FALSE);
		gtk_table_attach(GTK_TABLE(table), entry, 1, 2, row, row + 1,
				 (GtkAttachOptions) (GTK_FILL), (GtkAttachOptions) (0), 0, 0);

		button = gtk_button_new_with_label(_("Go->"));
		gtk_widget_show(button);
		gtk_table_attach(GTK_TABLE(table), button, 2, 3, row, row + 1,
				 (GtkAttachOptions) (GTK_FILL), (GtkAttachOptions) (0), 0, 0);
		gtk_signal_connect(GTK_OBJECT(button), "clicked",
				   GTK_SIGNAL_FUNC(on_go_button_clicked), links[row].url);
		row++;

	} while (links[row].label != NULL);

	hseparator = gtk_hseparator_new();
	gtk_widget_set_name(hseparator, "hseparator");
	gtk_widget_ref(hseparator);
	gtk_object_set_data_full(GTK_OBJECT(web_links_window), "hseparator", hseparator,
				 (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(hseparator);
	gtk_box_pack_start(GTK_BOX(top_vbox), hseparator, FALSE, TRUE, 12);

	hbuttonbox = gtk_hbutton_box_new();
	gtk_widget_set_name(hbuttonbox, "hbuttonbox");
	gtk_widget_ref(hbuttonbox);
	gtk_object_set_data_full(GTK_OBJECT(web_links_window), "hbuttonbox", hbuttonbox,
				 (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(hbuttonbox);
	gtk_box_pack_start(GTK_BOX(top_vbox), hbuttonbox, FALSE, TRUE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(hbuttonbox), 8);
	gtk_button_box_set_layout(GTK_BUTTON_BOX(hbuttonbox), GTK_BUTTONBOX_END);
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(hbuttonbox), 0);
	gtk_button_box_set_child_size(GTK_BUTTON_BOX(hbuttonbox), 90, 53);
	gtk_button_box_set_child_ipadding(GTK_BUTTON_BOX(hbuttonbox), 0, 0);

	button = gtk_button_new();
	gtk_widget_set_name(button, "button");
	gtk_widget_ref(button);
	gtk_object_set_data_full(GTK_OBJECT(web_links_window), "button", button,
				 (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(button);
	gtk_container_add(GTK_CONTAINER(hbuttonbox), button);
	GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
	gtk_widget_add_accelerator(button, "clicked", accel_group, GDK_O, 0, GTK_ACCEL_VISIBLE);

	hbox1 = gtk_hbox_new(FALSE, 0);
	gtk_widget_set_name(hbox1, "hbox1");
	gtk_widget_ref(hbox1);
	gtk_object_set_data_full(GTK_OBJECT(web_links_window), "hbox1", hbox1,
				 (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(hbox1);
	gtk_container_add(GTK_CONTAINER(button), hbox1);

	ok_button_pixmap = create_pixmap(web_links_window, "stock_ok.xpm");
	gtk_widget_set_name(ok_button_pixmap, "ok_button_pixmap");
	gtk_widget_ref(ok_button_pixmap);
	gtk_object_set_data_full(GTK_OBJECT(web_links_window), "ok_button_pixmap", ok_button_pixmap,
				 (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(ok_button_pixmap);
	gtk_box_pack_start(GTK_BOX(hbox1), ok_button_pixmap, TRUE, TRUE, 0);
	gtk_misc_set_alignment(GTK_MISC(ok_button_pixmap), 0.8, 0.5);

	ok_button_label = gtk_label_new("");
	ok_button_label_key = gtk_label_parse_uline(GTK_LABEL(ok_button_label), _("_OK"));
	gtk_widget_set_name(ok_button_label, "ok_button_label");
	gtk_widget_ref(ok_button_label);
	gtk_object_set_data_full(GTK_OBJECT(web_links_window), "ok_button_label", ok_button_label,
				 (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(ok_button_label);
	gtk_box_pack_start(GTK_BOX(hbox1), ok_button_label, TRUE, TRUE, 0);
	gtk_misc_set_alignment(GTK_MISC(ok_button_label), 1.49012e-08, 0.5);

	gtk_signal_connect(GTK_OBJECT(button), "clicked",
			   GTK_SIGNAL_FUNC(on_button_clicked_destroy_window), NULL);

	gtk_widget_add_accelerator(button, "clicked", accel_group,
				   ok_button_label_key, GDK_MOD1_MASK, (GtkAccelFlags) 0);

	gtk_widget_grab_default(button);
	gtk_window_add_accel_group(GTK_WINDOW(web_links_window), accel_group);

	return web_links_window;
}

