/*
 *  Copyright (C) 2002,2003,2004  Mattia Dongili<dongili@supereva.it>
 *                                George Staikos <staikos@0wned.org>
 *
 *  2004.08.22
 *  - added percentage/absolute frequency translation based on a patch submitted
 *    by Herv Eychenne
 *
 *  2003.16.08
 *  - added support for cpu monitoring, base code by Dietz Proepper and minor
 *    fixes by Mattia Dongili
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "cpufreqd.h"
#include "main.h"
#include "config_parser.h"

/*
 * Parse configuration file.
 *
 * As long as the config file is quite simple let's keep also a veryvery simple parser.
 */
int parse_config (general *g) {
  FILE *config;
  profile *last_profile = 0L, *p_iter = 0L;
  rule *last_rule = 0L, *r_iter = 0L;
  char buf[256];

  /* configuration file */
  cp_log(LOG_INFO, "parse_config(): reading configuration file %s\n", g->config_file);
  config = fopen(g->config_file, "r");
  if (!config) {
    cp_log(LOG_ERR, "parse_config(): %s: %s\n", g->config_file, strerror(errno));
    return -1;
  }

  while (!feof(config)) {
    char *clean = 0L;

    if (!fgets(buf, 255, config))
      break;

    clean = clean_config_line(buf);

    if (!clean[0]) /* returned an empty line */
      continue;

    clean = strip_comments_line(clean);

    if (!clean[0]) /* returned an empty line */
      continue;

    /* if General scan general options */
    if (strstr(clean,"[General]")) {
      if (parse_config_general (config, g) < 0) {
        return -1;
      }
      continue;
    }

    /* if Profile scan profile options */
    if (strstr(clean,"[Profile]")) {

      profile *prof = (profile *)malloc(sizeof(profile));
      memset(prof, 0, sizeof(profile));
      if (parse_config_profile (config, prof, g) < 0) {
        cp_log(LOG_CRIT, "parse_config(): [Profile] error parsing %s, see logs for details.\n", g->config_file);
        fclose(config);
        return -1;
      }
      if (!g->profiles) {
        g->profiles = last_profile = prof;
      } else {
        p_iter = g->profiles;
        /* checks duplicate names */
        while (p_iter) {
          if (strcmp(p_iter->name, prof->name) == 0) {
            cp_log(LOG_CRIT, "parse_config(): [Profile] name \"%s\" already exists.\n", prof->name);
            fclose(config);
            return -1;
          }
          p_iter = p_iter->next;
        }
        last_profile->next = prof;
        last_profile = last_profile->next;
      }
      continue;
    }

    /* if Rule scan rules options */
    if (strstr(clean,"[Rule]")) {
      rule *rul = (rule *)malloc(sizeof(rule));
      memset(rul, 0, sizeof(rule));
      if (parse_config_rule (config, rul) < 0) {
        cp_log(LOG_CRIT, "parse_config(): [Rule] error parsing %s, see logs for details.\n", g->config_file);
        return -1;
      }
      if (!g->rules) {
        g->rules = last_rule = rul;
      } else {
        r_iter = g->rules;
        /* check duplicate names */
        while (r_iter != NULL) {
          if (strcmp(r_iter->name, rul->name) == 0) {
            cp_log(LOG_CRIT, "parse_config(): [Rule] name \"%s\" already exists.\n", rul->name);
            return -1;
          }
          r_iter = r_iter->next;
        }
        last_rule->next = rul;
        last_rule = last_rule->next;
      }

      continue;
    }
  }
  fclose(config);

  /* did I read something? 
   * chek if I read at least one rule, otherwise exit
   */
  if (g->rules == NULL) {
    cp_log(LOG_ERR, "parse_config(): No rules found!\n");
    return -1;
  }
  
  /*
   * associate rules->profiles
   * go through rules and associate to the proper profile
   */
  r_iter = g->rules;
  while (r_iter != NULL) {
    int profile_found = 0;

    p_iter = g->profiles;
    /* go through profiles */
    while (p_iter != NULL) {
      if (strcmp(r_iter->profile_name, p_iter->name)==0) {
        /* a profile is allowed to be addressed by more than 1 rule */
        r_iter->prof = p_iter;
        profile_found = 1;
        break;
      }
      p_iter = p_iter->next;
    }

    if (!profile_found) {
      cp_log(LOG_CRIT, "parse_config(): Syntax error: no Profile section found for Rule \"%s\" \
                        (requested Profile \"%s\")\n",
                       r_iter->name,
                       r_iter->profile_name);
      return -1;
    }
    r_iter = r_iter->next;
  }

  r_iter = g->rules;
  while (r_iter != NULL) {
     cp_log(LOG_NOTICE, "parse_config(): Rule \"%s\" has Profile \"%s\"\n", r_iter->name, r_iter->prof->name);
     r_iter = r_iter->next;
  }

  return 0;
}

/*
 * parse the [General] section
 *
 * Returns -1 if required properties are missing, 0 otherwise
 */
int parse_config_general (FILE *config, general *g) {
char buf[256];

  while (!feof(config)) {
    char *clean;
    char *name;
    char *value;

    if (!fgets(buf, 255, config))
      break;

    clean = strip_comments_line(buf);

    if (!clean[0]) /* returned an empty line */
      continue;

    clean = clean_config_line(clean);

    if (!clean[0]) /* returned an empty line */
      break;

    name = strtok(clean, "=");
    value = strtok(NULL, "");

    if (strcmp(name,"poll_interval") == 0) {
      if (value != NULL) {
        g->poll_interval = atoi (value);
      }
      /* validate */
      if (g->poll_interval < 1) {
        cp_log(LOG_WARNING, "WARNING: [General] poll_interval has invalid value, using default.\n");
        g->poll_interval = DEFAULT_POLL;
      }
    } else if (strcmp(name,"verbosity") == 0) {
      if (!g->log_level_overridden) {
        if (value != NULL) {
          g->log_level = atoi (value);
          /* validate */
          if (g->log_level < 0 || g->log_level > 7) {
            cp_log(LOG_WARNING, "WARNING: [General] verbosity has invalid value, using default (%d).\n", DEFAULT_VERBOSITY);
            g->log_level = DEFAULT_VERBOSITY;
          }
        } else {
          g->log_level = DEFAULT_VERBOSITY;
          cp_log(LOG_WARNING, "WARNING: [General] verbosity has empty value, using default (%d).\n", DEFAULT_VERBOSITY);
        }
      } else {
          cp_log(LOG_DEBUG, "parse_config_general(): skipping \"verbosity\", overridden in the command line.\n");
      }
    } else if (strcmp(name,"pm_type") == 0) {
      if (value != NULL) {
        strncpy(g->pm_plugin, value, 101);
      } else {
        cp_log(LOG_WARNING, "parse_config_general(): empty \"pm_type\", using default "DEFAULT_PMPLUGIN".\n");
        strncpy(g->pm_plugin, DEFAULT_PMPLUGIN, 101);
      }
      g->pm_plugin[100] = 0;
    } else if (strcmp(name,"pidfile") == 0) {
      if (value != NULL) {
        strncpy(g->pidfile, value, 512);
      } else {
        cp_log(LOG_WARNING, "parse_config_general(): empty \"pidfile\", using default "CPUFREQD_PIDFILE".\n");
        strncpy(g->pidfile, CPUFREQD_PIDFILE, 512);
      }
      g->pidfile[511] = 0;
    } else if (strcmp(name,"acpi_workaround") == 0) {
      if (value != NULL) {
        g->acpi_workaround = atoi (value);
        cp_log(LOG_WARNING, "parse_config_general(): ACPI workaround enabled.\n");
      }
    } else {
      cp_log(LOG_WARNING, "WARNING: [General] skipping unknown config option \"%s\"\n", name);
    }
  }

  return 0;
}

/*
 * parse a [Profile] section
 *
 * Returns -1 if required properties are missing, 0 otherwise
 */
int parse_config_profile (FILE *config, profile *p, general *g) {
#define HAS_NAME    1
#define HAS_MIN     2
#define HAS_MAX     4
#define HAS_POLICY  8
int state = 0;
char buf[256];

  while (!feof(config)) {
    char *clean;
    char *name;
    char *value;

    if (!fgets(buf, 255, config))
      break;

    clean = strip_comments_line(buf);

    if (!clean[0]) /* returned an empty line */
      continue;

    clean = clean_config_line(clean);

    if (!clean[0]) /* returned an empty line */
      break;

    name = strtok(clean, "=");
    value = strtok(NULL, "");
    
    /* empty value: skip */
    if (value == NULL)
      continue;
    
    if (strcmp(name,"name")==0) {
      strncpy(p->name, value, 255);
      p->name[254] = 0;
      state |= HAS_NAME;
    } else if (strcmp(name,"minfreq")==0) {
      if (strstr(value, "%") != NULL) {
        /* set separator (useful for 2.4 kernels only) */
        if (p->sep != ':') {
          p->sep = '%';
        } else {
          cp_log(LOG_CRIT, "parse_config_profile(): cannot mix percentage and absolute kHz values in a single profile.\n");
          return -1;
        }
      } else {
        /* set separator (useful for 2.4 kernels only) */
        if (p->sep != '%') {
          p->sep = ':';
        } else {
          cp_log(LOG_CRIT, "parse_config_profile(): cannot mix percentage and absolute kHz values in a single profile.\n");
          return -1;
        }
      }
      p->min_freq = atol(value);
      state |= HAS_MIN;
      
    } else if (strcmp(name,"maxfreq") == 0) {
      
      if (strstr(value, "%") != NULL) {
        /* set separator (useful for 2.4 kernels only) */
        if (p->sep != ':') {
          p->sep = '%';
        } else {
          cp_log(LOG_CRIT, "parse_config_profile(): cannot mix percentage and absolute kHz values in a single profile.\n");
          return -1;
        }
      } else {
        /* set separator (useful for 2.4 kernels only) */
        if (p->sep != '%') {
          p->sep = ':';
        } else {
          cp_log(LOG_CRIT, "parse_config_profile(): cannot mix percentage and absolute kHz values in a single profile.\n");
          return -1;
        }
      }
      p->max_freq = atol(value);
      state |= HAS_MAX;
    
    } else if (strcmp(name,"policy") == 0) {
      
      strncpy(p->policy_name, value, 255);
      p->policy_name[254] = 0;
      state |= HAS_POLICY;
    
    } else {
      cp_log(LOG_WARNING, "WARNING: [Profile] skipping unknown config option \"%s\"\n", name);
    }
  }

  if (!(state & HAS_NAME)) {
    cp_log(LOG_ERR, "parse_config_profile(): [Profile] missing required property \"name\".\n");
    return -1;
  }
  if (!(state & HAS_MIN)) {
    cp_log(LOG_ERR, "parse_config_profile(): [Profile] \"%s\" missing required property \"minfreq\".\n", p->name);
    return -1;
  }
  if (!(state & HAS_MAX)) {
    cp_log(LOG_ERR, "parse_config_profile(): [Profile] \"%s\" missing required property \"maxfreq\".\n", p->name);
    return -1;
  }
  if (!(state & HAS_POLICY)) {
    cp_log(LOG_ERR, "parse_config_profile(): [Profile] \"%s\" missing required property \"policy\".\n", p->name);
    return -1;
  }
  
  /* translate max_freq to absolute value if necessary */
  if (g->has_sysfs && p->sep == '%') {
    p->max_freq = g->cpu_max_freq * ((float)p->max_freq / 100);
    cp_log(LOG_INFO, "parse_config_profile(): [Profile] \"%s\" translated max frequency to %d\n", p->name, p->max_freq);
    /* normalize if necessary */
    if (p->max_freq > g->cpu_max_freq) {
      cp_log(LOG_NOTICE, "parse_config_profile(): [Profile] \"%s\" max frequency %d out of range\n", p->name, p->max_freq);
      p->max_freq = g->cpu_max_freq;
    }
  }
  /* translate min_freq to absolute value if necessary */
  if (g->has_sysfs && p->sep == '%') {
    p->min_freq = g->cpu_max_freq * ((float)p->min_freq / 100);
    cp_log(LOG_INFO, "parse_config_profile(): [Profile] \"%s\" translated min frequency to %d\n", p->name, p->min_freq);
    /* normalize if necessary */
    if (p->min_freq < g->cpu_min_freq) {
      cp_log(LOG_NOTICE, "parse_config_profile(): [Profile] \"%s\" min frequency %d out of range\n", p->name, p->min_freq);
      p->min_freq = g->cpu_min_freq;
    }
  }

  return 0;
}

/*
 * parses a [Rule] section
 *
 * Returns -1 if required properties are missing, 0 otherwise
 */
int parse_config_rule (FILE *config, rule *r) {
#define HAS_NAME    1 
#define HAS_PROFILE 2
int state = 0;
char buf[256];

  /* reset profile ref */
  r->prof = 0;

  /* initialize battery_interval */
  r->bat = (battery_interval *) malloc(sizeof(battery_interval));
  r->bat->min = 101;
  r->bat->max = -1;
  
  /* initialize cpu_interval */
  r->cpu = (cpu_interval *) malloc(sizeof(cpu_interval));
  r->cpu->min = 101;    /* so we won't ever trigger them... */
  r->cpu->max = -1;

  /* initialize cpu_nice_scale to DEFAULT_NICE_SCALE */
  r->cpu_nice_scale = DEFAULT_NICE_SCALE;
  
  while (!feof(config)) {
    char *clean;
    char *name;
    char *value;

    if (!fgets(buf, 255, config))
      break;

    clean = strip_comments_line(buf);

    if (!clean[0]) /* returned an empty line */
      continue;

    clean = clean_config_line(clean);

    if (!clean[0]) /* returned an empty line */
      break;

    name = strtok(clean, "=");
    value = strtok(NULL, "");

    /* empty value: skip */
    if (value == NULL)
      continue;

    if (strcmp(name,"name") == 0) {
      strncpy(r->name, value, 255);
      r->name[254] = 0;
      state |= HAS_NAME;
    } else if (strcmp(name,"ac") == 0) {
      r->ac = strcmp(value,"on") == 0 ? 1:0;
    } else if (strcmp(name,"battery_interval") == 0) {
      r->bat->min = atoi(strtok(value,"-"));
      r->bat->max = atoi(strtok(NULL,""));
    } else if (strcmp(name,"cpu_interval") == 0) {
      r->cpu->min = atoi(strtok(value,"-"));
      r->cpu->max = atoi(strtok(NULL,""));
    } else if (strcmp(name,"cpu_nice_scale") == 0) {
      r->cpu_nice_scale = atof(value);
      if (r->cpu_nice_scale<=0) {
        cp_log(LOG_ERR, "parse_config_rule(): [Rule] invalid \"cpu_nice_scale\" value: %f.\n", r->cpu_nice_scale);
        return -1;
      }
    } else if (strcmp(name,"delay_cycles") == 0) {
      r->delay_cycles = atoi(value);
      if (r->delay_cycles<0) {
        cp_log(LOG_ERR, "parse_config_rule(): [Rule] invalid \"delay_cycles\" value: %d.\n", r->delay_cycles);
        return -1;
      }
    } else if (strcmp(name,"programs") == 0) {
      /* create program list */
      char *t_prog;
      r->program_list = string_list_new();
      t_prog = strtok(value,",");
      do {
        struct string_node *temp;
        
        if (t_prog == NULL)
          continue;
        
        temp = string_node_new();
        strncpy(temp->string, t_prog, 255);
        temp->string[254] = 0;

        string_list_append(r->program_list, temp);
        cp_log(LOG_DEBUG, "parse_config_rule(): read program: %s\n", temp->string);
      } while ((t_prog = strtok(NULL,",")) != NULL);

    } else if (strcmp(name, "profile") == 0) {
      strncpy(r->profile_name, value, 255);
      r->profile_name[254] = 0;
      state |= HAS_PROFILE;
    } else {
      cp_log(LOG_WARNING, "WARNING: [Rule] skipping unknown config option \"%s\"\n", name);
    }
  }

  if (!(state & HAS_NAME)) {
    cp_log(LOG_ERR, "parse_config_rule(): [Rule] missing required property \"name\".\n");
    return -1;
  }

  if (!(state & HAS_PROFILE)) {
    cp_log(LOG_ERR, "parse_config_rule(): [Rule] \"%s\" missing required property \"profile\".\n", 
                 r->name);
    return -1;
  }

  return 0;
}

/* char *clean_config_line (char *str)
 *
 * Removes trailing blanks and CR off the string str
 *
 * Returns a pointer to the cleaned string.
 * WARNING: it modifies the input string!
 * 
 */
char *clean_config_line (char *str) {
int i = 0;

  /* remove white spaces at the beginning */
  while (isspace(str[0])) {
    str++;
  }

  /* remove end line white space */
  i = strlen(str) - 1;
  while (i >= 0 && isspace(str[i])) {
    str[i] = 0;
    i--;
  }

  return str;
}

/* char *strip_comments_line (char *str)
 *
 * Removes comments off the string str
 *
 * Returns a pointer to the cleaned string.
 * WARNING: it modifies the input string!
 * 
 */
char *strip_comments_line (char *str) {
int i;

  /* remove comment */
  for (i = strlen(str); i >= 0; i--) {
    if (str[i] == '#') {
      str[i] = 0;
    }
  }

  return str;
}

/* void free_config(general *g)
 *
 * Frees the structures allocated.
 *
 */
void free_config(general *g) {
  profile *p_iter = 0L, *p_temp = 0L;
  rule *r_iter = 0L, *r_temp = 0L;
  
  /* cleanup config structs */
  r_iter = g->rules;
  while (r_iter != NULL) {
    r_temp = r_iter;
    r_iter = r_iter->next;
    
      cp_log(LOG_DEBUG, "free_config(): freeing rule %s.\n", r_temp->name);
      cp_log(LOG_DEBUG, "free_config(): freeing battery interval %d-%d.\n", r_temp->bat->min, r_temp->bat->max);
    
    free(r_temp->bat);
    
      cp_log(LOG_DEBUG, "free_config(): freeing cpu interval %d-%d.\n", r_temp->cpu->min, r_temp->cpu->max);
    
    free(r_temp->cpu);
    r_temp->bat = 0L;
    r_temp->cpu = 0L;
    
      cp_log(LOG_DEBUG, "free_config(): freeing program list.\n");
    
    if (r_temp->program_list) {
      string_list_free_sublist(r_temp->program_list, r_temp->program_list->first);
      free(r_temp->program_list);
    }
    
    free(r_temp);
  }
  g->rules = 0L;
  
  p_iter = g->profiles;
  while (p_iter != NULL) {
    p_temp = p_iter;
    p_iter = p_iter->next;
    cp_log(LOG_DEBUG, "free_config(): freeing profile %s.\n", p_temp->name);
    free(p_temp);
  }
  g->profiles = 0L;

  /* clean other values */
  g->poll_interval = DEFAULT_POLL;
  g->has_sysfs = 0;
  g->acpi_workaround = 0;
  g->cpu_min_freq = 0;
  g->cpu_max_freq = 0;
  
  if (!g->log_level_overridden)
    g->log_level = DEFAULT_VERBOSITY;
  
}
