#| 

   Fully Ordered Finite Sets, Version 0.81
   Copyright (C) 2003, 2004 by Jared Davis <jared@cs.utexas.edu>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public Lic-
   ense along with this program; if not, write to the Free Soft-
   ware Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA.



 typed.lisp

  This file allows you to extend the sets library and very quickly 
  define new, restricted sets wherein all elements satisfy some prop-
  erty defined by a predicate.  To do this, you only need to invoke 
  the macro introduce-typed-set.  

  The general form is:

    (SETS::introduce-typed-set element-recognizer "set-recognizer")

  As a concrete example:

    (SETS::introduce-typed-set integerp "integer-setp")

  This generates the following function:

    (SETS::integer-setp X)  
        Returns true iff X is a set which consists entirely of 
        integer elements.  Note: linear complexity.

  It also generates several theorems.  For brevity, let R be the name
  of the set recognizer, and let r be the name of the element recog-
  nizer.  Then we introduce the following theorems: (all of which are
  also introduced in the SETS:: package):

   R-set               (R X) => (setp X)
   R-tail-set          (R X) => (R (tail X))
   R-head-recognized   (R X) => (r (head X))
   R-sfix              (R X) => (R (sfix X))
   R-empty-set         (setp X) ^ (empty X) => (R x)
   R-insert            (R (insert a X)) = (r a) ^ ((R x) v (empty X))
   R-insert-chaining   (R X) ^ (r a) => (R (insert a X))
   R-member-recognized (R X) ^ (in a X) => (r a)
   R-union             (R X) ^ (R Y) => (R (union X Y))
   R-intersect         (R X) v (R Y) => (R (intersect X Y))
   R-difference        (R X) => (R (difference X Y))
   R-delete            (R x) => (R (delete a X))
    
  These theorems are also put into a deftheory called R-theory, so 
  you can easily disable them all.  For example:

    (in-theory (disable SETS::integer-setp-theory))

  You can also enable/disable them individually, e.g.

    (in-theory (disable SETS::integer-setp-tail-set
                        SETS::integer-setp-member-recognized
                        ...))
    
|#

(in-package "SETS")
(include-book "sets")
(set-verify-guards-eagerness 2)
(enable-set-reasoning)


; A generic recognizer function:

(defstub recognizer (*) => *)



; A generic typed-set recognizer:

(defun typed-listp (X) 
  (declare (xargs :guard (setp X)))
  (or (empty X) 
      (and (recognizer (head X))
           (typed-listp (tail X)))))

(defun typed-setp (X)
  (declare (xargs :guard t
                  :verify-guards nil))
  (mbe :logic (and (setp X) 
                   (or (empty X)
                       (and (recognizer (head X))
                            (typed-setp (tail X)))))
       :exec (and (setp X) 
                  (typed-listp X))))
      
(defthm typed-set-mbe-equivalence
  (implies (setp X)
           (equal (typed-listp X) (typed-setp X))))

(verify-guards typed-setp)



; Basic Properties of Typed Sets:

(defthm typed-set-set
  (implies (typed-setp X)
           (setp X)))

(defthm typed-set-tail-set
  (implies (typed-setp X) 
           (typed-setp (tail X))))

(defthm typed-set-head-recognized
  (implies (and (typed-setp X) 
                (not (empty X)))
           (recognizer (head X))))

(defthm typed-set-empty-set
  (implies (and (setp X) 
                (empty X)) 
           (typed-setp X)))

(defthm typed-set-sfix
  (implies (typed-setp X) 
           (typed-setp (sfix X))))

(defthm typed-set-member-recognized
  (implies (and (typed-setp X)
                (in a X))
           (recognizer a)))

(defthm typed-set-insert
  (equal (typed-setp (insert a X))
         (and (recognizer a)
              (or (typed-setp X)
                  (empty X))))
  :hints(("Goal" :induct (insert a X))))

(defthm typed-set-union
  (implies (and (typed-setp X) 
                (typed-setp Y))
           (typed-setp (union X Y))))

(defthm typed-set-intersect
  (implies (or (typed-setp X) 
               (typed-setp Y))
           (typed-setp (intersect X Y))))

(defthm typed-set-difference
  (implies (typed-setp X)
           (typed-setp (difference X Y))))

(defthm typed-set-delete
  (implies (typed-setp X)
           (typed-setp (delete a X))))




; Little macro to make symbols in the SETS package, so that all of
; the introduced typed sets become part of the sets package.

(defmacro sm (s1 s2)
  `(intern-in-package-of-symbol (string-append ,s1 ,s2) 'SETS::empty))

; BIG UGLY MACRO to instantiate a new typed set theory:


(defmacro introduce-typed-set (recognizer set-recognizer-name)

  (let* ((set-rec-name (string-upcase set-recognizer-name))
         (set-rec (intern-in-package-of-symbol set-rec-name 
                                              'SETS::empty)))

    `(encapsulate nil

       (defun ,(sm set-rec-name "-FAST") (X)
         (declare (xargs :guard (setp X)))
         (if (empty X)
             t
           (and (,recognizer (head X))
                (,(sm set-rec-name "-FAST") (tail X)))))

       (defun ,set-rec (X)
         (declare (xargs :guard t :verify-guards nil))
         (mbe :logic (and (setp X)
                          (if (empty X)
                              t
                            (and (,recognizer (head X))
                                 (,set-rec (tail X)))))
              :exec (and (setp X)
                         (,(sm set-rec-name "-FAST") X))))
   
       (local (defthm ,(sm set-rec-name "-MBE-EQUIVALENCE")
         (implies (setp X)
                  (equal (,(sm set-rec-name "-FAST") X)
                         (,set-rec X)))
         :hints(("Goal" 
                 :use ((:functional-instance 
                        typed-set-mbe-equivalence
                           (recognizer ,recognizer) 
                           (typed-setp ,set-rec)))))))

       (in-theory (disable ,(sm set-rec-name "-FAST")))

       (verify-guards ,set-rec)

       (local (in-theory 
         (disable ,(sm set-rec-name "-MBE-EQUIVALENCE"))))
                           


       (defthm ,(sm set-rec-name "-SET")
         (implies (,set-rec X) 
                  (setp X))
         :hints(("Goal" 
                 :use ((:functional-instance typed-set-set
                           (recognizer ,recognizer)
                           (typed-setp ,set-rec))))))


       (defthm ,(sm set-rec-name "-TAIL-SET")
         (implies (,set-rec X) 
                  (,set-rec (tail X)))
         :hints(("Goal" 
                 :use ((:functional-instance typed-set-tail-set
                           (recognizer ,recognizer)
                           (typed-setp ,set-rec)))))
         :rule-classes :forward-chaining)


       (defthm ,(sm set-rec-name "-HEAD-RECOGNIZED")
         (implies (and (,set-rec X) 
                       (not (empty X)))
                  (,recognizer (head X)))
         :hints(("Goal" 
                 :use ((:functional-instance 
                        typed-set-head-recognized
                           (recognizer ,recognizer)
                           (typed-setp ,set-rec)))))
         :rule-classes :forward-chaining)


       (defthm ,(sm set-rec-name "-SFIX")
         (implies (,set-rec X) 
                  (,set-rec (sfix X)))
         :hints(("Goal" 
                 :use ((:functional-instance typed-set-sfix
                           (recognizer ,recognizer)
                           (typed-setp ,set-rec))))))
   

       (defthm ,(sm set-rec-name "-EMPTY-SET")
         (implies (and (setp X) 
                       (empty X)) 
                  (,set-rec X))
         :hints(("Goal" 
                :use ((:functional-instance typed-set-empty-set
                          (recognizer ,recognizer)
                          (typed-setp ,set-rec))))))


       (defthm ,(sm set-rec-name "-INSERT")
         (equal (,set-rec (insert a X))
                (and (,recognizer a)
                     (or (,set-rec X) (empty X))))
         :hints(("Goal" 
                 :use ((:functional-instance typed-set-insert
                           (recognizer ,recognizer)
                           (typed-setp ,set-rec))))))

   
       (defthm ,(sm set-rec-name "-INSERT-CHAINING")
         (implies (and (,set-rec X)
                       (,recognizer a))
                  (,set-rec (insert a X)))
         :rule-classes :forward-chaining)


       (defthm ,(sm set-rec-name "-MEMBER-RECOGNIZED")
         (implies (and (,set-rec X)
                       (in a X))
                  (,recognizer a))
         :hints(("Goal" 
                 :use ((:functional-instance 
                        typed-set-member-recognized
                           (recognizer ,recognizer)
                           (typed-setp ,set-rec))))))
   

       (defthm ,(sm set-rec-name "-UNION")
         (implies (and (,set-rec X) 
                       (,set-rec Y))
                  (,set-rec (union X Y)))
         :hints(("Goal" 
                 :use ((:functional-instance typed-set-union
                           (recognizer ,recognizer)
                           (typed-setp ,set-rec))))))


       (defthm ,(sm set-rec-name "-INTERSECT")
         (implies (or (,set-rec X) 
                      (,set-rec Y))
                  (,set-rec (intersect X Y)))
         :hints(("Goal" 
                 :use ((:functional-instance typed-set-intersect
                           (recognizer ,recognizer)
                           (typed-setp ,set-rec))))))


       (defthm ,(sm set-rec-name "-DIFFERENCE")
         (implies (,set-rec X)
                  (,set-rec (difference X Y)))
         :hints(("Goal" 
                 :use ((:functional-instance typed-set-difference
                           (recognizer ,recognizer)
                           (typed-setp ,set-rec))))))


       (defthm ,(sm set-rec-name "-DELETE")
         (implies (,set-rec X)
                  (,set-rec (delete a X)))
         :hints(("Goal" 
                 :use ((:functional-instance typed-set-delete
                           (recognizer ,recognizer)
                           (typed-setp ,set-rec))))))

       (in-theory (disable (:definition ,set-rec)))

       (deftheory ,(sm set-rec-name "-THEORY")
         '(,(sm set-rec-name "-SET")
           ,(sm set-rec-name "-TAIL-SET")
           ,(sm set-rec-name "-HEAD-RECOGNIZED")
           ,(sm set-rec-name "-SFIX")
           ,(sm set-rec-name "-EMPTY-SET")
           ,(sm set-rec-name "-INSERT")
           ,(sm set-rec-name "-INSERT-CHAINING")
           ,(sm set-rec-name "-MEMBER-RECOGNIZED")
           ,(sm set-rec-name "-UNION")
           ,(sm set-rec-name "-INTERSECT")
           ,(sm set-rec-name "-DIFFERENCE")
           ,(sm set-rec-name "-DELETE")))

)))



; Disable all of the generic theorems so that they don't slow any
; rewriting down.  The need for this initially struck me as odd, but
; since the theorem typed-set-set concludes (setp X), it occasionally
; comes up in proof attempts, and then its hypothesis (typed-setp X)
; is introduced in backchaining, leading to a other theorems being
; tried.  So, I just disable them all and this will not happen.

(in-theory (disable typed-listp
                    typed-setp
                    typed-set-mbe-equivalence
                    typed-set-set
                    typed-set-tail-set
                    typed-set-head-recognized
                    typed-set-sfix
                    typed-set-empty-set
                    typed-set-insert
                    typed-set-member-recognized
                    typed-set-union
                    typed-set-intersect
                    typed-set-difference
                    typed-set-delete)) 

;(introduce-typed-set integerp "integer-setp")
     
