package OpenInteract2::CreateSecurity;

# $Id: CreateSecurity.pm,v 1.6 2004/03/19 05:54:50 lachoy Exp $

use strict;
use base qw( Class::Accessor::Fast );
use Log::Log4perl            qw( get_logger );
use OpenInteract2::Constants qw( :log );
use OpenInteract2::Context   qw( CTX );
use SPOPS::Secure            qw( :level :scope );

my @FIELDS = qw(
    scope scope_id level class where iterator website_dir
    start_time end_time num_processed num_failed
);
__PACKAGE__->mk_accessors( @FIELDS );

my %VALID_SCOPE = (
   world => SEC_SCOPE_WORLD,
   user  => SEC_SCOPE_USER,
   group => SEC_SCOPE_GROUP );

my %VALID_LEVEL = (
    none  => SEC_LEVEL_NONE,
    read  => SEC_LEVEL_READ,
    write => SEC_LEVEL_WRITE );

my %DEFAULTS = (
     iterator => 'yes',
);

my ( $log );

sub new {
    my ( $class, $params ) = @_;
    my $self = bless( {}, $class );
    for ( @FIELDS ) {
        my $value = $params->{ $_ } || $DEFAULTS{ $_ };
        $self->$_( $value ) if ( $value );
    }
    return $self;
}

sub errors {
    my ( $self ) = @_;
    $self->{errors} ||= {};
    return values %{ $self->{errors} };
}

sub errors_with_params {
    my ( $self ) = @_;
    $self->{errors} ||= {};
    return $self->{errors};
}

sub validate {
    my ( $self ) = @_;

    # Be sure all the necessary parameters have been issued before we
    # create the context

    my %errors = ();
    unless ( -d $self->website_dir ) {
        $errors{website_dir} = "'website_dir' must be set to website dir";
    }

    OpenInteract2::Context->create({ website_dir => $self->website_dir });

    my $scope    = $self->scope;
    my $scope_id = $self->scope_id;
    my $level    = $self->level;

    unless ( $scope =~ /^(world|group|user)$/ ) {
        $errors{scope} = "'scope' must be set to 'world', 'group', or 'user'";
    }
    if ( $scope ne 'world' and ! $scope_id ) {
        $errors{scope_id} = "'scope_id' must be set if the --scope is set " .
                            "to 'group' or 'user'";
    }
    unless ( $level =~ /^(none|read|write)$/ ) {
        $errors{level} = "'level' must be set to 'none', 'read' or 'write'";
    }

    my $spops_class = $self->class;
    if ( $spops_class ) {
        if ( ! $spops_class->can( 'fetch' ) ) {
            $errors{class} =
                    "Class '$spops_class' is not valid. Are you " .
                    "sure it is defined in your OpenInteract setup?";
        }
        elsif ( ! $spops_class->isa( 'SPOPS::Secure' ) ) {
            $errors{class} =
                    "Class '$spops_class' is not using security.";
        }
    }

    else {
        $errors{class} = "Class not set, must be set to the class of " .
                         "the object for which you want to set security " .
                         "(e.g., 'OpenInteract2::News')";
    }

    $self->scope( $VALID_SCOPE{ lc $scope } );
    $self->level( $VALID_LEVEL{ lc $level } );

    $self->{errors} = \%errors;

    if ( scalar keys %errors == 0 ) {
        $self->{_validated}++;
        return 1;
    }
    return 0;
}


sub run {
    my ( $self ) = @_;

    $self->start_time( time );
    return unless ( $self->{_validated} or $self->validate );

    $log ||= get_logger( LOG_OI );

    my $scope_id = $self->scope_id;
    my $default_objects = CTX->lookup_default_object_id;
    if ( $scope_id and $default_objects->{ $scope_id } ) {
        $scope_id = $default_objects->{ $scope_id };
    }

    my ( $object_store );
    my $spops_class = $self->class;

    if ( $self->iterator eq 'yes' ) {
        $object_store = $spops_class->fetch_iterator(
                              { skip_security => 1,
                                column_group  => '_id_field',
                                where         => $self->where });
    }
    else {
        $object_store = $spops_class->fetch_group(
                              { skip_security => 1,
                                column_group  => '_id_field',
                                where         => $self->where });
    }

    my ( $count, $failure ) = ( 0, 0 );

    my $scope = $self->scope;
    my $level = $self->level;

    my $sec_class = CTX->lookup_object( 'security' );

    while ( my $object = $self->_get_from_store( $object_store ) ) {
        $sec_class->new({ class          => $spops_class,
                          object_id      => $object->id,
                          scope          => $scope,
                          scope_id       => $scope_id,
                          security_level => $level } )->save();
        if ( $@ ) {
            $log->info( "FAIL: ", $object->id, " ($@)" );
            $failure++;
        }
        else {
            $log->info( "OK: ", $object->id );
        }
        $count++;
    }
    $self->num_processed( $count );
    $self->num_failed( $failure );
    $self->end_time( time );
    return $count;
}

# Use either an iterator or a list

sub _get_from_store {
    my ( $self, $store ) = @_;
    return undef unless ( $store );
    if ( ref $store eq 'ARRAY' ) {
        return undef unless ( scalar @{ $store } );
        return shift @{ $store };
    }
    return $store->get_next;
}

1;
