use strict;
use warnings;
use diagnostics;
use File::Temp qw(tempdir);
use File::Slurp qw( write_file);
use Test::More tests =>19;
use Test::Exception;
use Archive::BagIt;

my @ROOT = grep {length} 'src';
### TESTS
my $SRC_BAG = File::Spec->catdir( @ROOT, 'src_bag');
my $SRC_FILES = File::Spec->catdir( @ROOT, 'src_files');
my $DST_BAG = File::Spec->catdir(@ROOT, 'dst_bag');
my $Class = 'Archive::BagIt';
## tests
use_ok($Class);
my $bag = $Class->new({bag_path=>$SRC_BAG});

{
    is(Archive::BagIt::_check_key(""), "key should have a length > null (key='')", "_check_key('')");
    is(Archive::BagIt::_check_key("foo:"), "key should not contain a colon! (key='foo:')", "_check_key('foo:')" );
    like(Archive::BagIt::_check_key("fo\no"), qr{key should match .* but contains newlines}, "_check_key('fo\\no')" );
    is(Archive::BagIt::_check_key(), "key should match '[^\\r\\n:]+', but is not defined", "_check_key()");
    is(Archive::BagIt::_check_value(""), "value should have a length > null (value='')", "_check_value('')");
    is(Archive::BagIt::_check_value(), "value should match '[^\\r\\n:]+', but is not defined", "_check_value()");
}

{
    my $input =<<BAGINFO;
Foo: Bar
Foo: Baz
Foo2 : Bar2
Foo3:   Bar3
Foo4: Bar4
  Baz4
  Bay4
Foo5: Bar5
Foo6: Bar6: Baz6
BAGINFO
    my @expected = (
        { "Foo", "Bar" },
        { "Foo", "Baz"},
        { "Foo2", "Bar2"},
        { "Foo3", "Bar3"},
        { "Foo4", "Bar4\n  Baz4\n  Bay4"},
        { "Foo5", "Bar5"},
        { "Foo6", "Bar6: Baz6"}
    );
    my $got = $bag->_parse_bag_info( $input );
    is_deeply( $got, \@expected, "bag-info parsing valid");
    $bag->{"bag_info"} = $got;
    ok($bag->verify_baginfo(), "bag-info verify valid");
    is_deeply( $bag->{warnings}, ["Payload-Oxum was expected in bag-info.txt, but not found!"], "bag-info parsing valid, warning for missed payload oxum");
}

{
    delete $bag->{'warnings'};
    delete $bag->{'errors'};
    delete $bag->{'bag_info'};
    my $input =<<BAGINFO;
Foo:
BAGINFO
    my @expected = (
        { "Foo", "" },
    );
    my $got = $bag->_parse_bag_info( $input );
    is_deeply( $got, \@expected, "bag-info parsing invalid");
    $bag->{"bag_info"} = $got;
    ok(!$bag->verify_baginfo(), "bag-info verify invalid");
    is_deeply( $bag->{warnings}, ["Payload-Oxum was expected in bag-info.txt, but not found!"], "bag-info parsing valid, warning for missed payload oxum");
    is_deeply($bag->{errors}, ["value should have a length > null (value='')"], "bag-info parsing valid, error logged" );
}

{
    delete $bag->{'warnings'};
    delete $bag->{'errors'};
    delete $bag->{'bag_info'};
    my $input =<<BAGINFO;
 ::: foo
BAGINFO
    my $dir = tempdir(CLEANUP => 1);
    write_file(File::Spec->catfile($dir, "bag-info.txt"), $input);
    my $bag2 = $Class->new({bag_path=>$dir});
    my $got = $bag2->verify_baginfo();
    ok(!$bag2->verify_baginfo(), "bag-info verify fully invalid");
    is_deeply($bag2->{errors}, ["the baginfo file '$dir/bag-info.txt' could not be parsed correctly, because following residue not fullfill the match requirements: '$input'"], "bag-info parsing valid, error logged" );
}
{
    delete $bag->{'warnings'};
    delete $bag->{'errors'};
    delete $bag->{'bag_info'};
    my $input =<<BAGINFO;
Foo: Bar
Foo: Baz
Foo2 : Bar2
Foo3:   Bar3
Foo4: Bar4
  Baz4
  Bay4
Foo5: Bar5
Foo6: Bar6: Baz6
BAGINFO
    my $dir = tempdir(CLEANUP => 1);
    write_file(File::Spec->catfile($dir, "bag-info.txt"), $input);
    my $bag2 = $Class->new({bag_path=>$dir});
    ok($bag2->verify_baginfo(), "bag-info verify fully valid");
    #use Data::Printer; p($bag2);
    is_deeply($bag2->{warnings}, ["Payload-Oxum was expected in bag-info.txt, but not found!"], "bag-info parsing fully valid, warning for missed payload oxum");
    is_deeply($bag2->{errors}, [], "bag-info verify fully valid, no error log exists");
}

1;
