const char pwd_lua[] =
"local ffi   = require('ffi')\n"
"local errno = require('errno')\n"
"\n"
"-- {{{ Definitions\n"
"\n"
"-- GID_T, UID_T and TIME_T are, essentially, `integer types`.\n"
"-- http://pubs.opengroup.org/onlinepubs/009695399/basedefs/sys/types.h.html\n"
"ffi.cdef[[\n"
"    typedef int uid_t;\n"
"    typedef int gid_t;\n"
"    typedef long time_t;\n"
"]]\n"
"\n"
"-- POSIX demands to have three fields in struct group:\n"
"-- http://pubs.opengroup.org/onlinepubs/009695399/basedefs/grp.h.html\n"
"-- char   *gr_name The name of the group.\n"
"-- gid_t   gr_gid  Numerical group ID.\n"
"-- char  **gr_mem  Pointer to a null-terminated array of character pointers to\n"
"--                 member names.\n"
"--\n"
"-- So we'll extract only them.\n"
"ffi.cdef[[\n"
"    struct group {\n"
"        char    *gr_name;    /* group name */\n"
"        char    *gr_passwd;  /* group password */\n"
"        gid_t    gr_gid;     /* group id */\n"
"        char   **gr_mem;     /* group members */\n"
"    };\n"
"]]\n"
"\n"
"-- POSIX demands to have five fields in struct group:\n"
"-- char    *pw_name   User's login name.\n"
"-- uid_t    pw_uid    Numerical user ID.\n"
"-- gid_t    pw_gid    Numerical group ID.\n"
"-- char    *pw_dir    Initial working directory.\n"
"-- char    *pw_shell  Program to use as shell.\n"
"--\n"
"-- So we'll extract only them.\n"
"if ffi.os == 'OSX' or ffi.os == 'BSD' then\n"
"    ffi.cdef[[\n"
"        struct passwd {\n"
"            char    *pw_name;    /* user name */\n"
"            char    *pw_passwd;  /* encrypted password */\n"
"            uid_t    pw_uid;     /* user uid */\n"
"            gid_t    pw_gid;     /* user gid */\n"
"            time_t   pw_change;  /* password change time */\n"
"            char    *pw_class;   /* user access class */\n"
"            char    *pw_gecos;   /* Honeywell login info */\n"
"            char    *pw_dir;     /* home directory */\n"
"            char    *pw_shell;   /* default shell */\n"
"            time_t   pw_expire;  /* account expiration */\n"
"            int      pw_fields;  /* internal: fields filled in */\n"
"        };\n"
"    ]]\n"
"else\n"
"    ffi.cdef[[\n"
"        struct passwd {\n"
"            char *pw_name;   /* username */\n"
"            char *pw_passwd; /* user password */\n"
"            int   pw_uid;    /* user ID */\n"
"            int   pw_gid;    /* group ID */\n"
"            char *pw_gecos;  /* user information */\n"
"            char *pw_dir;    /* home directory */\n"
"            char *pw_shell;  /* shell program */\n"
"        };\n"
"    ]]\n"
"end\n"
"\n"
"ffi.cdef[[\n"
"    uid_t          getuid();\n"
"    struct passwd *getpwuid(uid_t uid);\n"
"    struct passwd *getpwnam(const char *login);\n"
"    void           endpwent();\n"
"    struct passwd *getpwent();\n"
"    void           setpwent();\n"
"\n"
"    gid_t          getgid();\n"
"    struct group  *getgrgid(gid_t gid);\n"
"    struct group  *getgrnam(const char *group);\n"
"    struct group  *getgrent();\n"
"    void           endgrent();\n"
"    void           setgrent();\n"
"]]\n"
"\n"
"-- }}}\n"
"\n"
"-- {{{ Error handling\n"
"\n"
"local pwgr_errstr = \"%s failed [errno %d]: %s\"\n"
"\n"
"-- Use it in the following way: set errno to zero, call a passwd /\n"
"-- group function, then call this function to check whether there\n"
"-- was an error.\n"
"local function pwgrcheck(func_name, pwgr)\n"
"    if pwgr ~= nil then\n"
"        return\n"
"    end\n"
"    if errno() == 0 then\n"
"        return\n"
"    end\n"
"    error(pwgr_errstr:format(func_name, errno(), errno.strerror()))\n"
"end\n"
"\n"
"-- }}}\n"
"\n"
"-- {{{ Call passwd / group database\n"
"\n"
"local function _getpw(uid)\n"
"    local pw = nil\n"
"    errno(0)\n"
"    if type(uid) == 'number' then\n"
"        pw = ffi.C.getpwuid(uid)\n"
"    elseif type(uid) == 'string' then\n"
"        pw = ffi.C.getpwnam(uid)\n"
"    else\n"
"        error(\"Bad type of uid (expected 'string'/'number')\")\n"
"    end\n"
"    pwgrcheck('_getpw', pw)\n"
"    return pw\n"
"end\n"
"\n"
"local function _getgr(gid)\n"
"    local gr = nil\n"
"    errno(0)\n"
"    if type(gid) == 'number' then\n"
"        gr = ffi.C.getgrgid(gid)\n"
"    elseif type(gid) == 'string' then\n"
"        gr = ffi.C.getgrnam(gid)\n"
"    else\n"
"        error(\"Bad type of gid (expected 'string'/'number')\")\n"
"    end\n"
"    pwgrcheck('_getgr', gr)\n"
"    return gr\n"
"end\n"
"\n"
"-- }}}\n"
"\n"
"-- {{{ Serialize passwd / group structures to tables\n"
"\n"
"local function grtotable(gr)\n"
"    local gr_mem, group_members = gr.gr_mem, {}\n"
"    local i = 0\n"
"    while true do\n"
"        local member = gr_mem[i]\n"
"        if member == nil then\n"
"            break\n"
"        end\n"
"        table.insert(group_members, ffi.string(member))\n"
"        i = i + 1\n"
"    end\n"
"    return {\n"
"        id      = tonumber(gr.gr_gid),\n"
"        name    = ffi.string(gr.gr_name),\n"
"        members = group_members,\n"
"    }\n"
"end\n"
"\n"
"-- gr is optional\n"
"local function pwtotable(pw, gr)\n"
"    local user = {\n"
"        name    = ffi.string(pw.pw_name),\n"
"        id      = tonumber(pw.pw_uid),\n"
"        workdir = ffi.string(pw.pw_dir),\n"
"        shell   = ffi.string(pw.pw_shell),\n"
"    }\n"
"    if gr ~= nil then\n"
"        user.group = grtotable(gr)\n"
"    end\n"
"    return user\n"
"end\n"
"\n"
"-- }}}\n"
"\n"
"-- {{{ Public API functions\n"
"\n"
"local function getgr(gid)\n"
"    if gid == nil then\n"
"        gid = tonumber(ffi.C.getgid())\n"
"    end\n"
"    local gr = _getgr(gid)\n"
"    if gr == nil then\n"
"        return nil\n"
"    end\n"
"    return grtotable(gr)\n"
"end\n"
"\n"
"local function getpw(uid)\n"
"    if uid == nil then\n"
"        uid = tonumber(ffi.C.getuid())\n"
"    end\n"
"    local pw = _getpw(uid)\n"
"    if pw == nil then\n"
"        return nil\n"
"    end\n"
"    local gr = _getgr(pw.pw_gid) -- can be nil\n"
"    return pwtotable(pw, gr)\n"
"end\n"
"\n"
"--\n"
"-- systemd v209 sets errno to ENOENT in\n"
"-- _nss_systemd_getpwent_r and _nss_systemd_getgrent_r\n"
"-- when there are no more entries left to enumerate.\n"
"--\n"
"-- This is a bug which has been fixed later in\n"
"-- systemd code but we have to deal with buggy\n"
"-- environment thus ignore such error if appears.\n"
"--\n"
"-- See the reference for details\n"
"-- https://github.com/systemd/systemd/issues/9585\n"
"--\n"
"-- The issue affects getpwent/getgrent calls only\n"
"-- thus provide own wrappings to keep workaround\n"
"-- in one place and do not affect any other calls\n"
"-- where ENOENT might become a valid error case.\n"
"--\n"
"-- Initially we observed this issue on Fedora 29\n"
"-- when a password database is traversed for the\n"
"-- first time.\n"
"local function getpwent()\n"
"    errno(0)\n"
"    local pw = ffi.C.getpwent()\n"
"    if pw == nil and errno() == errno.ENOENT then\n"
"        errno(0)\n"
"    end\n"
"    return pw\n"
"end\n"
"\n"
"local function getgrent()\n"
"    errno(0)\n"
"    local gr = ffi.C.getgrent()\n"
"    if gr == nil and errno() == errno.ENOENT then\n"
"        errno(0)\n"
"    end\n"
"    return gr\n"
"end\n"
"\n"
"local function getpwall()\n"
"    ffi.C.setpwent()\n"
"    local pws = {}\n"
"    -- Note: Don't call _getpw() during the loop: it leads to drop\n"
"    -- of a getpwent() current entry to a first one on CentOS 6\n"
"    -- and FreeBSD 12.\n"
"    while true do\n"
"        local pw = getpwent()\n"
"        if pw == nil then\n"
"            pwgrcheck('getpwall', pw)\n"
"            break\n"
"        end\n"
"        local gr = _getgr(pw.pw_gid) -- can be nil\n"
"        table.insert(pws, pwtotable(pw, gr))\n"
"    end\n"
"    ffi.C.endpwent()\n"
"    return pws\n"
"end\n"
"\n"
"local function getgrall()\n"
"    ffi.C.setgrent()\n"
"    local grs = {}\n"
"    -- Note: Don't call _getgr() during the loop: it leads to drop\n"
"    -- of a getgrent() current entry to a first one on CentOS 6\n"
"    -- and FreeBSD 12.\n"
"    while true do\n"
"        local gr = getgrent()\n"
"        if gr == nil then\n"
"            pwgrcheck('getgrall', gr)\n"
"            break\n"
"        end\n"
"        table.insert(grs, grtotable(gr))\n"
"    end\n"
"    ffi.C.endgrent()\n"
"    return grs\n"
"end\n"
"\n"
"-- }}}\n"
"\n"
"return {\n"
"    getpw = getpw,\n"
"    getgr = getgr,\n"
"    getpwall = getpwall,\n"
"    getgrall = getgrall,\n"
"}\n"
""
;
