// Copyright 2019 Drone IO, Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package encrypt

import (
	"crypto/aes"
	"crypto/cipher"
	"crypto/rand"
	"encoding/base64"
	"encoding/json"
	"io"
	"net/http"

	"github.com/drone/drone-go/drone"
	"github.com/drone/drone/core"
	"github.com/drone/drone/handler/api/render"
	"github.com/go-chi/chi"
)

type respEncrypted struct {
	Data string `json:"data"`
}

// Handler returns an http.HandlerFunc that processes http
// requests to create an encrypted secret.
func Handler(repos core.RepositoryStore) http.HandlerFunc {
	return func(w http.ResponseWriter, r *http.Request) {
		namespace := chi.URLParam(r, "owner")
		name := chi.URLParam(r, "name")
		repo, err := repos.FindName(r.Context(), namespace, name)
		if err != nil {
			render.NotFound(w, err)
			return
		}

		in := new(drone.Secret)
		err = json.NewDecoder(r.Body).Decode(in)
		if err != nil {
			render.BadRequest(w, err)
			return
		}

		// the secret is encrypted with a per-repository 256-bit
		// key. If the key is missing or malformed we should
		// return an error to the client.
		encrypted, err := encrypt([]byte(in.Data), []byte(repo.Secret))
		if err != nil {
			render.InternalError(w, err)
			return
		}

		// the encrypted secret is embedded in the yaml
		// configuration file and is json-encoded for
		// inclusion as a !binary attribute.
		encoded := base64.StdEncoding.EncodeToString(encrypted)

		render.JSON(w, &respEncrypted{Data: encoded}, 200)
	}
}

func encrypt(plaintext, key []byte) (ciphertext []byte, err error) {
	block, err := aes.NewCipher(key)
	if err != nil {
		return nil, err
	}

	gcm, err := cipher.NewGCM(block)
	if err != nil {
		return nil, err
	}

	nonce := make([]byte, gcm.NonceSize())
	_, err = io.ReadFull(rand.Reader, nonce)
	if err != nil {
		return nil, err
	}

	return gcm.Seal(nonce, nonce, plaintext, nil), nil
}
