// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_dq.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_dq__A3__F366

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr std::string_view PACKAGE_PATH { R"xyzw(amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊bf16@16_256_3_T_T_0___gfx120x)xyzw" };
constexpr std::string_view FUNC_NAME { R"xyzw(＊bf16@16_256_3_T_T_0)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx1201)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0x8a30a4dcu, 0x811808bau, 0, 155 }, // 8a30a4dc811808ba = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊bf16@16_256_3_T_T_0___gfx120x__P__16_16__CO__wave1_warp4_stg1--Arch_gfx1201
    { 0xdbcf0c43u, 0x9c043791u, 0, 103 }, // dbcf0c439c043791 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊bf16@16_256_3_T_T_0___gfx120x__P__16_16__CO__wave3_warp4_stg1--Arch_gfx1201
    { 0x87a3daddu, 0xcabf4e6bu, 40, 155 }, // 87a3daddcabf4e6b = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊bf16@16_256_3_T_T_0___gfx120x__P__32_16__CO__wave1_warp4_stg1--Arch_gfx1201
    { 0x3482183cu, 0xd9b62e4bu, 40, 86 }, // 3482183cd9b62e4b = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊bf16@16_256_3_T_T_0___gfx120x__P__32_16__CO__wave2_warp4_stg1--Arch_gfx1201
    { 0x1e0c35bau, 0x5c8f7344u, 40, 63 }, // 1e0c35ba5c8f7344 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊bf16@16_256_3_T_T_0___gfx120x__P__32_16__CO__wave4_warp4_stg1--Arch_gfx1201
    { 0x5495906au, 0x3fd9474au, 120, 155 }, // 5495906a3fd9474a = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊bf16@16_256_3_T_T_0___gfx120x__P__64_16__CO__wave1_warp4_stg1--Arch_gfx1201
    { 0x0592f7d7u, 0x172cf056u, 120, 86 }, // 0592f7d7172cf056 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊bf16@16_256_3_T_T_0___gfx120x__P__64_16__CO__wave2_warp4_stg1--Arch_gfx1201
    { 0x2b688317u, 0xb27fd98fu, 120, 103 }, // 2b688317b27fd98f = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊bf16@16_256_3_T_T_0___gfx120x__P__64_16__CO__wave3_warp4_stg1--Arch_gfx1201
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_dq_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_dq_packed_string);

static int8_t lut[1][10][10] =
{
// GPU gfx1201_mod0
{{1,1,1,1,0,1,1,1,1,1},
 {4,4,4,4,4,4,4,4,4,4},
 {5,5,4,4,4,4,4,4,4,1},
 {4,2,4,4,4,4,4,4,1,1},
 {2,6,5,5,7,7,7,4,4,4},
 {2,2,6,6,5,5,4,4,4,4},
 {3,3,5,5,6,7,4,4,4,4},
 {3,3,5,6,7,7,7,7,7,7},
 {3,3,6,6,5,6,5,5,6,7},
 {3,2,6,6,5,5,5,5,6,5}}
// End of GPU gfx1201_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelDqContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelDqContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_dq__A3__F366 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 5;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_dq__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 5;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*bf16:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// stride_bz = 0
// stride_dbz = 0
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = "fp32"
// philox_seed_ptr = "*u64"
// philox_offset1 = "*u64"
// philox_offset2 = "u64"
// Window_left = "i32"
// BLOCK_DMODEL = 256
// CAUSAL_TYPE = 3
// ENABLE_DROPOUT = True
// PADDED_HEAD = True
// BIAS_TYPE = 0

// vim: set fileencoding=utf-8

