// Copyright 2019 Drone IO, Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package repos

import (
	"net/http"
	"os"

	"github.com/drone/drone/core"
	"github.com/drone/drone/handler/api/render"
	"github.com/drone/drone/handler/api/request"
	"github.com/drone/drone/logger"

	"github.com/dchest/uniuri"
	"github.com/go-chi/chi"
)

// FEATURE FLAG enables a static secret value used to sign
// incoming requests routed through a proxy. This was implemented
// based on feedback from @chiraggadasc and should not be
// removed until we have a permanent solution in place.
var staticSigner = os.Getenv("DRONE_FEATURE_SERVER_PROXY_SECRET")

// HandleEnable returns an http.HandlerFunc that processes http
// requests to enable a repository in the system.
func HandleEnable(
	hooks core.HookService,
	repos core.RepositoryStore,
	sender core.WebhookSender,
) http.HandlerFunc {
	return func(w http.ResponseWriter, r *http.Request) {
		var (
			owner = chi.URLParam(r, "owner")
			name  = chi.URLParam(r, "name")
		)
		user, _ := request.UserFrom(r.Context())
		repo, err := repos.FindName(r.Context(), owner, name)
		if err != nil {
			render.NotFound(w, err)
			logger.FromRequest(r).
				WithError(err).
				WithField("namespace", owner).
				WithField("name", name).
				Debugln("api: repository not found")
			return
		}
		repo.Active = true
		repo.UserID = user.ID

		if repo.Config == "" {
			repo.Config = ".drone.yml"
		}
		if repo.Signer == "" {
			repo.Signer = uniuri.NewLen(32)
		}
		if repo.Secret == "" {
			repo.Secret = uniuri.NewLen(32)
		}
		if repo.Timeout == 0 {
			repo.Timeout = 60
		}

		if staticSigner != "" {
			repo.Signer = staticSigner
		}

		err = hooks.Create(r.Context(), user, repo)
		if err != nil {
			render.InternalError(w, err)
			logger.FromRequest(r).
				WithError(err).
				WithField("namespace", owner).
				WithField("name", name).
				Debugln("api: cannot create or update hook")
			return
		}

		err = repos.Activate(r.Context(), repo)
		if err == core.ErrRepoLimit {
			render.ErrorCode(w, err, 402)
			logger.FromRequest(r).
				WithError(err).
				WithField("namespace", owner).
				WithField("name", name).
				Errorln("api: cannot activate repository")
			return
		}
		if err != nil {
			render.InternalError(w, err)
			logger.FromRequest(r).
				WithError(err).
				WithField("namespace", owner).
				WithField("name", name).
				Debugln("api: cannot activate repository")
			return
		}

		err = sender.Send(r.Context(), &core.WebhookData{
			Event:  core.WebhookEventRepo,
			Action: core.WebhookActionEnabled,
			User:   user,
			Repo:   repo,
		})
		if err != nil {
			logger.FromRequest(r).
				WithError(err).
				WithField("namespace", owner).
				WithField("name", name).
				Warnln("api: cannot send webhook")
		}

		render.JSON(w, repo, 200)
	}
}
