// Copyright 2019 Drone IO, Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package core

import (
	"context"

	"github.com/drone/drone-yaml/yaml"
)

const (
	// RegistryPull policy allows pulling from a registry.
	RegistryPull = "pull"

	// RegistryPush Policy allows pushing to a registry for
	// all event types except pull requests.
	RegistryPush = "push"

	// RegistryPushPullRequest Policy allows pushing to a
	// registry for all event types, including pull requests.
	RegistryPushPullRequest = "push-pull-request"
)

type (
	// Registry represents a docker registry with credentials.
	Registry struct {
		Address  string `json:"address"`
		Username string `json:"username"`
		Password string `json:"password"`
		Policy   string `json:"policy"`
	}

	// RegistryArgs provides arguments for requesting
	// registry credentials from the remote service.
	RegistryArgs struct {
		Repo     *Repository    `json:"repo,omitempty"`
		Build    *Build         `json:"build,omitempty"`
		Conf     *yaml.Manifest `json:"-"`
		Pipeline *yaml.Pipeline `json:"-"`
	}

	// RegistryService provides registry credentials from an
	// external service.
	RegistryService interface {
		// List returns registry credentials from the global
		// remote registry plugin.
		List(context.Context, *RegistryArgs) ([]*Registry, error)
	}
)
