/*
 * LaTeD Version 1.1
 * (c) Gene Ressler 1993, 94, 97
 *   de8827@trotter.usma.edu
 *
 * LaTeD is a graphical editor for drawings in the LaTeX "picture" 
 * environment.  It runs under MSDOS or in a Windows DOS box.  The
 * distribution includes full sources, including LaTeX source for 
 * its documentation.
 *
 * No warranty of this software is expressed or implied by the author.
 *
 * Copy and use this program freely for any purpose except for sale
 * of the program (including the source code) itself.  That is, 
 * no one can copy this program for the purpose of providing it to 
 * another person in exchange for money or other compensation, even 
 * if this program is only part of the exchange.
 *
 * All copies of computer source code in this distribution, whether
 * copies in whole or in part, must have this notice attached.
 */

/* CANVAS.C -- Display lists of graphic objects and canvases. */

#include <stdio.h>
#include <stdlib.h> 
#include <assert.h>
#include <graphics.h>
#include <mem.h>
#include <values.h>
#include <string.h>
#include <ctype.h>
#include <dir.h>
#include <time.h>
#include "window.h"
#include "settings.h"
#include "canvas.h"
#include "resource.h"

enum {

canvas_border_width = 1,
canvas_border_color = cBLACK,
canvas_edit_color = cEDIT,
canvas_ruler_inc = 100,
canvas_ruler_tick_height = 1,
canvas_pick_color = cPICK,

};


/* ----- Display lists ---------------------------------------------- */

/* Dereference an object index. */
LOCAL(OBJECT) deref(DISPLAY_LIST dl, index i)
{
  assert(i != INULL);
  return dl->blk[i >> 8] + (i & 0xff);
}

OBJECT dereference(CANVAS c, index i)
{
  return (i == INULL) ? NULL : deref(&c->dl, i);
}

/* Allocate a fresh object.  Return its index, and
   a pointer at *op_rtn if op_rtn is non-NULL. */
LOCAL(index) oalloc(DISPLAY_LIST dl,
		    OBJECT_TYPE type,
		    LINE_THICKNESS lt,
		    CC x, CC y,
		    OBJECT *op_rtn)
{
  int i;
  index o, n;
  OBJECT op;

  if (dl->free == INULL) {
    n = dl->n_blks << 8;
    dl->blk[dl->n_blks++] = op = malloc(256 * sizeof(OBJECT_REC));
    for (i = 0; i < 256 - 1; ++i, ++op)
      op->next = n | (i + 1);
    op->next = INULL;
    dl->free = n;
  }
  o = dl->free;
  op = deref(dl, o);
  dl->free = op->next;

  op->type = type;
  op->line_thickness = lt;
  op->ref.x = x;
  op->ref.y = y;
  op->picked_p = op->edit_picked_p = 0;
  op->prev = op->next = o;

  if (op_rtn != NULL)
    *op_rtn = op;
  return o;
}

/* Free object onto display list free list. */
LOCAL(void) ofree(index o, OBJECT op, DISPLAY_LIST dl)
{
  op->type = oFREE;
  op->next = dl->free;
  dl->free = o;
}

/* Remove an object from a loop. Maintain origin invariant: One
   origin on display list, and its index is dl->origin. */
LOCAL(void) odelete2(index o, OBJECT op, DISPLAY_LIST dl)
{
  index next, prev;

  assert(o != INULL);

  if (op->next == o) 
    dl->head = INULL;
  else {
    next = op->next;
    prev = op->prev;
    deref(dl, next)->prev = prev;
    deref(dl, prev)->next = next;
    op->next = op->prev = o;
    if (dl->head == o)
     dl->head = next;
  }
  --dl->n_obj;
  dl->n_picked -= op->picked_p;
  if (o == dl->origin)
    dl->origin = INULL;
}

/* Same as above, but do the deref for the pointer param. */
LOCAL(void) odelete1(index o, DISPLAY_LIST dl) 
{ 
  odelete2(o, deref(dl, o), dl); 
}

/* Enqueue an object last in a circular display list of same. 
   Maintain origin invariant.  If enqueuing an origin over
   an old one, return the old one that was deleted. */
LOCAL(void) oenqueue(index o, DISPLAY_LIST dl, index *org_rtn)
{
  OBJECT op = deref(dl, o);
  index h = dl->head;
  assert(o != INULL);
  if (h == INULL)
    dl->head = o;
  else {
    OBJECT hp = deref(dl, h);
    op->next = h;
    op->prev = hp->prev;
    deref(dl, hp->prev)->next = o;
    hp->prev = o;
  }
  dl->n_picked += op->picked_p;
  ++dl->n_obj;
  if (org_rtn != NULL)
    *org_rtn = INULL;
  if (op->type == oORIGIN) {
    h = dl->origin;
    if (org_rtn != NULL)
      *org_rtn = h;
    if (h != INULL)
      odelete1(h, dl);
    dl->origin = o;
  }
}

/* Set pick flags of a vector of objects. */
LOCAL(void) set_object_pick_flags(DISPLAY_LIST dl, index *o, int n, int on_p)
{
  OBJECT op;
  int i;

  for (i = 0; i < n; ++i) {
    op = deref(dl, o[i]);
    dl->n_picked += on_p - op->picked_p;
    op->picked_p = on_p;
  }
}

/* Note a change in the display list by calling
   the change action.  We could do this in the
   above functions, but it's too slow for once
   per op. */
LOCAL(void) note_dl_change(DISPLAY_LIST dl)
{
  (*dl->change_action.code)(dl, dl->change_action.env);
}

void force_canvas_change(CANVAS c)
{
  note_dl_change(&c->dl);
}

/* ----- Common typdefs and functions for object methods ------------ */

/* Radius of ovals being read in from files.  
   Set by read_objects; used by read_oval. */
static CC read_oval_rad;

/* Convert canvas length to screen length. */
int cc2sc(CANVAS c, CC len)
{
  return scale_int(len, c->draw.width - 1, c->vp_x_size - 1);
}

/* Convert screen length to canvas length. */
CC sc2cc(CANVAS c, int len)
{
  return scale_int(len, c->vp_x_size - 1, c->draw.width - 1);
}

/* Convert canvas x to screen x. */
int cc2scx(CANVAS c, CC x)
{
  return cc2sc(c, x - c->vp_org.x);
}

/* Convert canvas y to screen y. */
int cc2scy(CANVAS c, CC y)
{
  return (c->draw.height - 1) - cc2sc(c, y - c->vp_org.y);
}

/* Convert screen x to canvas x. */
CC sc2ccx(CANVAS c, int x)
{
#ifndef NDEBUG
  CC tmp = sc2cc(c, x) + c->vp_org.x;
  assert(cc2scx(c, tmp) == x);
  return tmp;
#else
  return sc2cc(c, x) + c->vp_org.x;
#endif
}

/* Convert screen y to canvas y. */
CC sc2ccy(CANVAS c, int y)
{
#ifndef NDEBUG
  CC tmp = sc2cc(c, (c->draw.height - 1) - y) + c->vp_org.y;
  assert(cc2scy(c, tmp) == y);
  return tmp;
#else
  return sc2cc(c, (c->draw.height - 1) - y) + c->vp_org.y;
#endif
}

/* Convert screen oval mask to canvas oval mask. */
#define SOM2COM(M)      (((M&1)<<3)|((M&8)>>3)|((M&2)<<1)|((M&4)>>1))
char som2com_tbl[] = { 
  SOM2COM(0),  SOM2COM(1),  SOM2COM(2),  SOM2COM(3),
  SOM2COM(4),  SOM2COM(5),  SOM2COM(6),  SOM2COM(7),
  SOM2COM(8),  SOM2COM(9),  SOM2COM(10), SOM2COM(11),
  SOM2COM(12), SOM2COM(13), SOM2COM(14), SOM2COM(15),
};

/* Convert x and y screen coords to a canvas point. */
CP_REC sp2cp(CANVAS c, int x, int y)
{
  CP_REC p;
  p.x = sc2ccx(c, x);
  p.y = sc2ccy(c, y);
  return p;
}

/* Swap two canvas points. */
void swap_cp_recs(CP p1, CP p2)
{
  CP_REC tmp = *p1;
  *p1 = *p2;
  *p2 = tmp;
}

/* Make p0.x,y <= p1.x,y */
void diagonalize(CP p0, CP p1)
{
  if (p0->x > p1->x) swap(&p0->x, &p1->x);
  if (p0->y > p1->y) swap(&p0->y, &p1->y);
}

/* Structure for a rectangular extent. */
typedef struct {
  int valid_p;
  CP_REC p0, p1;
} EXTENT_REC, *EXTENT;

/* Return non-0 iff point p is in the pix box. */
LOCAL(int) pt_in_pick_box_p(CP_REC p, CANVAS c)
{
  return in_box_p(p.x, p.y, c->pick0.x, c->pick0.y, c->pick1.x, c->pick1.y);
}

/* Copy union portion of generic object. */
LOCAL(void) copy_obj(OBJECT from, OBJECT to)
{
  to->u = from->u;
}

/* Generic object destroyer. */
LOCAL(void) destroy_obj(CANVAS c, index o, OBJECT op)
{

#ifndef NDEBUG
  /* Make sure destroyed object isn't on display list. */
  do_display_list(obj, &c->dl, next, assert(_o != o); );
#endif

  ofree(o, op, &c->dl);
}

/* Compute non-reference point of a line or vector object. */
LOCAL(CP_REC) linear_p1(OBJECT o)
{
  CP_REC p1;

  if (o->u.line.den == 0) {
    p1.x = o->ref.x;
    p1.y = o->ref.y + o->u.line.num * o->u.line.len;
  }
  else {
    CC dx = o->u.line.len;
    if (o->u.line.den < 0)
      dx = -dx;
    p1.x = o->ref.x + dx;
    p1.y = o->ref.y + signed_scale_int(dx, o->u.line.num, o->u.line.den);
  }
  return p1;
}

/* Compute the extent of a line or vector object. */
LOCAL(void) linear_extent(OBJECT o, EXTENT ext)
{
  ext->p0 = o->ref;
  ext->p1 = linear_p1(o);
  diagonalize(&ext->p0, &ext->p1);
  ext->valid_p = 1;
}

/* Convert a canvas coord xxxxx into a 
   string "xxx.xx" */
LOCAL(char *) C(CC c)
{
# define N_CBUFS        8
  static char buf[N_CBUFS][8];
  static int next;
  char *b;
  int i;
  div_t r;

  b = buf[next];
  next = (next + 1) % N_CBUFS;
  i = 0;
  if (c < 0) {
    b[i++] = '-';
    c = -c;
  }
  r = div(c, 100);
  c = r.rem;
  if (r.quot != 0 || c == 0)
    i += sprintf(&b[i], "%d", r.quot);
  if (c > 0) {
    b[i++] = '.';
    b[i++] = c / 10 + '0';
    c %= 10;
    if (c > 0)
      b[i++] = c + '0';
    b[i] = '\0';
  }
  return b;
}

/* exported call point */
char *cc2str(CC c)
{
  return C(c);
}

/* Convert a canvas point (xxxxx,yyyyy)
   into a string "(xxx.xx,yyy.yy)". */
LOCAL(char *) P(CP_REC p)
{
# define N_PBUFS        3
  static char buf[N_PBUFS][18];
  static int next;
  char *b;

  b = buf[next];
  next = (next + 1) % N_PBUFS;
  sprintf(b, "(%s,%s)", C(p.x), C(p.y));
  return b;
}

/* exported call point */
char *cp2str(CP_REC p)
{
  return P(p);
}

/* Write the a multiput command that draws dots for short lines. */
LOCAL(void) write_short_linear(CANVAS c, OBJECT o, FILE *f)
{
  CC size;
  int k;
  CP_REC inc;
  char *sz, *hsz;

  if (o->u.line.num && o->u.line.den && o->u.line.len * c->unitlength < PT2SP(10.0) * 100) {

    size = o->line_thickness == ltTHIN ? 30 : 60;       /* hundredths of a point */
    sz = C(size);
    hsz = C(size/2);
    k = max(1, isqrt(sqr( (size * 65536L) / c->unitlength )
		     / sum_sqr(o->u.line.den, o->u.line.num)));
    inc.x = o->u.line.den * k;
    inc.y = o->u.line.num * k;
    fprintf(f, "\\multiput%s%s{%d}{\\kern-%spt\\vrule depth%spt height%spt width%spt}\n",
	    P(o->ref), P(inc), 
	    o->u.line.len / abs(inc.x) + 1, hsz, hsz, sz, sz);
  }
}

LOCAL(CP_REC) box_p1(OBJECT o)
{
  CP_REC p1;
  p1.x = o->ref.x + o->u.frame_box.sz.x;
  p1.y = o->ref.y + o->u.frame_box.sz.y;
  return p1;
}

/* Compute the extent of a box-like object. */
LOCAL(void) box_extent(OBJECT o, EXTENT ext)
{
  ext->p0 = o->ref;
  ext->p1 = box_p1(o);
  ext->valid_p = 1;
}

/* Return the distance from a line or vector
   to the pick point of a canvas. */
LOCAL(DIST) linear_dist(CANVAS c, OBJECT o)
{
  CP_REC p1 = linear_p1(o);
  return pt_seg_dist(c->pick0.x, c->pick0.y, o->ref.x, o->ref.y, p1.x, p1.y);
}

/* Return non-0 iff a line or vector is inside the pick box of a canvas. */
LOCAL(int) linear_inside_p(CANVAS c, OBJECT o)
{
  return pt_in_pick_box_p(o->ref, c) && pt_in_pick_box_p(linear_p1(o), c);
}

/* Return the distance from a box to the pick point of a given canvas. */
LOCAL(DIST) box_dist(CANVAS c, OBJECT o)
{
  return pt_box_dist(c->pick0.x, c->pick0.y, 
		     o->ref.x, o->ref.y, 
		     o->ref.x + o->u.frame_box.sz.x, 
		     o->ref.y + o->u.frame_box.sz.y);
}

/* Return non-0 iff a box is inside the pick box of a canvas. */
LOCAL(int) box_inside_p(CANVAS c, OBJECT o)
{
  return pt_in_pick_box_p(o->ref, c) && pt_in_pick_box_p(box_p1(o), c);
}

/* Copy the non-default fields of a box to a new box. */
LOCAL(void) copy_box(OBJECT from, OBJECT to)
{
  to->u = from->u;
  to->u.frame_box.str = (from->u.frame_box.str == NULL) ? NULL : strdup(from->u.frame_box.str);
}

/* Return edit information on a box. */
LOCAL(void) box_edit_info(OBJECT o, CP_REC pick, EDIT_INFO info)
{
  int i, min_dist_i;
  long d[4];
  CC x0, y0, x1, y1;

  info->cursor_mask = bit(cBOX);
  x0 = o->ref.x;
  y0 = o->ref.y;
  x1 = x0 + o->u.frame_box.sz.x;
  y1 = y0 + o->u.frame_box.sz.y;
  d[0] = sum_sqr(pick.x - x0, pick.y - y0);
  d[1] = sum_sqr(pick.x - x0, pick.y - y1);
  d[2] = sum_sqr(pick.x - x1, pick.y - y0); 
  d[3] = sum_sqr(pick.x - x1, pick.y - y1); 
  min_dist_i = 0;
  for (i = 1; i < 4; ++i)
    if (d[i] < d[min_dist_i])
      min_dist_i = i;
  switch (min_dist_i) {
    case 0:
      info->p0.x = x1;
      info->p0.y = y1;
      info->p1.x = x0;
      info->p1.y = y0;
      break;

    case 1:
      info->p0.x = x1;
      info->p0.y = y0;
      info->p1.x = x0;
      info->p1.y = y1;
      break;

    case 2:
      info->p0.x = x0;
      info->p0.y = y1;
      info->p1.x = x1;
      info->p1.y = y0;
      break;
  
    case 3:
      info->p0.x = x0;
      info->p0.y = y0;
      info->p1.x = x1;
      info->p1.y = y1;
      break;
  }
}

LOCAL(void) edit_box(OBJECT o, EDIT_INFO info)
{
  diagonalize(&info->p0, &info->p1);
  o->ref = info->p0;
  o->u.frame_box.sz.x = info->p1.x - info->p0.x;
  o->u.frame_box.sz.y = info->p1.y - info->p0.y;
}

/* Destroy a box, including its string. */
LOCAL(void) destroy_box(CANVAS c, index o, OBJECT op)
{
  if (op->u.text.str != NULL)
    free(op->u.text.str);
  destroy_obj(c, o, op);
}

/* Translate box justification fields to LaTeX bracketed pos list. */
LOCAL(char*) box_just_str(char hjust, char vjust)
{
  int i;
  static char buf[8];
  i = 0;
  if (hjust != CENTER_TEXT || vjust != CENTER_TEXT) {
    buf[i++] = '[';
    switch (hjust) {
      case LEFT_TEXT:   buf[i++] = 'l'; break;
      case RIGHT_TEXT:  buf[i++] = 'r'; break;
    }
    switch (vjust) {
      case TOP_TEXT:    buf[i++] = 't'; break;
      case BOTTOM_TEXT: buf[i++] = 'b'; break;
    }
    buf[i++] = ']';
  }
  buf[i] = '\0';
  return buf;
}

/* Convert null string pointer to null string. */
#define stringify(S) ((S) ? (S) : "")

/* Draw an icon to represent text in a box. */
LOCAL(void) draw_text_icon(CANVAS c, OBJECT o)
{
  int x0, y0, x1, y1, tx, ty, icon_width, icon_height;

  if (o->u.text.str == NULL)
    return;
  x0 = cc2scx(c, o->ref.x);
  y0 = cc2scy(c, o->ref.y + o->u.text.sz.y);
  x1 = cc2scx(c, o->ref.x + o->u.text.sz.x);
  y1 = cc2scy(c, o->ref.y);
  icon_width = cc2sc(c, strlen(o->u.text.str) * c->em_size / c->unitlength * (100/2));
  icon_height = cc2sc(c, c->ex_size / c->unitlength * 100);
  switch (o->u.text.hjust) {
    case LEFT_TEXT:     tx = x0;                                break;
    case CENTER_TEXT:   tx = x0 + (x1 - x0 - icon_width)/2;     break;
    case RIGHT_TEXT:    tx = x1 - icon_width + 1;               break;
    default: assert(0);
  }
  switch (o->u.text.vjust) {
    case BOTTOM_TEXT:   ty = y1 - icon_height + 1;              break;
    case CENTER_TEXT:   ty = y0 + (y1 - y0 - icon_height)/2;    break;
    case TOP_TEXT:      ty = y0;                                break;
    default: assert(0);
  }
  if (getcolor() != c->color[ccBACKGROUND])
    setfillstyle(SOLID_FILL, cCYAN);
  bar(tx, ty, tx+icon_width-1, ty+icon_height-1);
}

/* ----- Parsing routines used by object reading functions ---------- */

/* Note all these routines make assumptions about coord 
   transformations between LaTeX and canvas coords.  
   I.e.:  Only y coord                  needs to be transformed.
	  Only numerator of slopes        "    " "       "
	  Lengths need no transformation at all. 
 */

/* Match an arbitrary string token after ignoring whitespace. 
   Return non-0 iff the token matched.  Restore input stream
   on mismatch of initial character. */
int string(FILE *f, char *s)
{
  int ch;

  /* Get first non-white char. */
  while (isspace(ch = getc(f))) 
    /* skip */ ;

  /* Check for match with first char of string. */
  if (ch != *s++) 
    return 0;

  /* Match subsequent chars until string ends or mismatch. */
  while ((ch = *s++) != '\0') 
    if (ch != getc(f)) 
      return 0;

  return 1;
}

/* Skip whitespace. Match one character.  
   Restore non-matching char to input stream. */
LOCAL(int) token(FILE *f, char match_char)
{
  int ch;

  /* Get first non-white char. */
  while (isspace(ch = getc(f))) 
    /* skip */ ;

  /* Check for match. */
  if (ch == match_char)
    return 1;

  /* Restore input and return. */
  ungetc(ch, f);
  return 0;
}

/* Read a decimal number. Return non-0 on success. */
LOCAL(int) length(FILE *f, CC *x)
{
  CC val;
  int ch, d, sign;

  if (fscanf(f, "%d", &val) != 1)
    if (token(f, '.'))
      val = 0;
    else
      return 0;
  else {
    val *= 100;
    if (!token(f, '.')) {
      *x = val;
      return 1;
    }
  }

  if (val < 0) {
    sign = -1;
    val = -val;
  }
  else
    sign = 1;

  for (d = 10; isdigit(ch = getc(f)); d /= 10)
    val += (ch - '0') * d;
  ungetc(ch, f);

  *x = val * sign;

  return 1;
}

/* Read (X,Y) dimensions.  Return non-0 on success. */
LOCAL(int) dimens(FILE *f, CC *x, CC *y)
{
  return token(f, '(') 
      && length(f, x) && token(f, ',') && length(f, y) 
      && token(f, ')');
}

LOCAL(int) slope(FILE *f, CC *n, CC *d)
{
  return (fscanf(f, "(%d,%d)", d, n) == 2);
}

/* Others structures are similar. */
#define xcoord(F,X)     length(F,X)
#define ycoord(F,Y)     length(F,Y)
#define coords(F,X,Y)   dimens(F,X,Y)

/* Read optional [ <optionstring> ].  Return non-0
   if a correct string was read or nothing was read.
   I.e. return 0 if we hit eof without seeing ]. */
LOCAL(int) option(FILE *f, char *s)
{
  int i, ch;

  if (!token(f, '[')) {
    s[0] = '\0';
    return 1;
  }

  i = 0;
  while ((ch = getc(f)) != EOF) {
    if (ch == ']') {
      s[i] = '\0';
      return 1;
    }
    if (i < 7) s[i++] = ch;
  }
  return 0;
}

/* Read a left curly brace. Return non-0 on success. */
LOCAL(int) lcurl(FILE *f)
{
  return token(f, '{');
}

/* Read a right curly brace. Return non-0 on success. */
LOCAL(int) rcurl(FILE *f)
{
  return token(f, '}');
}

/* Check given text to see if it can later be
   read by "latext" below.  Must keep these
   two routines in synch.  This one is used to
   check user entries in dialogs that fetch
   contents of latext in boxes etc. */
int text_is_latext(char *text)
{
  char *p, ch;
  int brace_depth, escape_p;

  escape_p = 0;
  brace_depth = 0;
  p = text;
  while ((ch = *p++) != '\0') {

    switch (ch) {
      case '\\':
	escape_p = 1;
	break;

      case '{':
	if (!escape_p)
	  brace_depth++;
	escape_p = 0;
	break;

      case '}':
	if (!escape_p)
	  brace_depth--;
	escape_p = 0;
	break;

      default:
	escape_p = 0;
	break;
    }
  }
  return (!escape_p && brace_depth == 0);
}

/* Scoop up latex source into a malloc()'ed 
   buffer, ignoring its content except for 
   brace nesting. Ignore enclosing braces. */
LOCAL(int) latext(FILE *f, char **rtn)
{

# define BUF_ALLOC_SIZE 256
  char *text_buf;
  int brace_depth, ch, i, size, escape_p;

  *rtn = NULL;

  if (!lcurl(f)) 
    return 0;

  escape_p = 0;     
  brace_depth = 1;

  /* Do initial alloc. */
  text_buf = malloc(BUF_ALLOC_SIZE);
  i = 0;
  size = BUF_ALLOC_SIZE;

  /* Get text until matching brace. */
  do {
    if ((ch = getc(f)) == EOF) {
      free(text_buf);
      return 0;
    }

    /* Grow text buffer if necessary. */
    if (i == size)
      text_buf = realloc(text_buf, size += BUF_ALLOC_SIZE);

    /* Add char to buf and adjust depth count. */
    switch (text_buf[i++] = ch) {
      case '\\': 
	escape_p = 1; 
	break;

      case '{': 
	if (!escape_p)
	  ++brace_depth; 
	escape_p = 0;
	break;

      case '}': 
	if (!escape_p) 
	  --brace_depth; 
	escape_p = 0;
	break;

      default:
	escape_p = 0;
	break;
    }
  } while (brace_depth > 0);

  /* Terminate string in buffer, overwriting final brace. */
  text_buf[i - 1] = '\0';

  /* Shrink to fit text. */
  if (text_buf[0] == '\0' || !rtn) 
    free(text_buf);
  else 
    *rtn = realloc(text_buf, i);
  return 1;
}

/* Set hjust and vjust based on optional
   [ <justchars> ] read from input.  Follow
   LaTeX rules. */
LOCAL(int) just(FILE *f, int *hjust, int *vjust)
{
  int i;
  char buf[8];

  if (!option(f, buf))
    return 0;
  *hjust = *vjust = CENTER_TEXT;
  for (i = 0; ;++i) 
    switch(buf[i])  {
      case '\0':                        return 1;
      case 'l': *hjust = LEFT_TEXT;     break;
      case 'r': *hjust = RIGHT_TEXT;    break;
      case 't': *vjust = TOP_TEXT;      break;
      case 'b': *vjust = BOTTOM_TEXT;   break;
    }
}

/* String tokens that commands we can parse start with. */
typedef enum { 
  tEOF = -2,
  tERROR = -1,
  tBEGIN,
  tCAPTION,
  tCENTERING,
  tDOCUMENTSTYLE,
  tEND,
  tFBOX,
  tLABEL,
  tMULTIPUT,
  tNEWCOMMAND,
  tNEWENVIRONMENT,
  tPAGESTYLE,
  tPUT, 
  tRCURLTAG,
  tRENEWCOMMAND,
  tTHICK_LINES, 
  tTHIN_LINES, 
  tUNIT_LENGTH,
} LOOK_AHEAD;

static char *look_ahead_tbl[] = {
  "begin",
  "caption",
  "centering",
  "documentstyle",
  "end",
  "fbox",
  "label",
  "multiput",
  "newcommand",
  "newenvironment",
  "pagestyle",
  "put",
  "rcurltag",
  "renewcommand",
  "thicklines",
  "thinlines",
  "unitlength",
};

/* Read alphabetic chars into a buffer until
   the first non-alpha is found. Include * 
   for filled circles. */
LOCAL(void) _alphas(FILE *f, char *buf, int n)
{
  int i, ch;

  i = 0;
  while (isalpha(ch = getc(f)) || ch == '*') 
    if (i < n)
      buf[i++] = ch;
  buf[i] = '\0';
  ungetc(ch, f);
}

/* Use size of buffer with above for safety. */
#define alphas(f, buf)  _alphas(f, buf, sizeof buf - 1)

/* Comparison predicate for finding entries
   in units and command tables with bsearch(). */
int strpcmp(const void *v1, const void *v2)
{
  return strcmp(v1, *(char**)v2);
}

/* Read input stream until char or eof. */
LOCAL(int) skip_to(FILE *f, char the_char)
{
  int ch;
  while ((ch = getc(f)) != EOF && ch != the_char)
    /* skip */;
  return ch != EOF;
}

/* Skip comments. Return the next lookahead 
   token on the input stream. */
LOCAL(LOOK_AHEAD) look_ahead(FILE *f)
{
  char **p, buf[32];

  /* Skip comments. */
  while (token(f, '%'))
    skip_to(f, '\n');
  if (feof(f))
    return tEOF;
  if (getc(f) != '\\')
    return tERROR;
  alphas(f, buf);
  p = str_search(buf, look_ahead_tbl);
  return (p != NULL) ? p - look_ahead_tbl : tERROR;
}

/* Tokens and strings for environment names. */
typedef enum { 
  eERROR = -1,
  eCENTER,
  eDOCUMENT,
  eFIGURE,
  ePICTURE,
  eTEDDISPLAY,
} ENVIRONMENT_TYPE;
 
static char *environment_tbl[] = {
  "center",
  "document",
  "figure",
  "picture",
  "teddisplay",
};

/* Skip comments. Return the next lookahead 
   token on the input stream. */
LOCAL(ENVIRONMENT_TYPE) environment(FILE *f)
{
  char **p, buf[32];

  if (!lcurl(f))
    return eERROR;
  alphas(f, buf);
  if (!rcurl(f))
    return eERROR;
  p = str_search(buf, environment_tbl);
  return (p != NULL) ? p - environment_tbl: tERROR;
}


LOCAL(int) unitlength(FILE *f, SP *unitlength_rtn, char **str, UNITS_TYPE *units_rtn)
{
  char buf[80];

  token(f, '=');
  read_TeXlength_str(f, buf, sizeof buf - 1);  
  if (!str2TeXlength(buf, unitlength_rtn, units_rtn) || unitlength == 0)
    return 0;
  if (str != NULL)
    *str = strdup(buf);
  return 1;
}

/* Read the coordinate after \put and lex the following command token. */
LOCAL(int) prefix(FILE *f, CC *x, CC *y, OBJECT_TYPE *object_type)
{
  char **p;
  char buf[32];

  static char *object_cmd_tbl[] = {
#   define DEF_OBJECT(Tag, Id, Fields, Cmd, BoxTag, LTtag) #Cmd,
#   include "obj.def"
#   undef DEF_OBJECT
  };

  if (!coords(f, x, y) || !lcurl(f) || !token(f, '\\'))
    return 0;

  alphas(f, buf);
  p = str_search(buf, object_cmd_tbl);

  if (p == NULL)
    return 0;

  *object_type = (OBJECT_TYPE)(p - object_cmd_tbl);
  return 1;
}

LOCAL(int) waste_multiput(FILE *f)
{
  int x[1], y[1];

  return coords(f, x, y) && coords(f, x, y) && 
	 lcurl(f) && length(f, x) && rcurl(f) &&
	 lcurl(f) && skip_to(f, '}');
}

LOCAL(int) waste_braced_args(FILE *f, int n)
{
  while (n--)
    if (!lcurl(f) || !skip_to(f, '}'))
      return 0;
  return 1;
}

LOCAL(int) waste_teddisplay_def(FILE *f)
{
  return waste_braced_args(f, 2) && rcurl(f) && /* ugly but good enough */
	 waste_braced_args(f, 1) && rcurl(f);
	 
}

/* ----- Origin ----------------------------------------------------- */

/* Build a fresh origin object. */
index make_origin(CANVAS c, CC x, CC y, CC width, CC height, int auto_p)
{
  index o;
  OBJECT op;

  o = oalloc(&c->dl, oORIGIN, ltNONE, x, y, &op);
  op->ref.x = x;
  op->ref.y = y;
  op->u.origin.sz.x = width;
  op->u.origin.sz.y = height;
  op->u.origin.auto_p = auto_p;
  return o;
}

/* Draw an origin object. */
LOCAL(void) draw_origin(CANVAS c, OBJECT o)
{
  int x0, y0;

  if (o->u.origin.auto_p)
    setlinestyle(DOTTED_LINE, 0, 1);

  rectangle(x0 = cc2scx(c, o->ref.x),
	    y0 = cc2scy(c, o->ref.y),
	    cc2scx(c, o->ref.x + o->u.origin.sz.x),
	    cc2scy(c, o->ref.y + o->u.origin.sz.y));
  circle(x0, y0, 2);
  setlinestyle(SOLID_LINE, 0, 1);
}

/* Origin reader is not called like other objects. */
#define read_origin NULL

LOCAL(int) waste_picture(FILE *f)
{
  CC x, y, width, height;

  if (dimens(f, &width, &height)) {
    coords(f, &x, &y);
    if (token(f, '%'))
      skip_to(f, '\n');
    return 1;
  }
  return 0;
}

/* Read picture statement after \begin. */
LOCAL(index) read_picture(CANVAS c, FILE *f)
{
  int auto_p;
  CC x, y, width, height;

  if (!dimens(f, &width, &height))
    return INULL;
  if (!coords(f, &x, &y)) 
    x = y = 0;
  if (token(f, '%')) {
    auto_p = token(f, 'a');
    skip_to(f, '\n');
  }
  else
    auto_p = 0;
  return make_origin(c, x, y, width, height, auto_p);
}

#pragma argsused
LOCAL(void) write_origin(CANVAS c, OBJECT o, FILE *f)
{
  fprintf(f, "\\begin{picture}%s%s%%%s\n",
	  P(o->u.origin.sz), P(o->ref), o->u.origin.auto_p ? "auto" : "user");
}

#define origin_dist             box_dist
#define origin_inside_p         box_inside_p

#pragma argsused
LOCAL(void) origin_extent(OBJECT o, EXTENT ext) 
{ 
  memset(ext, 0, sizeof *ext);
}

#pragma argsused
LOCAL(void) copy_origin(OBJECT from, OBJECT to) 
{ 
  to->u.origin.auto_p = 0;
}

#define origin_edit_info        box_edit_info
#define edit_origin             edit_box
#define destroy_origin          destroy_obj

/* ----- Lines ------------------------------------------------------ */

index make_line(CANVAS c,
		LINE_THICKNESS lt,
		CC x, CC y,
		CC len,
		int num, int den,
		CP p1_rtn)
{
  index o;
  OBJECT op;

  o = oalloc(&c->dl, oLINE, lt, x, y, &op);
  op->u.line.len = len;
  op->u.line.num = num;
  op->u.line.den = den;
  if (p1_rtn != NULL)
    *p1_rtn = linear_p1(op);
  return o;
}

LOCAL(void) draw_line(CANVAS c, OBJECT o)
{
  CP_REC p1 = linear_p1(o);
  line(cc2scx(c, o->ref.x),
       cc2scy(c, o->ref.y),
       cc2scx(c, p1.x),
       cc2scy(c, p1.y));
}

LOCAL(index) read_line(CANVAS c, FILE *f, LINE_THICKNESS lt, CC x, CC y)
{
  CC num, den, len;
  return (slope(f, &num, &den) && 
	  lcurl(f) && 
	  length(f, &len) && 
	  rcurl(f)) ?
    make_line(c, lt, x, y, len, num, den, NULL) : INULL;
}

LOCAL(void) write_line(CANVAS c, OBJECT o, FILE *f)
{
  write_short_linear(c, o, f);
  fprintf(f, "\\put%s{\\line(%d,%d){%s}}\n",
	     P(o->ref), o->u.line.den, o->u.line.num, C(o->u.line.len));
}

#define line_dist       linear_dist
#define line_inside_p   linear_inside_p
#define line_extent     linear_extent
#define copy_line       copy_obj

LOCAL(void) line_edit_info(OBJECT o, CP_REC pick, EDIT_INFO info)
{
  CP_REC l_p1 = linear_p1(o);
  if (sum_sqr(pick.x - o->ref.x, pick.y - o->ref.y) < sum_sqr(pick.x - l_p1.x, pick.y - l_p1.y)) {
    info->p0 = l_p1;
    info->p1 = o->ref;
  }
  else { 
    info->p0 = o->ref;
    info->p1 = l_p1;
  }
  info->cursor_mask = bit(cLINE);
}

LOCAL(void) edit_line(OBJECT o, EDIT_INFO info)
{
  o->ref = info->p0;
  o->u.line.len = abs(info->p1.x == info->p0.x ? 
		      info->p1.y -  info->p0.y : 
		      info->p1.x -  info->p0.x);
  o->u.line.num = info->slope_num;
  o->u.line.den = info->slope_den;
}

#define destroy_line    destroy_obj

/* ----- Vectors ---------------------------------------------------- */

index make_vector(CANVAS c,
		  LINE_THICKNESS lt,
		  CC x, CC y,
		  CC len,
		  int num, int den,
		  CP p1_rtn)
{
  index o;
  OBJECT op;

  o = oalloc(&c->dl, oVECTOR, lt, x, y, &op);
  op->u.vector.len = len;
  op->u.vector.num = num;
  op->u.vector.den = den;
  if (p1_rtn != NULL)
    *p1_rtn = linear_p1(op);
  return o;
}

LOCAL(void) draw_vector(CANVAS c, OBJECT o)
{
  CP_REC p1;
  CC sx0, sy0, sx1, sy1;

  p1 = linear_p1(o);
  sx0 = cc2scx(c, o->ref.x);
  sy0 = cc2scy(c, o->ref.y);
  sx1 = cc2scx(c, p1.x);
  sy1 = cc2scy(c, p1.y);
  line(sx0, sy0, sx1, sy1);
  arrowhead(sx0, sy0, sx1, sy1);
}

LOCAL(index) read_vector(CANVAS c, FILE *f, LINE_THICKNESS lt, CC x, CC y)
{
  CC num, den, len;
  return (slope(f, &num, &den) && lcurl(f) && length(f, &len) && rcurl(f)) ?
    make_vector(c, lt, x, y, len, num, den, NULL) : INULL;
}

#pragma argsused
LOCAL(void) write_vector(CANVAS c, OBJECT o, FILE *f)
{
  write_short_linear(c, o, f);
  fprintf(f, "\\put%s{\\vector(%d,%d){%s}}\n",
	     P(o->ref), o->u.vector.den, o->u.vector.num, C(o->u.vector.len));
}

#define vector_dist     linear_dist
#define vector_inside_p linear_inside_p
#define vector_extent   linear_extent
#define copy_vector     copy_obj

LOCAL(void) vector_edit_info(OBJECT o, CP_REC pick, EDIT_INFO info)
{
  CP_REC l_p1 = linear_p1(o);
  if (sum_sqr(pick.x - o->ref.x, pick.y - o->ref.y) < sum_sqr(pick.x - l_p1.x, pick.y - l_p1.y)) {
    info->p0 = l_p1;
    info->p1 = o->ref;
    info->cursor_mask = bit(cVECTOR)|bit(cVARIANT);
  }
  else { 
    info->p0 = o->ref;
    info->p1 = l_p1;
    info->cursor_mask = bit(cVECTOR);
  }
}

LOCAL(void) edit_vector(OBJECT o, EDIT_INFO info)
{
  if (info->cursor_mask & bit(cVARIANT)) {
    swap_cp_recs(&info->p0, &info->p1);
    neg(&info->slope_num);
    neg(&info->slope_den);
  }
  o->ref = info->p0;
  o->u.vector.len = abs(info->p1.x == info->p0.x ? 
		      info->p1.y -  info->p0.y : 
		      info->p1.x -  info->p0.x);
  o->u.vector.num = info->slope_num;
  o->u.vector.den = info->slope_den;
}

#define destroy_vector  destroy_obj

/* ----- Ovals ------------------------------------------------------ */

index make_oval(CANVAS c, 
		LINE_THICKNESS lt,
		CC x, CC y, 
		CC width, CC height, CC rad,
		unsigned mask)
{
  index o;
  OBJECT op;

  o = oalloc(&c->dl, oOVAL, lt, x, y, &op);
  op->u.oval.sz.x = width;
  op->u.oval.sz.y = height;
  op->u.oval.mask = mask;
  op->u.oval.rad = rad;
  return o;
}

LOCAL(void) draw_oval(CANVAS c, OBJECT o)
{
  oval(cc2scx(c, o->ref.x), 
       cc2scy(c, o->ref.y),
       cc2scx(c, o->ref.x + o->u.oval.sz.x), 
       cc2scy(c, o->ref.y + o->u.oval.sz.y),
       cc2sc(c, o->u.oval.rad), 
       com2som(o->u.oval.mask));
}


static enum {_=0,t='t',b='b',r='r',l='l'};
static char/*0,1,2,3,4,5,6,7,8,9,a,b,c,d,e,f */
  x_ofs[] = {_,0,1,2,1,_,1,_,0,0,_,_,2,_,_,2},
  y_ofs[] = {_,0,0,0,1,_,2,_,1,2,_,_,1,_,_,2},
  w_fac[] = {_,2,2,1,2,_,2,_,2,2,_,_,1,_,_,1},
  h_fac[] = {_,2,2,2,2,_,1,_,2,1,_,_,2,_,_,1},
  c1[]    = {_,t,t,t,b,_,l,_,b,r,_,_,b,_,_,_},
  c2[]    = {_,r,l,_,l,_,_,_,r,_,_,_,_,_,_,_};

LOCAL(index) read_oval(CANVAS c, FILE *f, LINE_THICKNESS lt, CC x, CC y)
{
  CC w, h;
  int i, j;
  unsigned char mask;
  char buf[8];

  if (!dimens(f, &w, &h) || !option(f, buf))
    return INULL;

  /* Build mask from option chars. */
  mask = 15;
  for (i = 0; buf[i]; ++i) {
    static unsigned char 
      masks[] = { ~12, ~3, ~9, ~6 },
      chars[] = { 't', 'b','l','r'};
    for (j = 0; j < sizeof chars; ++j) {
      if (buf[i] == chars[j]) {
	mask &= masks[j];
	break;
      }
    }
  }
  /* Do inverse of write transformation. */
  w /= w_fac[mask];
  h /= h_fac[mask];
  if (x_ofs[mask]) x -= w/x_ofs[mask];
  if (y_ofs[mask]) y -= h/y_ofs[mask];
  return make_oval(c, lt, x, y, w, h, read_oval_rad, mask);
}

#pragma argsused
LOCAL(void) write_oval(CANVAS c, OBJECT o, FILE *f)
{
  CP_REC p, sz;
  unsigned char mask;
  char buf[8];
  int i;

  /* Convert from our representation of ovals to LaTeX's */

  p = o->ref;
  sz = o->u.oval.sz;
  mask = o->u.oval.mask;
  assert(mask <= 15);

  /* Offset origin by half or all of width/height as necessary. */
  if (x_ofs[mask]) p.x += sz.x/x_ofs[mask];
  if (y_ofs[mask]) p.y += sz.y/y_ofs[mask];

  /* Adjust width/height as necessary. */
  sz.x *= w_fac[mask];
  sz.y *= h_fac[mask];
  
  /* Build option string for portion of oval. */    
  i = 0;
  if (mask != 0xf) {
    assert(mask <= 12);
    buf[i++] = '[';
    if (c1[mask]) buf[i++] = c1[mask];
    if (c2[mask]) buf[i++] = c2[mask];
    buf[i++] = ']';
  }
  buf[i] = '\0';

  fprintf(f, "\\put%s{\\oval%s%s}\n", P(p), P(sz), buf);
}

LOCAL(DIST) oval_dist(CANVAS c, OBJECT o)
{
  return pt_oval_dist(c->pick0.x, c->pick0.y, 
		      o->ref.x, o->ref.y,
		      o->ref.x + o->u.oval.sz.x, o->ref.y + o->u.oval.sz.y,
		      o->u.oval.rad,
		      o->u.oval.mask);
}

#define oval_inside_p   box_inside_p
#define oval_extent     box_extent
#define copy_oval       copy_obj

LOCAL(void) oval_edit_info(OBJECT o, CP_REC pick, EDIT_INFO info)
{
  CC x0, y0, x1, y1, dx[4], dy[4];
  long dist2, new_dist2;
  int i, c;
  unsigned mask;
  unsigned char tbl_val;
  enum { __ = 0, 
	 prime = 0x80,
	 h  = cOVAL4H,
	 hp = cOVAL4H | prime, 
	 v  = cOVAL4V,
	 vp = cOVAL4V | prime,
	 H  = cOVAL2H,
	 Hp = cOVAL2H | prime,
	 V  = cOVAL2V,
	 Vp = cOVAL2V | prime,
	 O  = cOVAL };
  static const unsigned char cursor_tbl[][4] = {
  /* 0   1   2   3 */
    __, __, __, __,     /* 0 */
    hp,  v, __,  h,     /* 1 */
     v, hp,  h, __,     /* 2 */
     V,  V, Vp, Vp,     /* 3 */
    __,  h, hp,  v,     /* 4 */
    __, __, __, __,     /* 5 */
    Hp,  H,  H, Hp,     /* 6 */
    __, __, __, __,     /* 7 */
     h, __,  v, hp,     /* 8 */
     H, Hp, Hp,  H,     /* 9 */
    __, __, __, __,     /* 10 */
    __, __, __, __,     /* 11 */
    Vp, Vp,  V,  V,     /* 12 */
    __, __, __, __,     /* 13 */
    __, __, __, __,     /* 14 */
     O,  O,  O,  O,     /* 15 */
  };

  x0 = o->ref.x;
  y0 = o->ref.y;
  x1 = x0 + o->u.oval.sz.x;
  y1 = y0 + o->u.oval.sz.y;
  dx[0] = dx[3] = pick.x - x1;
  dx[1] = dx[2] = pick.x - x0;
  dy[0] = dy[1] = pick.y - y1;
  dy[2] = dy[3] = pick.y - y0;

  c = -1;
  for (i = 0; i < 4; ++i)
    if (cursor_tbl[o->u.oval.mask][i]) {
      new_dist2 = sum_sqr(dx[i], dy[i]);  
      if (c == -1 || new_dist2 < dist2) {
	c = i;
	dist2 = new_dist2;
      }
    }
  if (c < 2) {
    info->p1.y = y1;
    info->p0.y = y0;
  }
  else {
    info->p1.y = y0;
    info->p0.y = y1;
  }
  if (c == 0 || c == 3) {
    info->p1.x = x1;
    info->p0.x = x0;
  }
  else {
    info->p1.x = x0;
    info->p0.x = x1;
  }
  tbl_val = cursor_tbl[o->u.oval.mask][c];
  mask = bit(tbl_val & ~prime);
  if (tbl_val & prime)
    mask |= bit(cVARIANT);
  info->cursor_mask = mask;
}

LOCAL(void) edit_oval(OBJECT o, EDIT_INFO info)
{
  diagonalize(&info->p0, &info->p1);
  o->ref = info->p0;
  o->u.oval.sz.x = info->p1.x - info->p0.x;
  o->u.oval.sz.y = info->p1.y - info->p0.y;
  o->u.oval.mask = som2com(info->oval_mask);
}

#define destroy_oval    destroy_obj

/* ----- Circle ----------------------------------------------------- */

index make_circle(CANVAS c,
		  LINE_THICKNESS lt,
		  CC x, CC y, CC rad)
{
  index o;
  OBJECT op;

  o = oalloc(&c->dl, oCIRCLE, lt, x, y, &op);
  op->u.circle.rad = rad;
  return o;
}

LOCAL(void) draw_circle(CANVAS c, OBJECT o)
{
  circle(cc2scx(c, o->ref.x), cc2scy(c, o->ref.y), cc2sc(c, o->u.circle.rad));
}

LOCAL(index) read_circle(CANVAS c, FILE *f, LINE_THICKNESS lt, CC x, CC y)
{
  CC diam;

  return lcurl(f) && length(f, &diam) && rcurl(f) ?
    make_circle(c, lt, x, y, diam/2) : INULL;
}

#pragma argsused
LOCAL(void) write_circle(CANVAS c, OBJECT o, FILE *f)
{
  fprintf(f, "\\put%s{\\circle{%s}}\n", P(o->ref), C(2*o->u.circle.rad));
}

LOCAL(DIST) circle_dist(CANVAS c, OBJECT o)
{
  return abs(o->u.circle.rad - 
	     isqrt(sum_sqr(c->pick0.x - o->ref.x, 
			   c->pick0.y - o->ref.y)));
}

LOCAL(int) circle_inside_p(CANVAS c, OBJECT o)
{
  return o->ref.x - o->u.circle.rad >= c->pick0.x &&
	 o->ref.x + o->u.circle.rad <= c->pick1.x &&
	 o->ref.y - o->u.circle.rad >= c->pick0.y &&
	 o->ref.y + o->u.circle.rad <= c->pick1.y;
}

LOCAL(void) circle_extent(OBJECT o, EXTENT ext)
{
  ext->p0.x = o->ref.x - o->u.circle.rad;
  ext->p0.y = o->ref.y - o->u.circle.rad;
  ext->p1.x = o->ref.x + o->u.circle.rad;
  ext->p1.y = o->ref.y + o->u.circle.rad;
  ext->valid_p = 1;
}

#define copy_circle     copy_obj

#pragma argsused
LOCAL(void) circle_edit_info(OBJECT o, CP_REC pick, EDIT_INFO info)
{
  info->p0 = o->ref;
  info->cursor_mask = bit(cCIRCLE_RAD);
}

LOCAL(void) edit_circle(OBJECT o, EDIT_INFO info)
{
  o->u.circle.rad = isqrt(sum_sqr(info->p1.x - info->p0.x, info->p1.y - info->p0.y));
}

#define destroy_circle  destroy_obj

/* ----- Filled circle ---------------------------------------------- */

index make_fill_circle(CANVAS c,
		       LINE_THICKNESS lt,
		       CC x, CC y, CC rad)
{
  index o;
  OBJECT op;

  o = oalloc(&c->dl, oFILL_CIRCLE, lt, x, y, &op);
  op->u.fill_circle.rad = rad;
  return o;
}

LOCAL(void) draw_fill_circle(CANVAS c, OBJECT o)
{
  int rad = cc2sc(c, o->u.fill_circle.rad);
  fillellipse(cc2scx(c, o->ref.x), cc2scy(c, o->ref.y), rad, rad);
}

LOCAL(index) read_fill_circle(CANVAS c, FILE *f, LINE_THICKNESS lt, CC x, CC y)
{
  CC diam;

  return lcurl(f) && length(f, &diam) && rcurl(f) ?
    make_fill_circle(c, lt, x, y, diam/2) : INULL;
}

#pragma argsused
LOCAL(void) write_fill_circle(CANVAS c, OBJECT o, FILE *f)
{
  fprintf(f, "\\put%s{\\circle*{%s}}\n", P(o->ref), C(2*o->u.circle.rad));
}

LOCAL(DIST) fill_circle_dist(CANVAS c, OBJECT o)
{
  return max(0, isqrt(sum_sqr(c->pick0.x - o->ref.x, c->pick0.y - o->ref.y))
		- o->u.circle.rad);
}

#define fill_circle_inside_p    circle_inside_p
#define fill_circle_extent      circle_extent
#define copy_fill_circle        copy_obj
#define fill_circle_edit_info   circle_edit_info
#define edit_fill_circle        edit_circle
#define destroy_fill_circle     destroy_obj

/* ----- Filled box ------------------------------------------------- */

#pragma argsused

index make_fill_box(CANVAS c,
		    LINE_THICKNESS dummy1,
		    CC x, CC y,
		    CC width, CC height,
		    char *dummy2,
		    int dummy3, int dummy4,
		    int dummy5)
{
  index o;
  OBJECT op;

  o = oalloc(&c->dl, oFILL_BOX, ltNONE, x, y, &op);
  op->u.fill_box.sz.x = width;
  op->u.fill_box.sz.y = height;
  return o;
}

LOCAL(void) draw_fill_box(CANVAS c, OBJECT o)
{
  bar(cc2scx(c, o->ref.x),
      cc2scy(c, o->ref.y),
      cc2scx(c, o->ref.x + o->u.fill_box.sz.x),
      cc2scy(c, o->ref.y + o->u.fill_box.sz.y));
}

LOCAL(index) read_fill_box(CANVAS c, FILE *f, LINE_THICKNESS lt, CC x, CC y)
{
  CC width, height;

  return lcurl(f) && length(f, &width) && string(f, "\\unitlength") && rcurl(f)
      && lcurl(f) && length(f, &height) && string(f, "\\unitlength") && rcurl(f) ?
    make_fill_box(c, lt, x, y, width, height, NULL,0,0,0) : INULL;
}

#pragma argsused
LOCAL(void) write_fill_box(CANVAS c, OBJECT o, FILE *f)
{
  fprintf(f, "\\put%s{\\rule{%s\\unitlength}{%s\\unitlength}}\n",
	  P(o->ref), C(o->u.fill_box.sz.x), C(o->u.fill_box.sz.y));
}

LOCAL(DIST) fill_box_dist(CANVAS c, OBJECT o)
{
  return in_box_p(c->pick0.x, c->pick0.y, 
		  o->ref.x, o->ref.y, 
		  o->ref.x + o->u.fill_box.sz.x, 
		  o->ref.y + o->u.fill_box.sz.y) ? 0 : box_dist(c, o);
}

#define fill_box_inside_p       box_inside_p
#define fill_box_extent         box_extent
#define copy_fill_box           copy_obj
#define fill_box_edit_info      box_edit_info
#define edit_fill_box           edit_box
#define destroy_fill_box        destroy_obj

/* ----- Justified text --------------------------------------------- */

#pragma argsused

index make_text(CANVAS c,
		LINE_THICKNESS dummy1,
		CC x, CC y,
		CC width, CC height,
		char *str,
		int hjust, int vjust,
		CC dummy2)
{
  index o;
  OBJECT op;

  o = oalloc(&c->dl, oTEXT, ltNONE, x, y, &op);
  op->u.text.sz.x = width;
  op->u.text.sz.y = height;
  op->u.text.str = str;
  op->u.text.hjust = hjust;
  op->u.text.vjust = vjust;
  return o;
}

LOCAL(void) draw_text_box_icon(int x1, int y1, int x2, int y2)
{
  enum { visible_size = 4 };

  if (x2 - x1 > visible_size || y2 - y1 > visible_size) {
    setlinestyle(USERBIT_LINE, 0x1111, 1);
    rectangle(x1, y1, x2, y2);
    setlinestyle(SOLID_LINE, 0, 1);
  }
  else {
    line(x1, y1 + visible_size/2, x1, y2 - visible_size/2);
    line(x2, y1 + visible_size/2, x2, y2 - visible_size/2);
    line(x1 - visible_size/2, y1, x2 + visible_size/2, y1);
    line(x1 - visible_size/2, y2, x2 + visible_size/2, y2);
  }
}

LOCAL(void) draw_text(CANVAS c, OBJECT o)
{
  draw_text_icon(c, o);
  draw_text_box_icon(cc2scx(c, o->ref.x),
		     cc2scy(c, o->ref.y),
		     cc2scx(c, o->ref.x + o->u.dash_box.sz.x),
		     cc2scy(c, o->ref.y + o->u.dash_box.sz.y));
}

LOCAL(index) read_text(CANVAS c, FILE *f, LINE_THICKNESS lt, CC x, CC y)
{
  CC width, height;
  int hjust, vjust;
  char *str;

  return dimens(f, &width, &height) && just(f, &hjust, &vjust) && latext(f, &str) ?
    make_text(c, lt, x, y, width, height, str, hjust, vjust, 0) : INULL;
}

#pragma argsused
LOCAL(void) write_text(CANVAS c, OBJECT o, FILE *f)
{
  fprintf(f, "\\put%s{\\makebox%s%s{%s}}\n",
	  P(o->ref), P(o->u.text.sz), 
	  box_just_str(o->u.text.hjust, o->u.text.vjust), 
	  stringify(o->u.text.str));
}

#define text_dist       box_dist
#define text_inside_p   box_inside_p
#define text_extent     box_extent
#define copy_text       copy_box
#define text_edit_info  box_edit_info
#define edit_text       edit_box
#define destroy_text    destroy_box

/* ----- Framed text ------------------------------------------------ */

#pragma argsused

index make_frame_box(CANVAS c,
		     LINE_THICKNESS lt,
		     CC x, CC y,
		     CC width, CC height,
		     char *str,
		     int hjust, int vjust,
		     CC dummy) 
{
  index o;
  OBJECT op;

  o = oalloc(&c->dl, oFRAME_BOX, lt, x, y, &op);
  op->u.frame_box.sz.x = width;
  op->u.frame_box.sz.y = height;
  op->u.frame_box.str = str;
  op->u.frame_box.hjust = hjust;
  op->u.frame_box.vjust = vjust;
  return o;
}

LOCAL(void) draw_frame_box(CANVAS c, OBJECT o)
{
  draw_text_icon(c, o);
  rectangle(cc2scx(c, o->ref.x),
	    cc2scy(c, o->ref.y),
	    cc2scx(c, o->ref.x + o->u.frame_box.sz.x),
	    cc2scy(c, o->ref.y + o->u.frame_box.sz.y));
}

LOCAL(index) read_frame_box(CANVAS c, FILE *f, LINE_THICKNESS lt, CC x, CC y)
{
  CC width, height;
  int hjust, vjust;
  char *str;

  return dimens(f, &width, &height) && just(f, &hjust, &vjust) && latext(f, &str) ?
    make_frame_box(c, lt, x, y, width, height, str, hjust, vjust, 0) : INULL;
}

#pragma argsused
LOCAL(void) write_frame_box(CANVAS c, OBJECT o, FILE *f)
{
  fprintf(f, "\\put%s{\\framebox%s%s{%s}}\n",
	  P(o->ref), P(o->u.frame_box.sz), 
	  box_just_str(o->u.frame_box.hjust, o->u.frame_box.vjust),
	  stringify(o->u.frame_box.str));
}

#define frame_box_dist          box_dist
#define frame_box_inside_p      box_inside_p
#define frame_box_extent        box_extent
#define copy_frame_box          copy_box
#define frame_box_edit_info     box_edit_info
#define edit_frame_box          edit_box
#define destroy_frame_box       destroy_box

/* ----- Dash-line framed text -------------------------------------- */

index make_dash_box(CANVAS c,
		    LINE_THICKNESS lt,
		    CC x, CC y,
		    CC width, CC height,
		    char *str,
		    int hjust, int vjust,
		    CC dash_len) 
{
  index o;
  OBJECT op;

  o = oalloc(&c->dl, oDASH_BOX, lt, x, y, &op);
  op->u.dash_box.sz.x = width;
  op->u.dash_box.sz.y = height;
  op->u.dash_box.str = str;
  op->u.dash_box.hjust = hjust;
  op->u.dash_box.vjust = vjust;
  op->u.dash_box.dash_len = dash_len;
  return o;
}

TYPEDEF_LOCAL(void) (*DASHER)(CANVAS, CC, CC, CC);

LOCAL(void) hdash(CANVAS c, CC x, CC y, CC len)
{
  int sy = cc2scy(c, y);
  line(cc2scx(c, x), sy, cc2scx(c, x + len), sy);
}

LOCAL(void) vdash(CANVAS c, CC y, CC x, CC len)
{
  int sx = cc2scx(c, x);
  line(sx, cc2scy(c, y), sx, cc2scy(c, y + len));
}

LOCAL(void) dashes(CANVAS c, CC v, CC w, CC len, CC dash_len, DASHER dash)
{
  int n;
  CC partial_dash_len;

  if (dash_len <= 0)
    dash_len = 1;
  n = len / dash_len;
  if ((n & 1) == 0) {

    /* LaTeX macro could use this for better-looking dashes:
       partial_dash_len = min(len + (1 - n) * dash_len, len);
       partial_dash_len /= 2; */

    partial_dash_len = min(dash_len, len) / 2;
    (*dash)(c, v, w, partial_dash_len);
    (*dash)(c, v + len - partial_dash_len, w, partial_dash_len);
    v += partial_dash_len + dash_len;
    n -= 2;
  }
  while (n > 0) {
    (*dash)(c, v, w, dash_len);
    v += 2 * dash_len;
    n -= 2;
  }
}

LOCAL(void) draw_dash_box(CANVAS c, OBJECT o)
{
  CC x, y, w, h, dash_len;

  draw_text_icon(c, o);
  x = o->ref.x;
  y = o->ref.y;
  w = o->u.dash_box.sz.x;
  h = o->u.dash_box.sz.y;
  dash_len = o->u.dash_box.dash_len;
  dashes(c, x, y,     w, dash_len, hdash);
  dashes(c, x, y + h, w, dash_len, hdash);
  dashes(c, y, x,     h, dash_len, vdash);
  dashes(c, y, x + w, h, dash_len, vdash);
}

LOCAL(index) read_dash_box(CANVAS c, FILE *f, LINE_THICKNESS lt, CC x, CC y)
{
  CC width, height, dash_len;
  int hjust, vjust;
  char *str;

  return lcurl(f) && length(f, &dash_len) && rcurl(f)
      && dimens(f, &width, &height) && just(f, &hjust, &vjust) && latext(f, &str) ?
    make_dash_box(c, lt, x, y, width, height, str, hjust, vjust, dash_len) : INULL;
}

#pragma argsused
LOCAL(void) write_dash_box(CANVAS c, OBJECT o, FILE *f)
{
  fprintf(f, "\\put%s{\\dashbox{%s}%s%s{%s}}\n",
	  P(o->ref), C(o->u.dash_box.dash_len), P(o->u.dash_box.sz), 
	  box_just_str(o->u.dash_box.hjust, o->u.dash_box.vjust),
	  stringify(o->u.dash_box.str));
}

#define dash_box_dist           box_dist
#define dash_box_inside_p       box_inside_p
#define dash_box_extent         box_extent
#define copy_dash_box           copy_box
#define dash_box_edit_info      box_edit_info
#define edit_dash_box           edit_box
#define destroy_dash_box        destroy_box

/* ----- Tables ----------------------------------------------------- */

/* Use obj.def definitions to build virtual function
   tables for drawing, writing, distance, inside_p,
   and destructor. */
LOCAL(void) (*draw_tbl[])(CANVAS, OBJECT) = {
# define DEF_OBJECT(Tag, Id, Fields, Cmd, TextFlag, LTtag) draw_ ## Id,
# include "obj.def"
# undef DEF_OBJECT
};

LOCAL(index) (*read_tbl[])(CANVAS, FILE*, LINE_THICKNESS, CC, CC) = {
# define DEF_OBJECT(Tag, Id, Fields, Cmd, TextFlag, LTtag) read_ ## Id,
# include "obj.def"
# undef DEF_OBJECT
};

LOCAL(void) (*write_tbl[])(CANVAS, OBJECT, FILE*) = {
# define DEF_OBJECT(Tag, Id, Fields, Cmd, TextFlag, LTtag) write_ ## Id,
# include "obj.def"
# undef DEF_OBJECT
};

LOCAL(DIST) (*dist_tbl[])(CANVAS, OBJECT) = {
# define DEF_OBJECT(Tag, Id, Fields, Cmd, TextFlag, LTtag) Id ## _dist,
# include "obj.def"
# undef DEF_OBJECT
};

LOCAL(int) (*inside_p_tbl[])(CANVAS, OBJECT) = {
# define DEF_OBJECT(Tag, Id, Fields, Cmd, TextFlag, LTtag) Id ## _inside_p,
# include "obj.def"
# undef DEF_OBJECT
};

LOCAL(void) (*extent_tbl[])(OBJECT, EXTENT) = {
# define DEF_OBJECT(Tag, Id, Fields, Cmd, TextFlag, LTtag) Id ## _extent,
# include "obj.def"
# undef DEF_OBJECT
};

LOCAL(void) (*copy_tbl[])(OBJECT, OBJECT) = {
# define DEF_OBJECT(Tag, Id, Fields, Cmd, TextFlag, LTtag) copy_ ## Id,
# include "obj.def"
# undef DEF_OBJECT
};

LOCAL(void) (*destroy_tbl[])(CANVAS, index, OBJECT) = {
# define DEF_OBJECT(Tag, Id, Fields, Cmd, TextFlag, LTtag) destroy_ ## Id,
# include "obj.def"
# undef DEF_OBJECT
};

LOCAL(void) (*edit_info_tbl[])(OBJECT, CP_REC, EDIT_INFO) = {
# define DEF_OBJECT(Tag, Id, Fields, Cmd, TextFlag, LTtag) Id ## _edit_info,
# include "obj.def"
# undef DEF_OBJECT
};

LOCAL(void) (*edit_tbl[])(OBJECT, EDIT_INFO) = {
# define DEF_OBJECT(Tag, Id, Fields, Cmd, TextFlag, LTtag) edit_ ## Id,
# include "obj.def"
# undef DEF_OBJECT
};


/* ----- Virtual and aggregate functions on objects ----------------- */

/* ----- Drawing -------------------- */

/* Set the color of everything. */
LOCAL(void) set_color(int color)
{
  setcolor(color);
  setfillstyle(SOLID_FILL, color);
}

/* Set color based on object status. */
LOCAL(void) set_obj_color(CANVAS c, OBJECT o)
{
  set_color(o->edit_picked_p ? c->color[ccRULERS] :
	    o->picked_p ? canvas_pick_color :
	    o->type == oORIGIN ? c->color[ccRULERS] :
	    c->color[o->line_thickness == ltTHICK ? ccTHICK : ccTHIN]);
}

/* Draw (or undraw) objects in an object vector. */
LOCAL(void) draw_objects(CANVAS c, index *obj_vec, int n, int on_p)
{
  int i;
  OBJECT op;
  DISPLAY_LIST dl = &c->dl;

  if (!w_status_p(&c->draw, wsVISIBLE))
    return;

  push_graphics_state(&c->cursor.window, 1);
  setlinestyle(SOLID_LINE, 0, 1);
  protect_cursor(&c->window);

  if (on_p) {
    for (i = 0; i < n; ++i) {
      op = deref(dl, obj_vec[i]);
      set_obj_color(c, op);
      (*draw_tbl[op->type])(c, op);
    }
  }
  else {
    set_color(c->color[ccBACKGROUND]);
    for (i = 0; i < n; ++i) {
      op = deref(dl, obj_vec[i]);
      (*draw_tbl[op->type])(c, op);
    }
  }
  unprotect_cursor();
  pop_graphics_state();
}

/* Draw all objects on a canvas. */
LOCAL(void) draw_all_objects(CANVAS c)
{
  push_graphics_state(&c->cursor.window, 1);
  setlinestyle(SOLID_LINE, 0, 1);
  protect_cursor(&c->window);

  do_display_list(obj, &c->dl, next, 
    set_obj_color(c, obj);
    (*draw_tbl[obj->type])(c, obj); );

  unprotect_cursor();
  pop_graphics_state();
}

/* Erase and destroy all objects on a canvas. */
void clear_canvas(CANVAS c)
{
  DISPLAY_LIST dl = &c->dl;
  while (dl->head != INULL) {
    index o = dl->head;
    OBJECT op = deref(dl, o);
    odelete2(o, op, dl);
    (*destroy_tbl[op->type])(c, o, op);
  }
  assert(dl->origin == INULL);
  assert(dl->n_picked == 0);
  assert(dl->n_obj == 0);
  clear_window(&c->draw, 1);
  note_dl_change(dl);
}

/* ----- Input ---------------------- */

/* Read objects from a file directly onto the 
   display list of the given canvas.  If
   settings is NULL, we merge objects into
   canvas, ignoring the picture statement. */
int read_objects(CANVAS c, char *file_name, CC oval_rad, SETTINGS settings)
{
  FILE *f;
  int ok_p;
  int merge_p = !settings;
  LINE_THICKNESS current_thickness = ltTHIN;
  OBJECT_TYPE type;
  CC x, y;
  index o;
  unsigned read_status;
  char *caption, *unitlength_str;
  UNITS_TYPE units;

  /* Set internal global oval radius. */
  read_oval_rad = oval_rad;

  /* No figure, center, or caption until read. */
  read_status = 0;
  unitlength_str = caption = NULL;
  units = uERROR;

  if ((f = fopen(file_name, "r")) == NULL)
    ok_p = 0;
  else {

    ok_p = 1;

    do {
      switch (look_ahead(f)) {

	case tBEGIN:

	  switch (environment(f)) {

	    case ePICTURE:
	      if (merge_p)
		ok_p = waste_picture(f);
	      else if ((o = read_picture(c, f)) != INULL)
		add_objects(c, &o, 1, NULL);
	      else
		ok_p = 0;
	      break;

	    case eFIGURE:
	      read_status |= bit(sFIGURE);
	      break;

	    case eCENTER:
	      read_status |= bit(sCENTER);
	      break;

	    case eDOCUMENT:
	      read_status |= bit(sDOCUMENT);
	      break;

	    case eTEDDISPLAY:
	      /* do nothing to status */
	      break;

	    default:
	      ok_p = 0;
	      break;
	  }
	  break;

	case tEND:

	  switch (environment(f)) {

	    case ePICTURE:
	    case eFIGURE:
	    case eCENTER:
	    case eDOCUMENT:
	    case eTEDDISPLAY:
	      break;

	    default:
	      ok_p = 0;
	  }
	  break;

	case tCAPTION:
	  if (merge_p)
	    latext(f, 0);
	  else if (latext(f, &caption))     
	    read_status |= bit(sCAPTION);
	  else
	    ok_p = 0;
	  break;

	case tCENTERING:
	  read_status |= bit(sCENTER);
	  break;

	case tUNIT_LENGTH:
	  if (merge_p)
	    ok_p = unitlength(f, 0, 0, 0);
	  else if (unitlength(f, &c->unitlength, &unitlength_str, (UNITS_TYPE*)&units))
	    read_status |= bit(sUNIT_LENGTH);
	  else
	    ok_p = 0;
	  break;

	case tTHICK_LINES:
	  current_thickness = ltTHICK;
	  break;

	case tTHIN_LINES:
	  current_thickness = ltTHIN;
	  break;

	case tMULTIPUT:
	  ok_p = waste_multiput(f);
	  break;

	case tDOCUMENTSTYLE:
	case tPAGESTYLE:
	case tLABEL:
	  ok_p = waste_braced_args(f, 1);
	  break;

	case tNEWCOMMAND:
	case tRENEWCOMMAND:
	  ok_p = waste_braced_args(f, 2);
	  break;

	case tNEWENVIRONMENT:
	  ok_p = waste_teddisplay_def(f);
	  break;

	case tFBOX:
	  ok_p = lcurl(f);
	  break;

	case tRCURLTAG:
	  ok_p = rcurl(f);
	  break;

	case tPUT:
	  if (prefix(f, &x, &y, (OBJECT_TYPE*)(&type)) 
	  && (o = (*read_tbl[type])(c, f, current_thickness, x, y)) != INULL
	  && rcurl(f)) {
	    if (merge_p)
	      dereference(c, o)->picked_p = 1;
	    oenqueue(o, &c->dl, NULL);
	  }
	  else
	    ok_p = 0;
	  break;

	case tEOF:
	  goto break_break;

	case tERROR:
	  ok_p = 0;
	  break;
      }
      if (ferror(f)) ok_p = 0;
    } while (ok_p);

  break_break: 
    fclose(f);
  }
  if (ok_p) {
    draw_all_objects(c); 
    if (!merge_p) {
      set_read_settings(settings, read_status);
      if (read_status & bit(sCAPTION))
	set_caption(settings, caption);
      if (read_status & bit(sUNIT_LENGTH)) {
	set_unitlength(settings, unitlength_str);
	set_units(settings, units);
      }
    }
    note_dl_change(&c->dl);
  }
  return ok_p;
}

/* Set the canvas unitlength to a new value 
   and redraw all objects if it's changed. */
void set_canvas_unitlength(CANVAS c, SP unitlength)
{
  if (c->unitlength != unitlength) {
    c->unitlength = unitlength;
    clear_window(&c->draw, 1);
  }
}

/* ----- Extents -------------------- */

/* Install an origin on the canvas if there is none.
   Update the picture box if there was no origin at
   all or if there was one and caller requested. 
   Return non-0 iff bounding box was adjusted. */
LOCAL(int) check_origin(CANVAS c)
{
  EXTENT_REC ext, tmp;
  OBJECT oldp;
  int valid_p = 0;
  index old = c->dl.origin, old_rtn, new;

  /* Don't do check if display list is empty or
     if there's already an origin and it wasn't
     determined automatically. */
  if (c->dl.head == INULL)
    return 0;

  if (old != INULL) {
    oldp = deref(&c->dl, old);
    if (!oldp->u.origin.auto_p)
      return 0;
  }

  do_display_list(o, &c->dl, next, 
    if (!valid_p) {
      (*extent_tbl[o->type])(o, &ext);
      valid_p = ext.valid_p;
    }
    else {
      (*extent_tbl[o->type])(o, &tmp);
      if (tmp.valid_p) {
	if (tmp.p0.x < ext.p0.x) ext.p0.x = tmp.p0.x;
	if (tmp.p0.y < ext.p0.y) ext.p0.y = tmp.p0.y;
	if (tmp.p1.x > ext.p1.x) ext.p1.x = tmp.p1.x;
	if (tmp.p1.y > ext.p1.y) ext.p1.y = tmp.p1.y;
      }
    }
  );

  /* See if we found anything 
     valid on the display list. */
  if (!valid_p)
    return 0;

  new = make_origin(c, ext.p0.x, ext.p0.y, 
		    ext.p1.x - ext.p0.x, ext.p1.y - ext.p0.y, 1);
  add_objects(c, &new, 1, &old_rtn);
  assert(old == old_rtn);
  return 1;
}

/* ----- Output --------------------- */

LOCAL(void) write_header(FILE *f, unsigned status, char *unitlength)
{
  extern char version[];
  fprintf(f, "%% Ted V%s output %s\n", version, dtg());
  if (status & bit(sDOCUMENT))
    fprintf(f, 
      "\\documentstyle{article}\n"
      "\\begin{document}\n"
      "\\pagestyle{empty}\n"
      "\\renewcommand{\\fboxsep}{0in}\n"
      "\\renewcommand{\\fboxrule}{.1pt}\n"
      "\\newcommand{\\rcurltag}{}\n"
      "\\newenvironment{teddisplay}{\\begin{figure}}{\\end{figure}}\n");
  if (status & bit(sUNIT_LENGTH))
    fprintf(f, "\\unitlength=%s\n", unitlength);
  if (status & bit(sFIGURE))
    fprintf(f, "\\begin{figure}\n");
  else if (status & bit(sDOCUMENT))
    fprintf(f, "\\begin{teddisplay}\n");
  if (status & bit(sCENTER))
    fprintf(f, status & (bit(sFIGURE)|bit(sDOCUMENT)) ? "\\centering\n" : "\\begin{center}\n");
  if (status & bit(sDOCUMENT))
    fprintf(f, "\\fbox{\n");
}

LOCAL(char*) labelify(char *file_name, char *buf)
{
  static char prefix[] = "fig:";

  strcpy(buf, prefix);
  fnsplit(file_name, NULL, NULL, &buf[sizeof prefix - 1], NULL);
  strlwr(&buf[sizeof prefix - 1]);
  return buf;
}

LOCAL(void) write_trailer(FILE *f, unsigned status, char *caption, char *label)
{
  fprintf(f, "\\end{picture}\n");
  if (status & bit(sDOCUMENT))
    fprintf(f, "\\rcurltag}\n");
  if ((status & (bit(sCENTER)|bit(sFIGURE)|bit(sDOCUMENT))) == bit(sCENTER))
    fprintf(f, "\\end{center}\n");
  if (status & bit(sCAPTION))
    fprintf(f, "\\caption{%s}\n", caption);
  if (status & (bit(sCAPTION)|bit(sFIGURE)))
    fprintf(f, "\\label{%s}\n", label);
  if (status & bit(sFIGURE))
    fprintf(f, "\\end{figure}\n");
  else if (status & bit(sDOCUMENT))
    fprintf(f, "\\end{teddisplay}\n");
  if (status & bit(sDOCUMENT))
    fprintf(f,"\\end{document}\n");
  fprintf(f, "%% End of Ted output.\n");
}

/* Write all objects on a canvas to a file.
   Return non-0 on error. */
int write_objects(CANVAS c, char *file_name, SETTINGS settings, int save_mem_p)
{
  FILE *f;
  int err_p = 0;
  LINE_THICKNESS current_thickness = ltTHIN;
  static char *thickness_str[] = { NULL, "thin", "thick" };
  char buf[16];

  if ((f = fopen(file_name, "w")) == NULL)
    err_p = 1;
  else {
    write_header(f, settings->status, settings->unitlength);
    if (!save_mem_p)
      check_origin(c);
    do_display_list_from(o, &c->dl, origin, next, 
      if (o->line_thickness != ltNONE  && 
	  o->line_thickness != current_thickness) {
	current_thickness = o->line_thickness;
	fprintf(f, "\\%slines\n", thickness_str[current_thickness]);
      }
      (*write_tbl[o->type])(c, o, f); 
    );
    write_trailer(f, settings->status, settings->caption, labelify(file_name, buf));
    err_p = ferror(f);
    clearerr(f);
    fclose(f);
  }
  return err_p;
}

/* ----- Picks ---------------------- */

/* Set the pick flag of an object and 
   (re)draw it with the correct color. */
void pick_objects(CANVAS c, index *o, int n, int on_p)
{
  set_object_pick_flags(&c->dl, o, n, on_p);
  note_dl_change(&c->dl);
  draw_objects(c, o, n, 1);
}

/* Set the edit pick flag of an object and
   (re)draw it with the correct color. */
void edit_pick_object(CANVAS c, index o, int on_p)
{
  OBJECT op = deref(&c->dl, o);
  op->edit_picked_p = on_p;
  draw_objects(c, &o, 1, 1);
}

/* Return an array of object pointers 
   to the currently picked objects. */
int get_picks(CANVAS c, index *obj, PICK_PREDICATE elligible_p)
{
  int i = 0;
  do_display_list(op, &c->dl, next, 
    if (op->picked_p && (*elligible_p)(op))
      obj[i++] = _o;
  );
  return i;
}

/* Build buffer full of point picks: Store into array `buf' up 
   to `max_picks' pick records ordered from smallest to larger 
   pick distance from point pick0, of canvas c, but no farther 
   away than pick_rad.  The picked objects come from the display 
   list of canvas c. If pick_flag is 0 or 1, it is interpreted as
   a pick_p.  Only objects of the opposite disposition are 
   considered for return.  If the flag is 2, then an object is
   elligible for picking regardless of its pick status. */
int do_point_pick(CANVAS c, PICK_PREDICATE elligible_p, PICK buf, int max_picks, int x, int y, int rad)
{
  DISPLAY_LIST dl = &c->dl;
  PICK p, end, buf_end;
  long d;

  c->pick0 = sp2cp(c, x, y);
  c->pick_rad = sc2cc(c, rad);

  end = buf;
  buf_end = buf + max_picks;

  do_display_list(o, dl, next,

    /* Skip inelligible objects. */
    if(!(*elligible_p)(o))
      continue;

    /* Compute pick distance. */
    d = (*dist_tbl[o->type])(c, o);

    /* Skip objects not in pick radius. */
    if (d > c->pick_rad)
      continue;

    /* If sorted buf is full,
	 if last (farthest pick) is closer than this one
	   skip this object
	 else
	   overwrite last (farthest) pick with this one
       else
	 tack this object on end of (not full) buffer */
    if (end == buf_end)
      if ((end-1)->dist < d)
	continue;
      else
	p = end-1;
    else
      p = end++;

    /* Actually insert the object. */
    p->dist = d;
    p->o = _o;

    /* Bubble forward to correct location. */
    while (p > buf && p->dist < (p-1)->dist) {
      PICK_REC t = *p;
      *p = *(p-1);
      *--p = t;
    }
  );
  /* Return size of pick. */
  return end - buf;
}

/* Pick or unpick everything on the display list that's fully
   inside the given screen rectangle. */
void do_rect_pick(CANVAS c, int pick_p, int x0, int y0, int x1, int y1)
{
  c->pick0 = sp2cp(c, x0, y0);
  c->pick1 = sp2cp(c, x1, y1);
  diagonalize(&c->pick0, &c->pick1);

  /* Slow, modular version. 
  do_display_list(o, &c->dl, next,
    if (o->picked_p != pick_p && (*inside_p_tbl[o->type])(c, o))
      pick_object(c, _o, pick_p);
  );
  */

  /* Fast version with embedded pick and draw code. */
  assert(w_status_p(&c->window, wsVISIBLE));

  push_graphics_state(&c->cursor.window, 1);
  setlinestyle(SOLID_LINE, 0, 1);
  protect_cursor(&c->window);
  do_display_list(op, &c->dl, next, 
    if (pick_p != op->picked_p && (*inside_p_tbl[op->type])(c, op)) {
      _dl->n_picked += pick_p - op->picked_p;
      op->picked_p = pick_p;
      set_obj_color(c, op);
      (*draw_tbl[op->type])(c, op);
    }
  );
  unprotect_cursor();
  pop_graphics_state();
  note_dl_change(&c->dl);
}

/* Pick or unpick everything on the display list. */
void pick_all(CANVAS c, int pick_p)
{
  /* Slow, modular version. 
  do_display_list(o, &c->dl, next, pick_object(c, _o, pick_p); );
  */

  assert(w_status_p(&c->window, wsVISIBLE));

  push_graphics_state(&c->cursor.window, 1);
  setlinestyle(SOLID_LINE, 0, 1);
  protect_cursor(&c->window);
  do_display_list(op, &c->dl, next, 
    if (pick_p != op->picked_p) {
      op->picked_p = pick_p;
      set_obj_color(c, op);
      (*draw_tbl[op->type])(c, op);
    }
  );
  unprotect_cursor();
  pop_graphics_state();
  c->dl.n_picked = pick_p ? c->dl.n_obj : 0;
  note_dl_change(&c->dl);
}

/* ----- Edit Operations ------------ */

/* Add objects to a canvas.  Can't include
   an origin if there already is one. */
void add_objects(CANVAS c, index *o, int n, index *old_origin_rtn)
{
  int i;
  index origin, old_origin;

  old_origin = INULL;
  for (i = 0; i < n; ++i) {
    oenqueue(o[i], &c->dl, &origin);
    if (origin != INULL) {
      assert(old_origin == INULL);
      old_origin = origin;
      draw_objects(c, &origin, 1, 0);
    }
  }
  note_dl_change(&c->dl);
  /* If we erased origin and auto-redraw is on, do the redraw,
     else just draw the added objects. */
  if (old_origin != INULL && v_status_p(c, vAUTO_REDRAW)) 
    draw_all_objects(c);
  else
    draw_objects(c, o, n, 1);
  if (old_origin_rtn != NULL)
    *old_origin_rtn = old_origin;
}

void delete_objects(CANVAS c, index *o, int n)
{
  int i;

  draw_objects(c, o, n, 0);
  for (i = 0; i < n; ++i) 
    odelete1(o[i], &c->dl);
  note_dl_change(&c->dl);
  if (v_status_p(c, vAUTO_REDRAW))
    draw_all_objects(c);
}

/* Make copies of each object in a vector 
   of objects.  Offset the copy dx,dy from 
   the original. */
void copy_objects(CANVAS c, index *from, index *to, int n, CC dx, CC dy)
{
  int i;
  OBJECT op, new_op;
  DISPLAY_LIST dl = &c->dl;

  for (i = 0; i < n; ++i) {

    /* Get pointer to old object and copy. */
    op = deref(dl, from[i]);

    /* Make a partial copy. */
    to[i] = oalloc(dl, op->type, op->line_thickness, 
		   op->ref.x + dx, op->ref.y + dy, 
		   &new_op);

    /* Pick the copy. */
    new_op->picked_p = 1;

    /* Copy the type-dependent fields of the object. */
    (*copy_tbl[op->type])(op, new_op);
  }
}

index make_text_edited_box(CANVAS c, index o, char *str, int hjust, int vjust, int picked_p)
{
  OBJECT new_op, op = deref(&c->dl, o);
  index new;

  new = oalloc(&c->dl, op->type, op->line_thickness, op->ref.x, op->ref.y, &new_op);
  new_op->u = op->u;
  new_op->u.text.str = str;
  new_op->u.text.hjust = hjust;
  new_op->u.text.vjust = vjust;
  new_op->picked_p = picked_p;
  return new;
}

/* Move each object in a vector of objects. */
void move_objects(CANVAS c, index *o, int n, CP_REC d)
{
  int i;
  OBJECT op;

  draw_objects(c, o, n, 0);
  for (i = 0; i < n; ++i) {
    op = deref(&c->dl, o[i]);
    op->ref.x += d.x;
    op->ref.y += d.y;
  }
  if (v_status_p(c, vAUTO_REDRAW))
    draw_all_objects(c);
  else
    draw_objects(c, o, n, 1);
}

/* Free each object in a vector of objects. */
void destroy_objects(CANVAS c, index *o_vec, int n)
{
  int i;
  for (i = 0; i < n; ++i) {
    index o = o_vec[i];
    OBJECT op = deref(&c->dl, o);
    (*destroy_tbl[op->type])(c, o, op);
  }
}

void set_object_line_thickness(CANVAS c, index *o_vec, int n, LINE_THICKNESS lt)
{
  int i;
  for (i = 0; i < n; ++i) 
    deref(&c->dl, o_vec[i])->line_thickness = lt;
  draw_objects(c, o_vec, n, 1);
}

unsigned n_picked(CANVAS c, int include_origin_p)
{
  return (include_origin_p || c->dl.origin == INULL) ? 
	 c->dl.n_picked :
	 c->dl.n_picked - deref(&c->dl, c->dl.origin)->picked_p;
}

#define box_outside_box_p(a0, a1, b0, b1)       \
	  (a0.x >= b1.x || a1.x <= b0.x         \
	|| a0.y >= b1.y || a1.y <= b0.y)

/* Mark outside bits of objects in a vector 1 iff
   the object extent lies completely the given box. */
int mark_outside(CANVAS c, index *o_vec, int n, CP_REC org, CP_REC size)
{
  int i, n_outside;
  EXTENT_REC ext;
  CP_REC diag;
  OBJECT op;

  diag.x = org.x + size.x;
  diag.y = org.y + size.y;
  n_outside = 0;

  for (i = 0; i < n; ++i) {

    op = deref(&c->dl, o_vec[i]);

    (*extent_tbl[op->type])(op, &ext);

    if (ext.valid_p && box_outside_box_p(ext.p0, ext.p1, org, diag)) {
      ++n_outside;
      op->outside_p = 1;
    }
    else 
      op->outside_p = 0;
  }

  return n_outside;
}

/* Unpick all objects in a vector are marked outside.
   Compress the vector to consist only if inside objects. */
int unpick_outside(CANVAS c, index *o_vec, int n)
{
  int i, n_inside;
  OBJECT op;

  n_inside = 0;
  for (i = 0; i < n; ++i) {

    op = deref(&c->dl, o_vec[i]);

    if (op->outside_p)
      set_object_pick_flags(&c->dl, o_vec + i, 1, 0);
    else 
      o_vec[n_inside++] = o_vec[i];
  }
  note_dl_change(&c->dl);
  return n_inside;
}

/* Return information about an object to start a modification edit. */
void edit_info(CANVAS c, index o, CP_REC pick, EDIT_INFO info)
{
  OBJECT op = deref(&c->dl, o);
  (*edit_info_tbl[op->type])(op, pick, info);
}

index make_edited_object(CANVAS c, index o, EDIT_INFO info)
{
  OBJECT op, new_op;

  op = deref(&c->dl, o);
  o = oalloc(&c->dl, op->type, op->line_thickness, op->ref.x, op->ref.y, &new_op);
  new_op->picked_p = op->picked_p;
  (*copy_tbl[op->type])(op, new_op);
  (*edit_tbl[op->type])(new_op, info);
  return o;
}

/* ----- Ruler ------------------------------------------------------ */

LOCAL(int) round_up(int x, int inc)
{
  int t = x + (inc - 1);
  return t - t % inc;
}

enum { canvas_ruler_tick_sep = 4 };

LOCAL(void) draw_ruler(CANVAS c, int color)
{
  CC x, y;
  int sx, sy, no_num_space, inc, label_div;
  char buf[16];

  if (!w_status_p(&c->draw, wsVISIBLE))
    return;

  /* Find a tick increment dependent on current zoom. */
  for (inc = canvas_ruler_inc, label_div = canvas_ruler_inc * 10; 
       cc2sc(c, inc/2) >= canvas_ruler_tick_sep; 
       inc /= 10, label_div /= 10) {

    /* message(bit(mOK), root_window, "Inc: %d (%d %d %d)", inc, cc2sc(c,inc), cc2sc(c, inc/2), cc2sc(c, inc/5)); */

    if (cc2sc(c, inc/5) < canvas_ruler_tick_sep) {
      inc /= 2;
      label_div /= 10;
      break;
    }

    if (cc2sc(c, inc/10) < canvas_ruler_tick_sep) {
      inc /= 5;
      label_div /= 10;
      break;
    }

  }

  push_graphics_state(&c->draw, 1);
  protect_cursor(&c->draw);

  setcolor(color);
  settextstyle(SMALL_FONT, HORIZ_DIR, 4);
  settextjustify(CENTER_TEXT, BOTTOM_TEXT);
  sy = c->draw.height - 1;
  no_num_space = 2 * textheight("1");
  for (x = round_up(c->vp_org.x, inc); 
       (sx = cc2scx(c, x)) < c->draw.width; 
       x += inc) {
    moveto(sx, sy);
    if (x % (10 * inc) == 0) {
      linerel(0, -3 * canvas_ruler_tick_height);
      if (sx > no_num_space) {
	moverel(0, -2);
	outtext(itoa(x / label_div, buf, 10));
      }
    }
    else linerel(0, x % (5 * inc) == 0 ? 
		      -2 * canvas_ruler_tick_height :
		      -canvas_ruler_tick_height);
  }

  settextjustify(LEFT_TEXT, CENTER_TEXT);
  no_num_space = sy - no_num_space;
  for (y = round_up(c->vp_org.y, inc);
       (sy = cc2scy(c, y)) >= 0;
       y += inc) {
    moveto(0, sy);
    if (y % (10*inc) == 0) {
      linerel(3 * canvas_ruler_tick_height, 0);
      if (sy < no_num_space) {
	moverel(2, 0);
	outtext(itoa(y / label_div, buf, 10));
      }
    }
    else linerel(x % (5 * inc) == 0 ? 
		   2 * canvas_ruler_tick_height :
		   canvas_ruler_tick_height, 0);
  }

  unprotect_cursor();
  pop_graphics_state();
  if (v_status_p(c, vAUTO_REDRAW) || color == c->color[ccBACKGROUND])
    draw_all_objects(c);
}

/* ----- Canvas ----------------------------------------------------- */

/* A canvas is a window with four subwindows: an x and a y 
   scrollbar, a transparent cursor window and a frame for drawing. 
   The canvas also has a display list with up to 65536 objects
   in it.  These are drawn when the canvas is mapped.  Moreover,
   the cursor window is considered a viewport in the canvas,
   whose coordinates are liable to be much bigger than the cursor
   window.  Scrollbar action move the viewport over the canvas
   The viewport size in canvas coords can be changed by the caller
   to obtain zoom.  Objects can be inserted into and removed
   from the display list.  The display list can be sorted 
   and marked according to given predicates.  This allows
   a simple but general kind of pick strategy. */

/* Locate canvas subwindows based on the
   bits of canvas status mask. */
void locate_canvas_scrollbars(CANVAS c)
{                   
  int x0, x1, y0, y1, visible_p;

  visible_p = w_status_p(&c->window, wsVISIBLE);

  if (visible_p)
    unmap_window(&c->window);

  if (v_status_p(c, vRIGHT_SCROLLBAR)) {
    x0 = 0;
    x1 = root_window->width;
  }
  else {
    x0 = root_window->width; 
    x1 = 0;
  }

  if (v_status_p(c, vBOTTOM_SCROLLBAR)) {
    y0 = 0; 
    y1 = root_window->height;
  }
  else {
    y0 = root_window->height; 
    y1 = 0;
  }

  locate_window(&c->x_scrollbar.window, x0, y1, 0);
  locate_window(&c->y_scrollbar.window, x1, y0, 0);
  locate_window(&c->draw, x0, y0, 0);

  if (visible_p)
    map_window(&c->window);
}

/* Locate the canvas scrollbar sliders to get them in
   synch with the canvas origin. */
LOCAL(void) reset_canvas_scrollbars(CANVAS c)
{
  int vp_maxy = vp_y_size(c) - 1;

  c->x_scrollbar.n_pos = c->size - (c->vp_x_size - 1);
  c->y_scrollbar.n_pos = c->size - vp_maxy;
  c->x_scrollbar.n_step_pos = c->vp_x_size;
  c->y_scrollbar.n_step_pos = vp_maxy + 1;

  c->vp_org.x = min(c->x_scrollbar.n_pos - 1, max(0, c->vp_org.x));
  c->vp_org.y = min(c->y_scrollbar.n_pos - 1, max(0, c->vp_org.y));
  position_scrollbar(&c->x_scrollbar, c->vp_org.x);
  position_scrollbar(&c->y_scrollbar, c->vp_org.y);
}

/* Set origin and size of viewport of canvas. */
int set_canvas_viewport(CANVAS c,
			CC vp_x_size, CC vp_org_x, CC vp_org_y,
			int reset_scrollbars_p)
{
  CC d;

  if (vp_x_size == c->vp_x_size && 
      vp_org_x == c->vp_org.x && vp_org_y == c->vp_org.y)
    return 0;

  if (vp_x_size < c->draw.width) {
    c->vp_x_size = c->draw.width;
    vp_org_x += (vp_x_size - c->draw.width)/2;
    vp_org_y += (vp_x_size * c->draw.height/c->draw.width - c->draw.height)/2;
  }
  else 
    c->vp_x_size = vp_x_size;
  d = c->size - c->vp_x_size;
  if (vp_org_x > d)
    vp_org_x = d;
  if (vp_org_x < 0)
    vp_org_x = 0;
  d = c->size - (c->vp_x_size * c->draw.height + c->draw.width/2)/c->draw.width;
  if (vp_org_y > d)
    vp_org_y = d;
  if (vp_org_y < 0)
    vp_org_y = 0;

  c->vp_org.x = vp_org_x;
  c->vp_org.y = vp_org_y;

  if (reset_scrollbars_p)
    reset_canvas_scrollbars(c);

  if (w_status_p(&c->window, wsVISIBLE)) 
    clear_window(&c->draw, 1);

  return 1;
}

/* Canvas event handlers. */
static void handle_vertical_scroll(int pos, ENV env)
#define c ((CANVAS)env)
{
  c->vp_org.y = pos;
  clear_window(&c->draw, 1);
}
#undef c

static void handle_horizontal_scroll(int pos, ENV env)
#define c ((CANVAS)env)
{
  c->vp_org.x = pos;
  clear_window(&c->draw, 1);
}
#undef c

/* ----- Flashing colors on canvas ---------------------------------- */

/* Color setup is done with respect to the canvas colors 
   of the argument to `enable'.  There's not much choice
   because there's only one hardware palette.  This will
   be problematic only if there's more than one canvas
   with different colors visible at once. */
static int pick_palette_entry[2], pick_on_p;

#pragma argsused
static void toggle_flashing_pallete_entries(EVENT e)
{
  setpalette(cPICK, pick_palette_entry[pick_on_p = 1 - pick_on_p]);
}

LOCAL(void) set_flashing_palette_entries(CANVAS c)
{
  struct palettetype palette[1];

  getpalette(palette);
  pick_palette_entry[1] = palette->colors[c->color[ccPICK]];
  pick_palette_entry[0] = palette->colors[c->color[ccBACKGROUND]];
}

void enable_flashing_pick(CANVAS c)
{
  set_flashing_palette_entries(c);
  global_event_handler[eTICK(tUSER0)] = toggle_flashing_pallete_entries;
  start_timer(tUSER0, delta(.2), 1);
}

void disable_flashing_pick(void)
{
  stop_timer(tUSER0);
  global_event_handler[eTICK(tUSER0)] = null_handler_code;
  setpalette(cPICK, pick_palette_entry[1]);
}

void set_canvas_colors(CANVAS c, CANVAS_COLOR_VECTOR colors)
{
  if (memcmp(c->color, colors, sizeof(CANVAS_COLOR_VECTOR)) != 0) {
    memcpy(c->color, colors, sizeof(CANVAS_COLOR_VECTOR));
    if (!set_window_bg_color(&c->draw, colors[ccBACKGROUND], 1))
      clear_window(&c->draw, 1);
    set_flashing_palette_entries(c);
	setpalette(cPICK, pick_palette_entry[1]);
    set_cursor_color(&c->cursor, colors[ccHAIRS]);
  }
}

/* ----- Miscellaneous handlers ------------------------------------- */

LOCAL(void) handle_map(EVENT e)
{
  CANVAS c = (CANVAS)e->map.window->parent;
  draw_all_objects(c);
  if (v_status_p(c, vRULER))
    draw_ruler(c, c->color[ccRULERS]);
}

BeginDefDispatch(canvas_draw)
  Dispatch(eMAP, handle_map)
EndDefDispatch(canvas_draw)

LOCAL(void) handle_key(EVENT e)
{
  CANVAS c = (CANVAS)e->keystroke.window;
  int key = e->keystroke.key;
  unsigned mask;

  if (key == '\t') {
    unset_focus(&c->window);
    return;
  }
  mask = e->keystroke.mask;
  if (!position_scrollbar_by_key(&c->x_scrollbar, key, mask) &&
      !position_scrollbar_by_key(&c->y_scrollbar, key, mask))
    refuse_keystroke(e);
}

BeginDefDispatch(canvas)
  Dispatch(eKEYSTROKE, handle_key)
EndDefDispatch(canvas)

/* Activate a canvas. */
void open_canvas(CANVAS c, int x, int y, WINDOW parent)
{
  open_window(&c->window, parent, x, y, c->window.width, c->window.height,
	      canvas_border_width, canvas_border_color, cDARKGRAY, bit(eKEYSTROKE));
  SetDispatch(&c->window, canvas);

  open_window(&c->draw, &c->window, 0, 0,
	      c->window.width - scrollbar_thickness(1), 
	      c->window.height - scrollbar_thickness(1),
	      0, NO_COLOR, c->color[ccBACKGROUND], bit(eMAP));
  SetDispatch(&c->draw, canvas_draw);
  map_window(&c->draw);

  open_cursor(&c->cursor, 
	      0, 0, c->draw.width, c->draw.height, 
	      c->color[ccHAIRS], &c->draw);
  map_window(&c->cursor.window);

  SetScrollbar(&c->x_scrollbar, NoTitle, NoHotChar, 
	       FullLength, 20000, 400, 1000, 0, 
	       Enabled|OneOfTwo, handle_horizontal_scroll, c);
  SetScrollbar(&c->y_scrollbar, NoTitle, NoHotChar,
	       FullLength, 20000, 400, 1000, 0, 
	       Enabled|Vertical|OneOfTwo|Reverse, handle_vertical_scroll, c);

  open_scrollbar(&c->x_scrollbar, DEFAULT, DEFAULT, &c->window);
  map_window(&c->x_scrollbar.window);
  open_scrollbar(&c->y_scrollbar, DEFAULT, DEFAULT, &c->window);
  map_window(&c->y_scrollbar.window);

  locate_canvas_scrollbars(c);
  reset_canvas_scrollbars(c);

  set_cursor(&c->cursor, bit(cMOUSE));
  show_cursor(&c->cursor, 1);

  memset(c->dl.blk, 0, sizeof c->dl.blk);
  c->dl.n_obj = c->dl.n_picked = c->dl.n_blks = 0;
  c->dl.head = c->dl.free = c->dl.origin = INULL;
  set_cont_pos_invalid(&c->cont_pos);
  set_copy_displacement_invalid(&c->copy_displacement);
  note_dl_change(&c->dl);
}

void enable_canvas_ruler(CANVAS c)
{
  if (!v_status_p(c, vRULER)) {
    c->status |= bit(vRULER);
    draw_ruler(c, c->color[ccRULERS]);
  }
}

void disable_canvas_ruler(CANVAS c)
{
  if (v_status_p(c, vRULER)) {
    c->status &= notbit(vRULER);
    draw_ruler(c, c->color[ccBACKGROUND]);
  }
}
